% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/findScale.R
\name{findScale}
\alias{findScale}
\title{Find an appropriate smoothing parameter}
\usage{
findScale(
  tracks,
  scaleARS = TRUE,
  res = NULL,
  sumTrips = NULL,
  scalesFPT = NULL,
  peakWidth = 1,
  peakMethod = "first"
)
}
\arguments{
\item{tracks}{SpatialPointsDataFrame. Must be projected into an equal-area
coordinate system; if not, first run \code{\link{projectTracks}}.}

\item{scaleARS}{logical scalar (TRUE/FALSE). Do you want to calculate the
scale of area-restricted search using First Passage Time analysis? NOTE: does
 not allow for duplicate date-time stamps.}

\item{res}{numeric. The desired grid cell resolution (square kilometers) for
subsequent kernel analysis (NOT performed in this function). If this is not
specified, the scale of movement is compared to a 500-cell grid, with spatial
 extent determined by the latitudinal and longitudinal extent of the data.}

\item{sumTrips}{data.frame. Output of \code{\link{tripSummary}} function. If
not specified, \code{\link{tripSummary}} will be called within the function.}

\item{scalesFPT}{numeric vector. Set of spatial scales at which to calculate
First Passage Time. If not specified, the distribution of between-point
distances will be used to derive a set.}

\item{peakWidth}{numeric. How many scale-steps either side of focal scale
used to identify a peak. Default is 1, whereby a peak is defined as any scale
 at which the variance in log FPT increases from the previous scale, and
 decreases for the following one.}

\item{peakMethod}{character. Which method should be used to select the focal
peak for each ID. Options are "first", "max", and "steep". "steep" is a value
 of scalesFPT at which the variance in log FPT changes most rapidly compared
 to the surrounding scale(s).}
}
\value{
This function returns a one-row dataframe with the foraging range in
the first column (i.e. 'med_max_distance') calculated by
\code{\link{tripSummary}}, and the median step length
(i.e. between point distance) for the data set. The subsequent columns
contain various candidate smoothing parameter ('h') values calculated in the
following ways:
\enumerate{
  \item 'mag' - log of the foraging range (i.e. median maximum trip distance)
  \item 'href' - reference bandwidth a simple, data-driven method which takes
   into account the number of points, and the variance in X and Y directions.

   \eqn{sqrt((X + Y)* (n^(-1/6)))}; where X=Longitude/Easting,
   Y=Latitude/Northing, and n=number of relocations
  \item 'scaleARS' - spatial scale of area-restricted Search behavior as
  estimated using First Passage Time analysis
  (see \code{\link[adehabitatLT]{fpt}})
}

If the scaleARS option is used, a diagnostic plot is shown which illustrates
the change in variance of log-FPT  values calculated at each FPT scale. Grey
 vertical lines indicate the peaks identified for each individual using
 peakMethod method chosen, and the red line is the median of these, and the
 resulting scaleARS in the output table.

All values are in kilometers.
}
\description{
\code{findScale} takes a tracking data set and outputs a series of candidate
smoothing parameter values. Additionally, it compares the scale of movement
resolved by the sampling resolution of the data set, to a grid of desired
resolution.
}
\details{
The purpose of this function is to provide guidance regarding the two most
sensitive steps in the track2KBA analysis: specification of the (1) smoothing
 parameter and the (2) grid cell size for kernel density estimation (KDE).
 Specifically, the goal is to allow for exploration of the effect of these
 parameters and their inter-relatedness, so that an informed decision may be
 made regarding their specification in subsequent track2KBA steps.

Kernel density estimation has been identified as particularly sensitive to
the specification of the smoothing parameter (AKA bandwidth, or 'H' value),
that is, the parameter that defines the width of the normal distribution
around each location. Many techniques for identifying 'optimal' smoothing
parameters have been proposed (see Gitzen, Millspaugh, and Kernohan for a
classic review; see Fleming and Calabreses 2017 for a later implementation)
and many of these techniques have their merits; however, in the track2KBA
implementation of KDE we have opted for simplicity.

In the context of the track2KBA analysis, the smoothing parameter ought to
represent the relevant scale at which the animal interacts with the
environment. Therefore, when selecting a \emph{Scale} value for subsequent
analysis, the user must take into account the movement ecology of the study
species. For species which use Area-Restricted Search (ARS) behavior when
foraging, First Passage Time analysis may be used to identify the scale of
interaction with the environment (Fauchald and Tveraa 2003), however not all
species use ARS when foraging and therefore different techniques must be
used.

What minimum spatial scales are detectable by the data also depends on the
sampling resolution. Therefore, when applying First Passage Time analysis,
\code{findScale} sets the range of scales at which movements are analyzed
based on the distribution of forward, between-point displacements in the
data.

The grid cell size also affects the output of kernel density-based space use
analyses. Therefore, by specifying the \emph{res} parameter you can check
whether your desired grid cell size is reasonable, given the scale of
movement resolved by your data.
}
\examples{
## make some play data
dataGroup <- data.frame(Longitude = c(1, 1.01, 1.02, 1.04, 1.05, 1),
  Latitude =  c(1, 1.01, 1.02, 1.03, 1.021, 1), 
  ID = rep("A", 6),
  DateTime = format(
   lubridate::ymd_hms("2021-01-01 00:00:00") + 
   lubridate::hours(0:5)
   )
 )
 colony <- data.frame(
  Longitude = dataGroup$Longitude[1], Latitude = dataGroup$Latitude[1]
 )
 ## split data into trips
 trips <- tripSplit(dataGroup, colony=colony,
  innerBuff = 1, returnBuff = 1, duration = 0.5, 
  rmNonTrip = TRUE
 )  
 ## summarize trip characteristics
 sumTrips <- tripSummary(trips, colony)
 ## project tracks
 tracks_prj <- projectTracks(
   trips,
   projType = "azim",
   custom = "TRUE"
 )
 ## calculate candidate smoothing parameter values
 h_vals <- findScale(tracks_prj, sumTrips = sumTrips, scaleARS = FALSE)

}
