################################################################################
# --------------------- LOG-SPECTRAL DENSITY ESTIMATION ---------------------- #
################################################################################

#' Log-spectral density estimator
#' 
#' Estimates the log-spectral density.
#'
#' @param X       Time series of dimension \code{n}.
#' @param Te      Integer. Number of bins.
#' @param m       Integer. Number of observations per bin.
#' @param shift   Numeric. Shifting constant applied after data transformation, 
#'                computed as \code{(digamma(m/2) - log(m/2)) / sqrt(2)}.
#' @param q       Integer. Penalization order.
#' @param D       Matrix. DCT-I matrix of dimension \code{n}.
#' @param method  String. Method for selecting the smoothing parameter.
#' @param f_true  Vector or \code{NULL}. True regression function evaluated at 
#'                equi-spaced points, can be \code{NULL} if \code{method} is set
#'                to "\code{ML-oracle}" or "\code{GCV-oracle}".
#' @param evals   eigenvalues of the \code{q}-th order roughness penalty opera-
#'                tor in the Demmler-Reinsch basis.
#' 
#' @return g Vector. Estimated log-spectral density at equi-spaced points in 
#'            \[0,1\].
#' 
#' @keywords internal
#' @noRd
logsdf_estimator <- function(X, Te, m, shift, q, D, method, f_true, evals) {
  Y <- data_transformation(x = X, D = D, Te = Te, m = m) - shift
  lambda <- smoothpar_selection(
    y = Y, q = q, evals = evals, 
    method = method, f.true = f_true, sigma = 1 / m)
  g <- SSper_estimator(y = Y, lambda = lambda, evals = evals)
  return(g)
}

#' Periodic Smoothing Spline
#'
#' Computes the smoothing spline estimator of order q using the periodic 
#' Demmler-Reinsch basis.
#' 
#' @param y       vector with data points on equi-spaced design points.
#' @param lambda  smoothing parameter.
#' @param evals   eigenvalues of the \code{q}-th order roughness penalty opera-
#'                tor in the Demmler-Reinsch basis.
#' 
#' @return f periodic smoothing spline estimator evaluated in equi-spaced points
#'           in \[0,1\].
#' 
#' @keywords internal
#' @noRd
SSper_estimator <- function(y, lambda, evals) {
  if (lambda <= 0) {
    stop("lambda must be positive")
  }
  N <- length(y)
  f <- stats::fft(y[N:1])[N:1]/(1+lambda*evals)
  f <- Re(stats::fft(f[N:1])[N:1])/N
  return(f)
}