% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pCure.R
\name{pCure}
\alias{pCure}
\title{Cure Rate Model with pseudo-observation approach}
\usage{
pCure(
  formula1,
  formula2,
  time,
  status,
  data,
  subset,
  t0,
  model = c("mixture", "promotion"),
  nfolds = 5,
  lambda1 = NULL,
  exclude1 = NULL,
  penalty1 = c("lasso", "scad"),
  lambda2 = NULL,
  exclude2 = NULL,
  penalty2 = c("lasso", "scad"),
  control = list()
)
}
\arguments{
\item{formula1}{A formula object starting with \code{~} for the model formula.
This specifies the covariates in the incidence component and the long-term component under
the mixture cure model and the bounded cumulative model, respectively.}

\item{formula2}{A formula object starting with \code{~} for the model formula.
This specifies the covariates in the latency component and the short-term component under
the mixture cure model and the bounded cumulative model, respectively.}

\item{time}{A numeric vector for the observed survival times.}

\item{status}{A numeric vector for the event indicator;
0 indicates right-censoring and 1 indicates events.}

\item{data}{An optional data frame that contains the covariates and response variables
(\code{time} and \code{event}).}

\item{subset}{An optional logical vector specifying a subset of
observations to be used in the fitting process.}

\item{t0}{A vector of times, where the pseudo-observations are constructed.
When not specified, the default values are the 10, 20, ..., 90th percentiles of
uncensored event times.}

\item{model}{A character string specifying the underlying model.
The available functional form are \code{"mixture"} and \code{"promotion"}
correspond to the mixture cure model and the bounded cumulative model, respectively.}

\item{nfolds}{An optional integer value specifying the number of folds.
The default value is 5.}

\item{lambda1, lambda2}{An option for specifying the tuning parameter used in penalization.
When this is unspecified or has a \code{NULL} value,
penalization will not be applied and \code{pCure()} will uses all covariates
specified in the formulas.
Alternatively, this can be specified as a vector numeric vector of non-negative values
or "auto" for auto selection.}

\item{exclude1, exclude2}{A character string specifying which variables to exclude from variable selection.
Variables matching elements in this string will not be penalized during the variable selection process.
in variable selection.}

\item{penalty1, penalty2}{A character string specifying the penalty function.
The available options are \code{"lasso"} and \code{"scad"}.}

\item{control}{A list of control parameters. See detail.}
}
\value{
An object of class \code{"pCure"} representing a cure model fit.
}
\description{
Fits either a mixture cure model or a bounded cumulative hazard (promotion time) model
with pseudo-observation approach.
}
\examples{
## Function to generate simulated data under the PHMC model
simMC <- function(n) {
  p <- 10
  a <- c(1, 0, -1, 0, 0, 0, 0, 0, 0, 0) # incidence coefs.
  b <- c(-1, 0, 1, 0, 0, 0, 0, 0, 0, 0) # latency coefs.
  X <- data.frame(x = matrix(runif(n * p), n))
  X$x.3 <- 1 * (X$x.3 > .5)
  X$x.4 <- 1 * (X$x.4 > .5)
  X[,5:10] <- apply(X[,5:10], 2, qnorm)  
  time <- -3 * exp(-colSums(b * t(X))) * log(runif(n))
  cure.prob <- 1 / (1 + exp(-2 - colSums(a * t(X))))
  Y <- rbinom(n, 1, cure.prob) 
  cen <- rexp(n, .02)
  dat <- NULL  
  dat$Time <- pmin(time / Y, cen)
  dat$Status <- 1 * (dat$Time == time)
  data.frame(dat, X)
}

## Fix seed and generate data
set.seed(1); datMC <- simMC(200)

## Oracle model with an unpenalized PHMC model
summary(fit1 <- pCure(~ x.1 + x.3, ~ x.1 + x.3, Time, Status, datMC))


## Penalized PHMC model with tuning parameters selected by 10-fold cross validation
## User specifies the range of tuning parameters
summary(fit2 <- pCure(~ ., ~ ., Time, Status, datMC, lambda1 = 1:10 / 10, lambda2 = 1:10 / 10))

## Penalized PHMC model given tuning parameters
summary(update(fit2, lambda1 = 0.7, lambda2 = 0.4))
}
\references{
Su, C.-L., Chiou, S., Lin, F.-C., and Platt, R. W. (2022)
Analysis of survival data with cure fraction and variable selection: A pseudo-observations approach
\emph{Statistical Methods in Medical Research}, \bold{31}(11): 2037--2053.
}
