/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5private.h"
#include "H5Eprivate.h"
#include "H5MMprivate.h" 
#include "H5Oprivate.h"
#include "H5VMprivate.h"

typedef struct H5VM_memcpy_ud_t {
    unsigned char       *dst; 
    const unsigned char *src; 
} H5VM_memcpy_ud_t;

#define H5VM_HYPER_NDIMS H5O_LAYOUT_NDIMS

static void H5VM__stride_optimize1(unsigned *np , hsize_t *elmt_size ,
                                   const hsize_t *size, hsize_t *stride1);
static void H5VM__stride_optimize2(unsigned *np , hsize_t *elmt_size ,
                                   const hsize_t *size, hsize_t *stride1, hsize_t *stride2);

static void
H5VM__stride_optimize1(unsigned *np , hsize_t *elmt_size , const hsize_t *size,
                       hsize_t *stride1)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(1 == H5VM_vector_reduce_product(0, NULL));

    
    while (*np && stride1[*np - 1] > 0 && (hsize_t)(stride1[*np - 1]) == *elmt_size) {
        *elmt_size *= size[*np - 1];
        if (--*np)
            stride1[*np - 1] += size[*np] * stride1[*np];
    }

    FUNC_LEAVE_NOAPI_VOID
}

static void
H5VM__stride_optimize2(unsigned *np , hsize_t *elmt_size , const hsize_t *size,
                       hsize_t *stride1, hsize_t *stride2)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(1 == H5VM_vector_reduce_product(0, NULL));
    assert(*elmt_size > 0);

    

    
    switch (*np) {
        case 1: 
            if (stride1[0] == *elmt_size && stride2[0] == *elmt_size) {
                *elmt_size *= size[0];
                --*np; 
            }          
            break;

        case 2: 
            if (stride1[1] == *elmt_size && stride2[1] == *elmt_size) {
                *elmt_size *= size[1];
                --*np; 
                stride1[0] += size[1] * stride1[1];
                stride2[0] += size[1] * stride2[1];

                if (stride1[0] == *elmt_size && stride2[0] == *elmt_size) {
                    *elmt_size *= size[0];
                    --*np; 
                }          
            }              
            break;

        case 3: 
            if (stride1[2] == *elmt_size && stride2[2] == *elmt_size) {
                *elmt_size *= size[2];
                --*np; 
                stride1[1] += size[2] * stride1[2];
                stride2[1] += size[2] * stride2[2];

                if (stride1[1] == *elmt_size && stride2[1] == *elmt_size) {
                    *elmt_size *= size[1];
                    --*np; 
                    stride1[0] += size[1] * stride1[1];
                    stride2[0] += size[1] * stride2[1];

                    if (stride1[0] == *elmt_size && stride2[0] == *elmt_size) {
                        *elmt_size *= size[0];
                        --*np; 
                    }          
                }              
            }                  
            break;

        case 4: 
            if (stride1[3] == *elmt_size && stride2[3] == *elmt_size) {
                *elmt_size *= size[3];
                --*np; 
                stride1[2] += size[3] * stride1[3];
                stride2[2] += size[3] * stride2[3];

                if (stride1[2] == *elmt_size && stride2[2] == *elmt_size) {
                    *elmt_size *= size[2];
                    --*np; 
                    stride1[1] += size[2] * stride1[2];
                    stride2[1] += size[2] * stride2[2];

                    if (stride1[1] == *elmt_size && stride2[1] == *elmt_size) {
                        *elmt_size *= size[1];
                        --*np; 
                        stride1[0] += size[1] * stride1[1];
                        stride2[0] += size[1] * stride2[1];

                        if (stride1[0] == *elmt_size && stride2[0] == *elmt_size) {
                            *elmt_size *= size[0];
                            --*np; 
                        }          
                    }              
                }                  
            }                      
            break;

        default:
            while (*np && stride1[*np - 1] == *elmt_size && stride2[*np - 1] == *elmt_size) {
                *elmt_size *= size[*np - 1];
                if (--*np) {
                    stride1[*np - 1] += size[*np] * stride1[*np];
                    stride2[*np - 1] += size[*np] * stride2[*np];
                }
            }
            break;
    } 

    FUNC_LEAVE_NOAPI_VOID
}

hsize_t
H5VM_hyper_stride(unsigned n, const hsize_t *size, const hsize_t *total_size, const hsize_t *offset,
                  hsize_t *stride )
{
    hsize_t skip;      
    hsize_t acc;       
    int     i;         
    hsize_t ret_value; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(n <= H5VM_HYPER_NDIMS);
    assert(size);
    assert(total_size);
    assert(stride);

    
    assert(n > 0);
    stride[n - 1] = 1;
    skip          = offset ? offset[n - 1] : 0;

    switch (n) {
        case 2: 
            assert(total_size[1] >= size[1]);
            stride[0] = total_size[1] - size[1]; 
            acc       = total_size[1];
            skip += acc * (offset ? offset[0] : 0);
            break;

        case 3: 
            assert(total_size[2] >= size[2]);
            stride[1] = total_size[2] - size[2]; 
            acc       = total_size[2];
            skip += acc * (offset ? (hsize_t)offset[1] : 0);

            assert(total_size[1] >= size[1]);
            stride[0] = acc * (total_size[1] - size[1]); 
            acc *= total_size[1];
            skip += acc * (offset ? (hsize_t)offset[0] : 0);
            break;

        case 4: 
            assert(total_size[3] >= size[3]);
            stride[2] = total_size[3] - size[3]; 
            acc       = total_size[3];
            skip += acc * (offset ? (hsize_t)offset[2] : 0);

            assert(total_size[2] >= size[2]);
            stride[1] = acc * (total_size[2] - size[2]); 
            acc *= total_size[2];
            skip += acc * (offset ? (hsize_t)offset[1] : 0);

            assert(total_size[1] >= size[1]);
            stride[0] = acc * (total_size[1] - size[1]); 
            acc *= total_size[1];
            skip += acc * (offset ? (hsize_t)offset[0] : 0);
            break;

        default:
            
            for (i = (int)(n - 2), acc = 1; i >= 0; --i) {
                assert(total_size[i + 1] >= size[i + 1]);
                stride[i] = acc * (total_size[i + 1] - size[i + 1]); 
                acc *= total_size[i + 1];
                skip += acc * (offset ? (hsize_t)offset[i] : 0);
            }
            break;
    } 

    
    ret_value = skip;

    FUNC_LEAVE_NOAPI(ret_value)
}

htri_t
H5VM_hyper_eq(unsigned n, const hsize_t *offset1, const hsize_t *size1, const hsize_t *offset2,
              const hsize_t *size2)
{
    hsize_t  nelmts1 = 1, nelmts2 = 1;
    unsigned i;
    htri_t   ret_value = true; 

    
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (n == 0)
        HGOTO_DONE(true);

    for (i = 0; i < n; i++) {
        if ((offset1 ? offset1[i] : 0) != (offset2 ? offset2[i] : 0))
            HGOTO_DONE(false);
        if ((size1 ? size1[i] : 0) != (size2 ? size2[i] : 0))
            HGOTO_DONE(false);
        if (0 == (nelmts1 *= (size1 ? size1[i] : 0)))
            HGOTO_DONE(false);
        if (0 == (nelmts2 *= (size2 ? size2[i] : 0)))
            HGOTO_DONE(false);
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

herr_t
H5VM_hyper_fill(unsigned n, const hsize_t *_size, const hsize_t *total_size, const hsize_t *offset,
                void *_dst, unsigned fill_value)
{
    uint8_t *dst = (uint8_t *)_dst;        
    hsize_t  size[H5VM_HYPER_NDIMS];       
    hsize_t  dst_stride[H5VM_HYPER_NDIMS]; 
    hsize_t  dst_start;                    
    hsize_t  elmt_size = 1;                
    herr_t   ret_value;                    
#ifndef NDEBUG
    unsigned u;
#endif

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(n > 0 && n <= H5VM_HYPER_NDIMS);
    assert(_size);
    assert(total_size);
    assert(dst);
#ifndef NDEBUG
    for (u = 0; u < n; u++) {
        assert(_size[u] > 0);
        assert(total_size[u] > 0);
    }
#endif

    
    H5VM_vector_cpy(n, size, _size);

    
    dst_start = H5VM_hyper_stride(n, size, total_size, offset, dst_stride);
    H5VM__stride_optimize1(&n, &elmt_size, size, dst_stride);

    
    ret_value = H5VM_stride_fill(n, elmt_size, size, dst_stride, dst + dst_start, fill_value);

    FUNC_LEAVE_NOAPI(ret_value)
}

herr_t
H5VM_hyper_copy(unsigned n, const hsize_t *_size, const hsize_t *dst_size, const hsize_t *dst_offset,
                void *_dst, const hsize_t *src_size, const hsize_t *src_offset, const void *_src)
{
    const uint8_t *src = (const uint8_t *)_src;  
    uint8_t       *dst = (uint8_t *)_dst;        
    hsize_t        size[H5VM_HYPER_NDIMS];       
    hsize_t        src_stride[H5VM_HYPER_NDIMS]; 
    hsize_t        dst_stride[H5VM_HYPER_NDIMS]; 
    hsize_t        dst_start, src_start;         
    hsize_t        elmt_size = 1;                
    herr_t         ret_value;                    
#ifndef NDEBUG
    unsigned u;
#endif

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(n > 0 && n <= H5VM_HYPER_NDIMS);
    assert(_size);
    assert(dst_size);
    assert(src_size);
    assert(dst);
    assert(src);
#ifndef NDEBUG
    for (u = 0; u < n; u++) {
        assert(_size[u] > 0);
        assert(dst_size[u] > 0);
        assert(src_size[u] > 0);
    }
#endif

    
    H5VM_vector_cpy(n, size, _size);

    
#ifdef NO_INLINED_CODE
    dst_start = H5VM_hyper_stride(n, size, dst_size, dst_offset, dst_stride);
    src_start = H5VM_hyper_stride(n, size, src_size, src_offset, src_stride);
#else  
    
    {
        hsize_t dst_acc; 
        hsize_t src_acc; 
        int     ii;      

        
        assert(n > 0);
        dst_stride[n - 1] = 1;
        src_stride[n - 1] = 1;
        dst_start         = dst_offset ? dst_offset[n - 1] : 0;
        src_start         = src_offset ? src_offset[n - 1] : 0;

        
        switch (n) {
            case 2:
                assert(dst_size[1] >= size[1]);
                assert(src_size[1] >= size[1]);
                dst_stride[0] = dst_size[1] - size[1]; 
                src_stride[0] = src_size[1] - size[1]; 
                dst_acc       = dst_size[1];
                src_acc       = src_size[1];
                dst_start += dst_acc * (dst_offset ? dst_offset[0] : 0);
                src_start += src_acc * (src_offset ? src_offset[0] : 0);
                break;

            case 3:
                assert(dst_size[2] >= size[2]);
                assert(src_size[2] >= size[2]);
                dst_stride[1] = dst_size[2] - size[2]; 
                src_stride[1] = src_size[2] - size[2]; 
                dst_acc       = dst_size[2];
                src_acc       = src_size[2];
                dst_start += dst_acc * (dst_offset ? dst_offset[1] : 0);
                src_start += src_acc * (src_offset ? src_offset[1] : 0);

                assert(dst_size[1] >= size[1]);
                assert(src_size[1] >= size[1]);
                dst_stride[0] = dst_acc * (dst_size[1] - size[1]); 
                src_stride[0] = src_acc * (src_size[1] - size[1]); 
                dst_acc *= dst_size[1];
                src_acc *= src_size[1];
                dst_start += dst_acc * (dst_offset ? dst_offset[0] : 0);
                src_start += src_acc * (src_offset ? src_offset[0] : 0);
                break;

            case 4:
                assert(dst_size[3] >= size[3]);
                assert(src_size[3] >= size[3]);
                dst_stride[2] = dst_size[3] - size[3]; 
                src_stride[2] = src_size[3] - size[3]; 
                dst_acc       = dst_size[3];
                src_acc       = src_size[3];
                dst_start += dst_acc * (dst_offset ? dst_offset[2] : 0);
                src_start += src_acc * (src_offset ? src_offset[2] : 0);

                assert(dst_size[2] >= size[2]);
                assert(src_size[2] >= size[2]);
                dst_stride[1] = dst_acc * (dst_size[2] - size[2]); 
                src_stride[1] = src_acc * (src_size[2] - size[2]); 
                dst_acc *= dst_size[2];
                src_acc *= src_size[2];
                dst_start += dst_acc * (dst_offset ? dst_offset[1] : 0);
                src_start += src_acc * (src_offset ? src_offset[1] : 0);

                assert(dst_size[1] >= size[1]);
                assert(src_size[1] >= size[1]);
                dst_stride[0] = dst_acc * (dst_size[1] - size[1]); 
                src_stride[0] = src_acc * (src_size[1] - size[1]); 
                dst_acc *= dst_size[1];
                src_acc *= src_size[1];
                dst_start += dst_acc * (dst_offset ? dst_offset[0] : 0);
                src_start += src_acc * (src_offset ? src_offset[0] : 0);
                break;

            default:
                
                for (ii = (int)(n - 2), dst_acc = 1, src_acc = 1; ii >= 0; --ii) {
                    assert(dst_size[ii + 1] >= size[ii + 1]);
                    assert(src_size[ii + 1] >= size[ii + 1]);
                    dst_stride[ii] = dst_acc * (dst_size[ii + 1] - size[ii + 1]); 
                    src_stride[ii] = src_acc * (src_size[ii + 1] - size[ii + 1]); 
                    dst_acc *= dst_size[ii + 1];
                    src_acc *= src_size[ii + 1];
                    dst_start += dst_acc * (dst_offset ? dst_offset[ii] : 0);
                    src_start += src_acc * (src_offset ? src_offset[ii] : 0);
                }
                break;
        } 
    }
#endif 

    
    H5VM__stride_optimize2(&n, &elmt_size, size, dst_stride, src_stride);

    
    ret_value =
        H5VM_stride_copy(n, elmt_size, size, dst_stride, dst + dst_start, src_stride, src + src_start);

    FUNC_LEAVE_NOAPI(ret_value)
}

herr_t
H5VM_stride_fill(unsigned n, hsize_t elmt_size, const hsize_t *size, const hsize_t *stride, void *_dst,
                 unsigned fill_value)
{
    uint8_t *dst = (uint8_t *)_dst; 
    hsize_t  idx[H5VM_HYPER_NDIMS]; 
    hsize_t  nelmts;                
    hsize_t  i;                     
    int      j;                     
    bool     carry;                 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(elmt_size < SIZE_MAX);

    H5VM_vector_cpy(n, idx, size);
    nelmts = H5VM_vector_reduce_product(n, size);
    for (i = 0; i < nelmts; i++) {
        
        H5_CHECK_OVERFLOW(elmt_size, hsize_t, size_t);
        memset(dst, (int)fill_value, (size_t)elmt_size);

        
        for (j = (int)(n - 1), carry = true; j >= 0 && carry; --j) {
            dst += stride[j];

            if (--idx[j])
                carry = false;
            else {
                assert(size);
                idx[j] = size[j];
            } 
        }
    }

    FUNC_LEAVE_NOAPI(SUCCEED)
}

herr_t
H5VM_stride_copy(unsigned n, hsize_t elmt_size, const hsize_t *size, const hsize_t *dst_stride, void *_dst,
                 const hsize_t *src_stride, const void *_src)
{
    uint8_t       *dst = (uint8_t *)_dst;       
    const uint8_t *src = (const uint8_t *)_src; 
    hsize_t        idx[H5VM_HYPER_NDIMS];       
    hsize_t        nelmts;                      
    hsize_t        i;                           
    int            j;                           
    bool           carry;                       

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(elmt_size < SIZE_MAX);

    if (n) {
        H5VM_vector_cpy(n, idx, size);
        nelmts = H5VM_vector_reduce_product(n, size);
        for (i = 0; i < nelmts; i++) {

            
            H5_CHECK_OVERFLOW(elmt_size, hsize_t, size_t);
            H5MM_memcpy(dst, src, (size_t)elmt_size);

            
            for (j = (int)(n - 1), carry = true; j >= 0 && carry; --j) {
                src += src_stride[j];
                dst += dst_stride[j];

                if (--idx[j])
                    carry = false;
                else {
                    assert(size);
                    idx[j] = size[j];
                }
            }
        }
    }
    else {
        H5_CHECK_OVERFLOW(elmt_size, hsize_t, size_t);
        H5MM_memcpy(dst, src, (size_t)elmt_size);
    }

    FUNC_LEAVE_NOAPI(SUCCEED)
}

herr_t
H5VM_stride_copy_s(unsigned n, hsize_t elmt_size, const hsize_t *size, const hssize_t *dst_stride, void *_dst,
                   const hssize_t *src_stride, const void *_src)
{
    uint8_t       *dst = (uint8_t *)_dst;       
    const uint8_t *src = (const uint8_t *)_src; 
    hsize_t        idx[H5VM_HYPER_NDIMS];       
    hsize_t        nelmts;                      
    hsize_t        i;                           
    int            j;                           
    bool           carry;                       

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(elmt_size < SIZE_MAX);

    if (n) {
        H5VM_vector_cpy(n, idx, size);
        nelmts = H5VM_vector_reduce_product(n, size);
        for (i = 0; i < nelmts; i++) {

            
            H5_CHECK_OVERFLOW(elmt_size, hsize_t, size_t);
            H5MM_memcpy(dst, src, (size_t)elmt_size);

            
            for (j = (int)(n - 1), carry = true; j >= 0 && carry; --j) {
                src += src_stride[j];
                dst += dst_stride[j];

                if (--idx[j])
                    carry = false;
                else {
                    assert(size);
                    idx[j] = size[j];
                }
            }
        }
    }
    else {
        H5_CHECK_OVERFLOW(elmt_size, hsize_t, size_t);
        H5MM_memcpy(dst, src, (size_t)elmt_size);
    }

    FUNC_LEAVE_NOAPI(SUCCEED)
}

herr_t
H5VM_array_fill(void *_dst, const void *src, size_t size, size_t count)
{
    size_t   copy_size;             
    size_t   copy_items;            
    size_t   items_left;            
    uint8_t *dst = (uint8_t *)_dst; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(dst);
    assert(src);
    assert(size < SIZE_MAX && size > 0);
    assert(count < SIZE_MAX && count > 0);

    H5MM_memcpy(dst, src, size); 

    
    copy_size  = size;
    copy_items = 1;
    items_left = count - 1;
    dst += size;

    
    while (items_left >= copy_items) {
        H5MM_memcpy(dst, _dst, copy_size); 
        dst += copy_size;                  
        items_left -= copy_items;          

        copy_size *= 2;  
        copy_items *= 2; 
    }                    
    if (items_left > 0)  
        H5MM_memcpy(dst, _dst, items_left * size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

void
H5VM_array_down(unsigned n, const hsize_t *total_size, hsize_t *down)
{
    hsize_t acc; 
    int     i;   

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(n <= H5VM_HYPER_NDIMS);
    assert(total_size);
    assert(down);

    
    for (i = (int)(n - 1), acc = 1; i >= 0; i--) {
        down[i] = acc;
        acc *= total_size[i];
    }

    FUNC_LEAVE_NOAPI_VOID
} 

hsize_t
H5VM_array_offset_pre(unsigned n, const hsize_t *acc, const hsize_t *offset)
{
    unsigned u;         
    hsize_t  ret_value; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    assert(n <= H5VM_HYPER_NDIMS);
    assert(acc);
    assert(offset);

    
    for (u = 0, ret_value = 0; u < n; u++)
        ret_value += acc[u] * offset[u];

    FUNC_LEAVE_NOAPI(ret_value)
} 

hsize_t
H5VM_array_offset(unsigned n, const hsize_t *total_size, const hsize_t *offset)
{
    hsize_t acc_arr[H5VM_HYPER_NDIMS]; 
    hsize_t ret_value;                 

    FUNC_ENTER_NOAPI_NOERR

    assert(n <= H5VM_HYPER_NDIMS);
    assert(total_size);
    assert(offset);

    
    H5VM_array_down(n, total_size, acc_arr);

    
    ret_value = H5VM_array_offset_pre(n, acc_arr, offset);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VM_array_calc_pre(hsize_t offset, unsigned n, const hsize_t *down, hsize_t *coords)
{
    unsigned u; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(n <= H5VM_HYPER_NDIMS);
    assert(coords);

    
    for (u = 0; u < n; u++) {
        coords[u] = offset / down[u];
        offset %= down[u];
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5VM_array_calc(hsize_t offset, unsigned n, const hsize_t *total_size, hsize_t *coords)
{
    hsize_t idx[H5VM_HYPER_NDIMS]; 
    herr_t  ret_value = SUCCEED;   

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(n <= H5VM_HYPER_NDIMS);
    assert(total_size);
    assert(coords);

    
    H5VM_array_down(n, total_size, idx);

    
    if (H5VM_array_calc_pre(offset, n, idx, coords) < 0)
        HGOTO_ERROR(H5E_INTERNAL, H5E_BADVALUE, FAIL, "can't compute coordinates");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hsize_t
H5VM_chunk_index(unsigned ndims, const hsize_t *coord, const hsize_t *chunk, const hsize_t *down_nchunks)
{
    hsize_t scaled_coord[H5VM_HYPER_NDIMS]; 
    hsize_t chunk_idx;                      

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(ndims <= H5VM_HYPER_NDIMS);
    assert(coord);
    assert(chunk);
    assert(down_nchunks);

    
    chunk_idx = H5VM_chunk_index_scaled(ndims, coord, chunk, down_nchunks, scaled_coord);

    FUNC_LEAVE_NOAPI(chunk_idx)
} 

void
H5VM_chunk_scaled(unsigned ndims, const hsize_t *coord, const hsize_t *chunk, hsize_t *scaled)
{
    unsigned u; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(ndims <= H5VM_HYPER_NDIMS);
    assert(coord);
    assert(chunk);
    assert(scaled);

    
    
    for (u = 0; u < ndims; u++)
        scaled[u] = coord[u] / chunk[u];

    FUNC_LEAVE_NOAPI_VOID
} 

hsize_t
H5VM_chunk_index_scaled(unsigned ndims, const hsize_t *coord, const hsize_t *chunk,
                        const hsize_t *down_nchunks, hsize_t *scaled)
{
    hsize_t  chunk_idx; 
    unsigned u;         

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(ndims <= H5VM_HYPER_NDIMS);
    assert(coord);
    assert(chunk);
    assert(down_nchunks);
    assert(scaled);

    
    
    for (u = 0; u < ndims; u++)
        scaled[u] = coord[u] / chunk[u];

    
    chunk_idx = H5VM_array_offset_pre(ndims, down_nchunks, scaled);

    FUNC_LEAVE_NOAPI(chunk_idx)
} 

ssize_t
H5VM_opvv(size_t dst_max_nseq, size_t *dst_curr_seq, size_t dst_len_arr[], hsize_t dst_off_arr[],
          size_t src_max_nseq, size_t *src_curr_seq, size_t src_len_arr[], hsize_t src_off_arr[],
          H5VM_opvv_func_t op, void *op_data)
{
    hsize_t *max_dst_off_ptr, *max_src_off_ptr; 
    hsize_t *dst_off_ptr, *src_off_ptr;         
    size_t  *dst_len_ptr, *src_len_ptr;         
    hsize_t  tmp_dst_off, tmp_src_off;          
    size_t   tmp_dst_len, tmp_src_len;          
    size_t   acc_len;                           
    ssize_t  ret_value = 0;                     

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(dst_curr_seq);
    assert(*dst_curr_seq < dst_max_nseq);
    assert(dst_len_arr);
    assert(dst_off_arr);
    assert(src_curr_seq);
    assert(*src_curr_seq < src_max_nseq);
    assert(src_len_arr);
    assert(src_off_arr);
    assert(op);

    
    dst_len_ptr = dst_len_arr + *dst_curr_seq;
    dst_off_ptr = dst_off_arr + *dst_curr_seq;
    src_len_ptr = src_len_arr + *src_curr_seq;
    src_off_ptr = src_off_arr + *src_curr_seq;

    
    tmp_dst_len = *dst_len_ptr;
    tmp_dst_off = *dst_off_ptr;
    tmp_src_len = *src_len_ptr;
    tmp_src_off = *src_off_ptr;

    
    max_dst_off_ptr = dst_off_arr + dst_max_nseq;
    max_src_off_ptr = src_off_arr + src_max_nseq;

    
    

    
    if (tmp_src_len < tmp_dst_len) {
src_smaller:
        acc_len = 0;
        do {
            
            if ((*op)(tmp_dst_off, tmp_src_off, tmp_src_len, op_data) < 0)
                HGOTO_ERROR(H5E_INTERNAL, H5E_CANTOPERATE, FAIL, "can't perform operation");

            
            acc_len += tmp_src_len;

            
            tmp_dst_off += tmp_src_len;
            tmp_dst_len -= tmp_src_len;

            
            src_off_ptr++;
            if (src_off_ptr >= max_src_off_ptr) {
                
                *dst_off_ptr = tmp_dst_off;
                *dst_len_ptr = tmp_dst_len;

                
                goto finished;
            } 
            tmp_src_off = *src_off_ptr;

            
            src_len_ptr++;
            tmp_src_len = *src_len_ptr;
        } while (tmp_src_len < tmp_dst_len);

        
        ret_value += (ssize_t)acc_len;

        
        if (tmp_dst_len < tmp_src_len)
            goto dst_smaller;
        else
            goto equal;
    } 
    
    else if (tmp_dst_len < tmp_src_len) {
dst_smaller:
        acc_len = 0;
        do {
            
            if ((*op)(tmp_dst_off, tmp_src_off, tmp_dst_len, op_data) < 0)
                HGOTO_ERROR(H5E_INTERNAL, H5E_CANTOPERATE, FAIL, "can't perform operation");

            
            acc_len += tmp_dst_len;

            
            tmp_src_off += tmp_dst_len;
            tmp_src_len -= tmp_dst_len;

            
            dst_off_ptr++;
            if (dst_off_ptr >= max_dst_off_ptr) {
                
                *src_off_ptr = tmp_src_off;
                *src_len_ptr = tmp_src_len;

                
                goto finished;
            } 
            tmp_dst_off = *dst_off_ptr;

            
            dst_len_ptr++;
            tmp_dst_len = *dst_len_ptr;
        } while (tmp_dst_len < tmp_src_len);

        
        ret_value += (ssize_t)acc_len;

        
        if (tmp_src_len < tmp_dst_len)
            goto src_smaller;
        else
            goto equal;
    } 
    
    else {
equal:
        acc_len = 0;
        do {
            
            if ((*op)(tmp_dst_off, tmp_src_off, tmp_dst_len, op_data) < 0)
                HGOTO_ERROR(H5E_INTERNAL, H5E_CANTOPERATE, FAIL, "can't perform operation");

            
            acc_len += tmp_dst_len;

            
            src_off_ptr++;
            dst_off_ptr++;
            if (src_off_ptr >= max_src_off_ptr || dst_off_ptr >= max_dst_off_ptr)
                
                goto finished;
            tmp_src_off = *src_off_ptr;
            tmp_dst_off = *dst_off_ptr;

            
            src_len_ptr++;
            tmp_src_len = *src_len_ptr;

            
            dst_len_ptr++;
            tmp_dst_len = *dst_len_ptr;
        } while (tmp_dst_len == tmp_src_len);

        
        ret_value += (ssize_t)acc_len;

        
        if (tmp_dst_len < tmp_src_len)
            goto dst_smaller;
        else
            goto src_smaller;
    } 

finished:
    
    ret_value += (ssize_t)acc_len;

    
    *dst_curr_seq = (size_t)(dst_off_ptr - dst_off_arr);
    *src_curr_seq = (size_t)(src_off_ptr - src_off_arr);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

ssize_t
H5VM_memcpyvv(void *_dst, size_t dst_max_nseq, size_t *dst_curr_seq, size_t dst_len_arr[],
              hsize_t dst_off_arr[], const void *_src, size_t src_max_nseq, size_t *src_curr_seq,
              size_t src_len_arr[], hsize_t src_off_arr[])
{
    unsigned char       *dst;                   
    const unsigned char *src;                   
    hsize_t *max_dst_off_ptr, *max_src_off_ptr; 
    hsize_t *dst_off_ptr, *src_off_ptr;         
    size_t  *dst_len_ptr, *src_len_ptr;         
    size_t   tmp_dst_len;                       
    size_t   tmp_src_len;                       
    size_t   acc_len;                           
    ssize_t  ret_value = 0;                     

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(_dst);
    assert(dst_curr_seq);
    assert(*dst_curr_seq < dst_max_nseq);
    assert(dst_len_arr);
    assert(dst_off_arr);
    assert(_src);
    assert(src_curr_seq);
    assert(*src_curr_seq < src_max_nseq);
    assert(src_len_arr);
    assert(src_off_arr);

    
    dst_len_ptr = dst_len_arr + *dst_curr_seq;
    dst_off_ptr = dst_off_arr + *dst_curr_seq;
    src_len_ptr = src_len_arr + *src_curr_seq;
    src_off_ptr = src_off_arr + *src_curr_seq;

    
    tmp_dst_len = *dst_len_ptr;
    tmp_src_len = *src_len_ptr;

    
    max_dst_off_ptr = dst_off_arr + dst_max_nseq;
    max_src_off_ptr = src_off_arr + src_max_nseq;

    
    dst = (unsigned char *)_dst + *dst_off_ptr;
    src = (const unsigned char *)_src + *src_off_ptr;

    
    

    
    if (tmp_src_len < tmp_dst_len) {
src_smaller:
        acc_len = 0;
        do {
            
            H5MM_memcpy(dst, src, tmp_src_len);

            
            acc_len += tmp_src_len;

            
            tmp_dst_len -= tmp_src_len;

            
            src_off_ptr++;
            if (src_off_ptr >= max_src_off_ptr) {
                
                *dst_off_ptr += acc_len;
                *dst_len_ptr = tmp_dst_len;

                
                goto finished;
            } 

            
            dst += tmp_src_len;

            
            src_len_ptr++;
            tmp_src_len = *src_len_ptr;
            src         = (const unsigned char *)_src + *src_off_ptr;
        } while (tmp_src_len < tmp_dst_len);

        
        ret_value += (ssize_t)acc_len;

        
        if (tmp_dst_len < tmp_src_len)
            goto dst_smaller;
        else
            goto equal;
    } 
    
    else if (tmp_dst_len < tmp_src_len) {
dst_smaller:
        acc_len = 0;
        do {
            
            H5MM_memcpy(dst, src, tmp_dst_len);

            
            acc_len += tmp_dst_len;

            
            tmp_src_len -= tmp_dst_len;

            
            dst_off_ptr++;
            if (dst_off_ptr >= max_dst_off_ptr) {
                
                *src_off_ptr += acc_len;
                *src_len_ptr = tmp_src_len;

                
                goto finished;
            } 

            
            src += tmp_dst_len;

            
            dst_len_ptr++;
            tmp_dst_len = *dst_len_ptr;
            dst         = (unsigned char *)_dst + *dst_off_ptr;
        } while (tmp_dst_len < tmp_src_len);

        
        ret_value += (ssize_t)acc_len;

        
        if (tmp_src_len < tmp_dst_len)
            goto src_smaller;
        else
            goto equal;
    } 
    
    else {
equal:
        acc_len = 0;
        do {
            
            H5MM_memcpy(dst, src, tmp_dst_len);

            
            acc_len += tmp_dst_len;

            
            src_off_ptr++;
            dst_off_ptr++;
            if (src_off_ptr >= max_src_off_ptr || dst_off_ptr >= max_dst_off_ptr)
                
                goto finished;

            
            src_len_ptr++;
            tmp_src_len = *src_len_ptr;
            src         = (const unsigned char *)_src + *src_off_ptr;

            
            dst_len_ptr++;
            tmp_dst_len = *dst_len_ptr;
            dst         = (unsigned char *)_dst + *dst_off_ptr;
        } while (tmp_dst_len == tmp_src_len);

        
        ret_value += (ssize_t)acc_len;

        
        if (tmp_dst_len < tmp_src_len)
            goto dst_smaller;
        else
            goto src_smaller;
    } 

finished:
    
    ret_value += (ssize_t)acc_len;

    
    *dst_curr_seq = (size_t)(dst_off_ptr - dst_off_arr);
    *src_curr_seq = (size_t)(src_off_ptr - src_off_arr);

    FUNC_LEAVE_NOAPI(ret_value)
} 
