/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5F_FRIEND      
#include "H5MFmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Fpkg.h"     
#include "H5MFpkg.h"    

#define EXTEND_THRESHOLD 0.10F

static herr_t  H5MF__aggr_free(H5F_t *f, H5FD_mem_t type, H5F_blk_aggr_t *aggr);
static haddr_t H5MF__aggr_alloc(H5F_t *f, H5F_blk_aggr_t *aggr, H5F_blk_aggr_t *other_aggr, H5FD_mem_t type,
                                hsize_t size);
static herr_t  H5MF__aggr_reset(H5F_t *f, H5F_blk_aggr_t *aggr);
static htri_t  H5MF__aggr_can_shrink_eoa(H5F_t *f, H5FD_mem_t type, H5F_blk_aggr_t *aggr);

haddr_t
H5MF_aggr_vfd_alloc(H5F_t *f, H5FD_mem_t alloc_type, hsize_t size)
{
    haddr_t ret_value = HADDR_UNDEF; 

    FUNC_ENTER_NOAPI(HADDR_UNDEF)
#ifdef H5MF_AGGR_DEBUG
    Rfprintf(Rstderr, "%s: alloc_type = %u, size = %" PRIuHSIZE "\n", __func__, (unsigned)alloc_type, size);
#endif 

    
    assert(f);
    assert(f->shared);
    assert(f->shared->lf);
    assert(size > 0);

    
    if (alloc_type != H5FD_MEM_DRAW && alloc_type != H5FD_MEM_GHEAP) {
        
        if (HADDR_UNDEF == (ret_value = H5MF__aggr_alloc(f, &(f->shared->meta_aggr), &(f->shared->sdata_aggr),
                                                         alloc_type, size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, HADDR_UNDEF, "can't allocate metadata");
    } 
    else {
        
        if (HADDR_UNDEF == (ret_value = H5MF__aggr_alloc(f, &(f->shared->sdata_aggr), &(f->shared->meta_aggr),
                                                         H5FD_MEM_DRAW, size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, HADDR_UNDEF, "can't allocate raw data");
    } 

    
    assert(H5_addr_le((ret_value + size), f->shared->tmp_addr));

done:
#ifdef H5MF_AGGR_DEBUG
    Rfprintf(Rstderr, "%s: Leaving: ret_value = %" PRIuHADDR ", size = %" PRIuHSIZE "\n", __func__, ret_value,
            size);
#endif 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static haddr_t
H5MF__aggr_alloc(H5F_t *f, H5F_blk_aggr_t *aggr, H5F_blk_aggr_t *other_aggr, H5FD_mem_t type, hsize_t size)
{
    haddr_t eoa_frag_addr = HADDR_UNDEF; 
    hsize_t eoa_frag_size = 0;           
    haddr_t eoa           = HADDR_UNDEF; 
    haddr_t ret_value     = HADDR_UNDEF; 

    FUNC_ENTER_PACKAGE
#ifdef H5MF_AGGR_DEBUG
    Rfprintf(Rstderr, "%s: type = %u, size = %" PRIuHSIZE "\n", __func__, (unsigned)type, size);
#endif 

    
    assert(f);
    assert(aggr);
    assert(aggr->feature_flag == H5FD_FEAT_AGGREGATE_METADATA ||
           aggr->feature_flag == H5FD_FEAT_AGGREGATE_SMALLDATA);
    assert(other_aggr);
    assert(other_aggr->feature_flag == H5FD_FEAT_AGGREGATE_METADATA ||
           other_aggr->feature_flag == H5FD_FEAT_AGGREGATE_SMALLDATA);
    assert(other_aggr->feature_flag != aggr->feature_flag);
    assert(type >= H5FD_MEM_DEFAULT && type < H5FD_MEM_NTYPES);
    assert(size > 0);

    
    if (HADDR_UNDEF == (eoa = H5F_get_eoa(f, type)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGET, HADDR_UNDEF, "Unable to get eoa");

    

    
    if ((f->shared->feature_flags & aggr->feature_flag) &&
        f->shared->fs_strategy != H5F_FSPACE_STRATEGY_NONE &&
        (!f->shared->closing || !f->shared->fs_persist)) {
#ifdef REPLACE
        if ((f->shared->feature_flags & aggr->feature_flag) &&
            f->shared->fs_strategy != H5F_FSPACE_STRATEGY_NONE && !f->shared->closing) {
#endif
            haddr_t    aggr_frag_addr = HADDR_UNDEF; 
            hsize_t    aggr_frag_size = 0;           
            hsize_t    alignment;                    
            hsize_t    aggr_mis_align = 0;           
            H5FD_mem_t alloc_type, other_alloc_type; 

#ifdef H5MF_AGGR_DEBUG
            Rfprintf(Rstderr, "%s: aggr = {%" PRIuHADDR ", %" PRIuHSIZE ", %" PRIuHSIZE "}\n", __func__,
                    aggr->addr, aggr->tot_size, aggr->size);
#endif 

            
            alignment = H5F_ALIGNMENT(f);
            if (!((alignment > 1) && (size >= H5F_THRESHOLD(f))))
                alignment = 0; 

            
            if (alignment && H5_addr_gt(aggr->addr, 0) &&
                (aggr_mis_align = (aggr->addr + H5F_BASE_ADDR(f)) % alignment)) {
                aggr_frag_addr = aggr->addr;
                aggr_frag_size = alignment - aggr_mis_align;
            } 

            alloc_type =
                aggr->feature_flag == H5FD_FEAT_AGGREGATE_METADATA ? H5FD_MEM_DEFAULT : H5FD_MEM_DRAW;
            other_alloc_type =
                other_aggr->feature_flag == H5FD_FEAT_AGGREGATE_METADATA ? H5FD_MEM_DEFAULT : H5FD_MEM_DRAW;

            
            if ((size + aggr_frag_size) > aggr->size) {
                htri_t extended = false; 

                
                if (size >= aggr->alloc_size) {
                    hsize_t ext_size = size + aggr_frag_size;

                    
                    if (H5_addr_gt((aggr->addr + aggr->size + ext_size), f->shared->tmp_addr))
                        HGOTO_ERROR(H5E_RESOURCE, H5E_BADRANGE, HADDR_UNDEF,
                                    "'normal' file space allocation request will overlap into 'temporary' "
                                    "file space");

                    if ((aggr->addr > 0) &&
                        (extended = H5F__try_extend(f, alloc_type, (aggr->addr + aggr->size), ext_size)) < 0)
                        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, HADDR_UNDEF, "can't extending space");
                    else if (extended) {
                        
                        ret_value = aggr->addr + aggr_frag_size;
                        aggr->addr += ext_size;
                        aggr->tot_size += ext_size;
                    }
                    else {
                        
                        if ((other_aggr->size > 0) &&
                            (H5_addr_eq((other_aggr->addr + other_aggr->size), eoa)) &&
                            (other_aggr->tot_size > other_aggr->size) &&
                            ((other_aggr->tot_size - other_aggr->size) >= other_aggr->alloc_size)) {
                            if (H5MF__aggr_free(f, other_alloc_type, other_aggr) < 0)
                                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTFREE, HADDR_UNDEF,
                                            "can't free aggregation block");
                        } 

                        
                        if (HADDR_UNDEF ==
                            (ret_value = H5F__alloc(f, alloc_type, size, &eoa_frag_addr, &eoa_frag_size)))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, HADDR_UNDEF,
                                        "can't allocate file space");
                    } 
                }     
                else {
                    hsize_t ext_size = aggr->alloc_size;

                    
#ifdef H5MF_AGGR_DEBUG
                    Rfprintf(Rstderr, "%s: Allocating block\n", __func__);
#endif 

                    if (aggr_frag_size > (ext_size - size))
                        ext_size += (aggr_frag_size - (ext_size - size));

                    
                    if (H5_addr_gt((aggr->addr + aggr->size + ext_size), f->shared->tmp_addr))
                        HGOTO_ERROR(H5E_RESOURCE, H5E_BADRANGE, HADDR_UNDEF,
                                    "'normal' file space allocation request will overlap into 'temporary' "
                                    "file space");

                    if ((aggr->addr > 0) &&
                        (extended = H5F__try_extend(f, alloc_type, (aggr->addr + aggr->size), ext_size)) < 0)
                        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, HADDR_UNDEF, "can't extending space");
                    else if (extended) {
                        aggr->addr += aggr_frag_size;
                        aggr->size += (ext_size - aggr_frag_size);
                        aggr->tot_size += ext_size;
                    } 
                    else {
                        haddr_t new_space; 

                        
                        if ((other_aggr->size > 0) &&
                            (H5_addr_eq((other_aggr->addr + other_aggr->size), eoa)) &&
                            (other_aggr->tot_size > other_aggr->size) &&
                            ((other_aggr->tot_size - other_aggr->size) >= other_aggr->alloc_size)) {
                            if (H5MF__aggr_free(f, other_alloc_type, other_aggr) < 0)
                                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTFREE, HADDR_UNDEF,
                                            "can't free aggregation block");
                        } 

                        
                        if (HADDR_UNDEF == (new_space = H5F__alloc(f, alloc_type, aggr->alloc_size,
                                                                   &eoa_frag_addr, &eoa_frag_size)))
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, HADDR_UNDEF,
                                        "can't allocate file space");

                        
                        if (aggr->size > 0)
                            if (H5MF_xfree(f, alloc_type, aggr->addr, aggr->size) < 0)
                                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTFREE, HADDR_UNDEF,
                                            "can't free aggregation block");

                        
                        if (eoa_frag_size && !alignment) {
                            assert(eoa_frag_addr + eoa_frag_size == new_space);
                            aggr->addr     = eoa_frag_addr;
                            aggr->size     = aggr->alloc_size + eoa_frag_size;
                            aggr->tot_size = aggr->size;

                            
                            eoa_frag_addr = HADDR_UNDEF;
                            eoa_frag_size = 0;
                        } 
                        else {
                            
                            aggr->addr     = new_space;
                            aggr->size     = aggr->alloc_size;
                            aggr->tot_size = aggr->alloc_size;
                        } 
                    }     

                    
                    ret_value = aggr->addr;
                    aggr->size -= size;
                    aggr->addr += size;
                } 

                
                if (eoa_frag_size)
                    if (H5MF_xfree(f, alloc_type, eoa_frag_addr, eoa_frag_size) < 0)
                        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTFREE, HADDR_UNDEF, "can't free eoa fragment");

                
                if (extended && aggr_frag_size)
                    if (H5MF_xfree(f, alloc_type, aggr_frag_addr, aggr_frag_size) < 0)
                        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTFREE, HADDR_UNDEF,
                                    "can't free aggregation fragment");
            } 
            else {
                
                ret_value = aggr->addr + aggr_frag_size;
                aggr->size -= (size + aggr_frag_size);
                aggr->addr += (size + aggr_frag_size);

                
                if (aggr_frag_size)
                    if (H5MF_xfree(f, alloc_type, aggr_frag_addr, aggr_frag_size) < 0)
                        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTFREE, HADDR_UNDEF,
                                    "can't free aggregation fragment");
            } 
        }     
        else {
            
            if (HADDR_UNDEF == (ret_value = H5F__alloc(f, type, size, &eoa_frag_addr, &eoa_frag_size)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, HADDR_UNDEF, "can't allocate file space");

            
            if (eoa_frag_size)
                
                if (H5MF_xfree(f, type, eoa_frag_addr, eoa_frag_size) < 0)
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTFREE, HADDR_UNDEF, "can't free eoa fragment");
        } 

        
        assert(H5_addr_le((ret_value + size), f->shared->tmp_addr));

        
        if (H5F_ALIGNMENT(f) && size >= H5F_THRESHOLD(f))
            assert(!((ret_value + H5FD_get_base_addr(f->shared->lf)) % H5F_ALIGNMENT(f)));

done:
#ifdef H5MF_AGGR_DEBUG
        Rfprintf(Rstderr, "%s: ret_value = %" PRIuHADDR "\n", __func__, ret_value);
#endif 
        FUNC_LEAVE_NOAPI(ret_value)
    } 

    
    htri_t H5MF__aggr_try_extend(H5F_t * f, H5F_blk_aggr_t * aggr, H5FD_mem_t type, haddr_t blk_end,
                                 hsize_t extra_requested)
    {
        htri_t ret_value = false; 

        FUNC_ENTER_PACKAGE

        
        assert(f);
        assert(aggr);
        assert(aggr->feature_flag == H5FD_FEAT_AGGREGATE_METADATA ||
               aggr->feature_flag == H5FD_FEAT_AGGREGATE_SMALLDATA);

        
        if (f->shared->feature_flags & aggr->feature_flag) {
            
            if (H5_addr_eq(blk_end, aggr->addr)) {
                haddr_t eoa; 

                
                if (HADDR_UNDEF == (eoa = H5F_get_eoa(f, type)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGET, FAIL, "Unable to get eoa");

                
                if (H5_addr_eq(eoa, aggr->addr + aggr->size)) {
                    
                    if (extra_requested <= (hsize_t)(EXTEND_THRESHOLD * (float)aggr->size)) {
                        aggr->size -= extra_requested;
                        aggr->addr += extra_requested;

                        
                        HGOTO_DONE(true);
                    } 
                    
                    else {
                        hsize_t extra =
                            (extra_requested < aggr->alloc_size) ? aggr->alloc_size : extra_requested;

                        if ((ret_value = H5F__try_extend(f, type, (aggr->addr + aggr->size), extra)) < 0)
                            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTEXTEND, FAIL, "error extending file");
                        else if (ret_value == true) {
                            
                            
                            aggr->addr += extra_requested;

                            
                            aggr->tot_size += extra;

                            
                            
                            aggr->size += extra;
                            aggr->size -= extra_requested;
                        } 
                    }     
                }         
                else {
                    
                    
                    if (aggr->size >= extra_requested) {
                        
                        aggr->size -= extra_requested;
                        aggr->addr += extra_requested;

                        
                        HGOTO_DONE(true);
                    } 
                }     
            }         
        }             

done:
        FUNC_LEAVE_NOAPI(ret_value)
    } 

    
    htri_t H5MF__aggr_can_absorb(const H5F_t *f, const H5F_blk_aggr_t *aggr, const H5MF_free_section_t *sect,
                                 H5MF_shrink_type_t *shrink)
    {
        htri_t ret_value = false; 

        FUNC_ENTER_PACKAGE_NOERR

        
        assert(f);
        assert(aggr);
        assert(aggr->feature_flag == H5FD_FEAT_AGGREGATE_METADATA ||
               aggr->feature_flag == H5FD_FEAT_AGGREGATE_SMALLDATA);
        assert(sect);
        assert(shrink);

        
        if (f->shared->feature_flags & aggr->feature_flag) {
            
            if (H5_addr_eq((sect->sect_info.addr + sect->sect_info.size), aggr->addr) ||
                H5_addr_eq((aggr->addr + aggr->size), sect->sect_info.addr)) {
#ifdef H5MF_AGGR_DEBUG
                Rfprintf(Rstderr,
                        "%s: section {%" PRIuHADDR ", %" PRIuHSIZE "} adjoins aggr = {%" PRIuHADDR
                        ", %" PRIuHSIZE "}\n",
                        "H5MF__aggr_can_absorb", sect->sect_info.addr, sect->sect_info.size, aggr->addr,
                        aggr->size);
#endif 
                
                if ((aggr->size + sect->sect_info.size) >= aggr->alloc_size)
                    *shrink = H5MF_SHRINK_SECT_ABSORB_AGGR;
                else
                    *shrink = H5MF_SHRINK_AGGR_ABSORB_SECT;

                
                HGOTO_DONE(true);
            } 
        }     

done:
        FUNC_LEAVE_NOAPI(ret_value)
    } 

    
    herr_t H5MF__aggr_absorb(const H5F_t H5_ATTR_UNUSED *f, H5F_blk_aggr_t *aggr, H5MF_free_section_t *sect,
                             bool allow_sect_absorb)
    {
        FUNC_ENTER_PACKAGE_NOERR

        
        assert(f);
        assert(aggr);
        assert(aggr->feature_flag == H5FD_FEAT_AGGREGATE_METADATA ||
               aggr->feature_flag == H5FD_FEAT_AGGREGATE_SMALLDATA);
        assert(f->shared->feature_flags & aggr->feature_flag);
        assert(sect);

        
        if ((aggr->size + sect->sect_info.size) >= aggr->alloc_size && allow_sect_absorb) {
            
            if (H5_addr_eq((sect->sect_info.addr + sect->sect_info.size), aggr->addr)) {
#ifdef H5MF_AGGR_DEBUG
                Rfprintf(Rstderr,
                        "%s: aggr {%" PRIuHADDR ", %" PRIuHSIZE "} adjoins front of section = {%" PRIuHADDR
                        ", %" PRIuHSIZE "}\n",
                        "H5MF__aggr_absorb", aggr->addr, aggr->size, sect->sect_info.addr,
                        sect->sect_info.size);
#endif 
                
                sect->sect_info.size += aggr->size;
            } 
            else {
                
                assert(H5_addr_eq((aggr->addr + aggr->size), sect->sect_info.addr));

#ifdef H5MF_AGGR_DEBUG
                Rfprintf(Rstderr,
                        "%s: aggr {%" PRIuHADDR ", %" PRIuHSIZE "} adjoins end of section = {%" PRIuHADDR
                        ", %" PRIuHSIZE "}\n",
                        "H5MF__aggr_absorb", aggr->addr, aggr->size, sect->sect_info.addr,
                        sect->sect_info.size);
#endif 
                
                sect->sect_info.addr -= aggr->size;
                sect->sect_info.size += aggr->size;
            } 

            
            aggr->tot_size = 0;
            aggr->addr     = 0;
            aggr->size     = 0;
        } 
        else {
            
            if (H5_addr_eq((sect->sect_info.addr + sect->sect_info.size), aggr->addr)) {
#ifdef H5MF_AGGR_DEBUG
                Rfprintf(Rstderr,
                        "%s: section {%" PRIuHADDR ", %" PRIuHSIZE "} adjoins front of aggr = {%" PRIuHADDR
                        ", %" PRIuHSIZE "}\n",
                        "H5MF__aggr_absorb", sect->sect_info.addr, sect->sect_info.size, aggr->addr,
                        aggr->size);
#endif 
                
                aggr->addr -= sect->sect_info.size;
                aggr->size += sect->sect_info.size;

                
                aggr->tot_size -= MIN(aggr->tot_size, sect->sect_info.size);
            } 
            else {
                
                assert(H5_addr_eq((aggr->addr + aggr->size), sect->sect_info.addr));

#ifdef H5MF_AGGR_DEBUG
                Rfprintf(Rstderr,
                        "%s: section {%" PRIuHADDR ", %" PRIuHSIZE "} adjoins end of aggr = {%" PRIuHADDR
                        ", %" PRIuHSIZE "}\n",
                        "H5MF__aggr_absorb", sect->sect_info.addr, sect->sect_info.size, aggr->addr,
                        aggr->size);
#endif 
                
                aggr->size += sect->sect_info.size;
            } 
            
            assert(!allow_sect_absorb || (aggr->size < aggr->alloc_size));
        } 

        FUNC_LEAVE_NOAPI(SUCCEED)
    } 

    
    herr_t H5MF__aggr_query(const H5F_t *f, const H5F_blk_aggr_t *aggr, haddr_t *addr, hsize_t *size)
    {
        FUNC_ENTER_PACKAGE_NOERR

        
        assert(f);
        assert(aggr);
        assert(aggr->feature_flag == H5FD_FEAT_AGGREGATE_METADATA ||
               aggr->feature_flag == H5FD_FEAT_AGGREGATE_SMALLDATA);

        
        if (f->shared->feature_flags & aggr->feature_flag) {
            if (addr)
                *addr = aggr->addr;
            if (size)
                *size = aggr->size;
        } 

        FUNC_LEAVE_NOAPI(SUCCEED)
    } 

    
    static herr_t H5MF__aggr_reset(H5F_t * f, H5F_blk_aggr_t * aggr)
    {
        H5FD_mem_t alloc_type;          
        herr_t     ret_value = SUCCEED; 

        FUNC_ENTER_PACKAGE

        
        assert(f);
        assert(aggr);
        assert(aggr->feature_flag == H5FD_FEAT_AGGREGATE_METADATA ||
               aggr->feature_flag == H5FD_FEAT_AGGREGATE_SMALLDATA);

        
        alloc_type = (aggr->feature_flag == H5FD_FEAT_AGGREGATE_METADATA
                          ? H5FD_MEM_DEFAULT
                          : H5FD_MEM_DRAW); 

        
        if (f->shared->feature_flags & aggr->feature_flag) {
            haddr_t tmp_addr; 
            hsize_t tmp_size; 

            
            tmp_addr = aggr->addr;
            tmp_size = aggr->size;
#ifdef H5MF_AGGR_DEBUG
            Rfprintf(Rstderr, "%s: tmp_addr = %" PRIuHADDR ", tmp_size = %" PRIuHSIZE "\n", __func__, tmp_addr,
                    tmp_size);
#endif 

            
            aggr->tot_size = 0;
            aggr->addr     = 0;
            aggr->size     = 0;

            
            if (tmp_size > 0 && (H5F_INTENT(f) & H5F_ACC_RDWR))
                if (H5MF_xfree(f, alloc_type, tmp_addr, tmp_size) < 0)
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTFREE, FAIL, "can't release aggregator's free space");
        } 

done:
        FUNC_LEAVE_NOAPI(ret_value)
    } 

    
    herr_t H5MF_free_aggrs(H5F_t * f)
    {
        H5F_blk_aggr_t *first_aggr;              
        H5F_blk_aggr_t *second_aggr;             
        haddr_t         ma_addr   = HADDR_UNDEF; 
        hsize_t         ma_size   = 0;           
        haddr_t         sda_addr  = HADDR_UNDEF; 
        hsize_t         sda_size  = 0;           
        herr_t          ret_value = SUCCEED;     

        FUNC_ENTER_NOAPI(FAIL)

        
        assert(f);
        assert(f->shared);
        assert(f->shared->lf);

        
        if (H5MF__aggr_query(f, &(f->shared->meta_aggr), &ma_addr, &ma_size) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGET, FAIL, "can't query metadata aggregator stats");

        
        if (H5MF__aggr_query(f, &(f->shared->sdata_aggr), &sda_addr, &sda_size) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGET, FAIL, "can't query small data aggregator stats");

        
        
        if (H5_addr_defined(ma_addr) && H5_addr_defined(sda_addr)) {
            if (H5_addr_lt(ma_addr, sda_addr)) {
                first_aggr  = &(f->shared->sdata_aggr);
                second_aggr = &(f->shared->meta_aggr);
            } 
            else {
                first_aggr  = &(f->shared->meta_aggr);
                second_aggr = &(f->shared->sdata_aggr);
            } 
        }     
        else {
            first_aggr  = &(f->shared->meta_aggr);
            second_aggr = &(f->shared->sdata_aggr);
        } 

        
        if (H5MF__aggr_reset(f, first_aggr) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTFREE, FAIL, "can't reset metadata block");
        if (H5MF__aggr_reset(f, second_aggr) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTFREE, FAIL, "can't reset 'small data' block");
done:
        FUNC_LEAVE_NOAPI(ret_value)
    } 

    
    static htri_t H5MF__aggr_can_shrink_eoa(H5F_t * f, H5FD_mem_t type, H5F_blk_aggr_t * aggr)
    {
        haddr_t eoa       = HADDR_UNDEF; 
        htri_t  ret_value = false;       

        FUNC_ENTER_PACKAGE

        
        assert(f);
        assert(aggr);
        assert(aggr->feature_flag == H5FD_FEAT_AGGREGATE_METADATA ||
               aggr->feature_flag == H5FD_FEAT_AGGREGATE_SMALLDATA);

        
        if (HADDR_UNDEF == (eoa = H5F_get_eoa(f, type)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGET, FAIL, "Unable to get eoa");

        
        if (aggr->size > 0 && H5_addr_defined(aggr->addr))
            ret_value = H5_addr_eq(eoa, aggr->addr + aggr->size);

done:
        FUNC_LEAVE_NOAPI(ret_value)
    } 

    
    static herr_t H5MF__aggr_free(H5F_t * f, H5FD_mem_t type, H5F_blk_aggr_t * aggr)
    {
        herr_t ret_value = SUCCEED; 

        FUNC_ENTER_PACKAGE

        
        assert(f);
        assert(f->shared->lf);
        assert(aggr);
        assert(H5_addr_defined(aggr->addr));
        assert(aggr->size > 0);
        assert(H5F_INTENT(f) & H5F_ACC_RDWR);
        assert(aggr->feature_flag == H5FD_FEAT_AGGREGATE_METADATA ||
               aggr->feature_flag == H5FD_FEAT_AGGREGATE_SMALLDATA);
        assert(f->shared->feature_flags & aggr->feature_flag);

        
        if (H5F__free(f, type, aggr->addr, aggr->size) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTFREE, FAIL, "can't free aggregation block");

        
        aggr->tot_size = 0;
        aggr->addr     = HADDR_UNDEF;
        aggr->size     = 0;

done:
        FUNC_LEAVE_NOAPI(ret_value)
    } 

    
    htri_t H5MF__aggrs_try_shrink_eoa(H5F_t * f)
    {
        htri_t ma_status;        
        htri_t sda_status;       
        htri_t ret_value = FAIL; 

        FUNC_ENTER_PACKAGE

        
        assert(f);
        assert(f->shared);

        if ((ma_status = H5MF__aggr_can_shrink_eoa(f, H5FD_MEM_DEFAULT, &(f->shared->meta_aggr))) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGET, FAIL, "can't query metadata aggregator stats");
        if (ma_status > 0)
            if (H5MF__aggr_free(f, H5FD_MEM_DEFAULT, &(f->shared->meta_aggr)) < 0)
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTSHRINK, FAIL, "can't check for shrinking eoa");

        if ((sda_status = H5MF__aggr_can_shrink_eoa(f, H5FD_MEM_DRAW, &(f->shared->sdata_aggr))) < 0)
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGET, FAIL, "can't query small data aggregator stats");
        if (sda_status > 0)
            if (H5MF__aggr_free(f, H5FD_MEM_DRAW, &(f->shared->sdata_aggr)) < 0)
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTSHRINK, FAIL, "can't check for shrinking eoa");

        ret_value = (ma_status || sda_status);

done:
        FUNC_LEAVE_NOAPI(ret_value)
    } 
