% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/schema.r
\name{schema}
\alias{schema}
\alias{schema.list}
\alias{schema.data.frame}
\alias{enforce_schema}
\alias{enforce_schema.with_schema}
\alias{add_to_schema}
\alias{add_to_schema.with_schema}
\title{Ensure the truth of data-masked R expressions and cast/recycle named
elements.}
\usage{
schema(data, ...)

\method{schema}{list}(
  data,
  ...,
  .names = NULL,
  .size = NULL,
  .error_call = caller_env(),
  .darg = caller_arg(data)
)

\method{schema}{data.frame}(
  data,
  ...,
  .names = NULL,
  .size = NULL,
  .error_call = caller_env(),
  .darg = caller_arg(data)
)

enforce_schema(data, ...)

\method{enforce_schema}{with_schema}(data, ..., .error_call = caller_env(), .darg = caller_arg(data))

add_to_schema(data, ...)

\method{add_to_schema}{with_schema}(
  data,
  ...,
  .names = NULL,
  .size = NULL,
  .error_call = caller_env(),
  .darg = caller_arg(data)
)
}
\arguments{
\item{data}{a data.frame or list to use as the data mask.}

\item{...}{any number of R expressions or formulas to be evaluated using
\code{data} as a data mask. Formulas can use tidyselect syntax
on the lhs and either functions or formulas that evaluate to logical, or one
of the type/size functions: \link{cast}, \link{recycle} and \link{coerce}
on the rhs. The rhs of a formula can also be a \link{list} of multiple
functions/formulas/calls. If an expression is named, or if the list
element on the rhs of a formula is named, the name is passed to
\link[cli:format_inline]{format_inline} and is used in the error message.}

\item{.names}{character vector of names which must be present in the \code{data}
data.frame/list.}

\item{.size}{positive scalar integerish value for the size that the \code{data}
data.frame/list must be.}

\item{.error_call}{the call environment to use for error messages
(passed to \link[rlang:abort]{abort}).}

\item{.darg}{the argument name of \code{data} to use in error messages.}
}
\value{
\code{data} is returned with attached class \code{with_schema} and
attribute \code{schema} containing the schema call to be enforced later.
}
\description{
If any of the expressions in \code{...}, evaluated within the data mask
\code{'data'} (see \link[rlang:args_data_masking]{data masking}), are not all \code{TRUE}, \link[rlang:abort]{abort}
is called for the first which was not (\link{all}) \code{TRUE}. Alternatively,
\href{https://rlang.r-lib.org/reference/new_formula.html}{rlang formulas} can
be used to take advantage of \href{https://tidyselect.r-lib.org/index.html}{tidyselect}
features and pass multiple named elements in \code{data} to validation
formulas/functions, and/or attempt safe type casting and size recycling
using the \link{cast}, \link{recycle} and \link{coerce} functions.
The rhs of formulas can be given in a \link{list} to pass multiple
functions/formulas/calls. The \code{.names} and \code{.size} arguments can also be used
to check for given names and size of the data.frame/list itself. Type
casting, size checking, and recycling are undertaken using the
\href{https://vctrs.r-lib.org/}{vctrs} package and thus apply
\href{https://vctrs.r-lib.org/articles/type-size.html}{vctrs type and size rules}.
}
\details{
See \link{abort_if_not} for a non-data-masked validation tool and
\link{enforce} for a non-data-masked version of this function.
}
\examples{
# NB: Some of these examples are expected to produce an error. To
#     prevent them from terminating a run with example() they are
#     piped into a call to try().

li <- list(x = 1L, y = "hi", z = \(x) x > 1)
li <- li |>
  schema(x == 1, is.character(y), is.function(z)) # all TRUE

# The schema call is attached to the returned object and
# can be re-evaluated using enforce_schema():
li <- enforce_schema(li) # no error
li2 <- li
li2$x <- 2L
enforce_schema(li2) |> try()

# Calling `schema()` again overwrites any existing schema.
# Alternatively use `add_to_schema()` to add arguments to
# an existing schema (.size overwrites, other args append):
li <- li |>
  add_to_schema(is.numeric(x), .names = c("x", "y"), .size = 3)

# A custom error message can be given for each expression by
# naming it:
schema(li,
  "{.var y} must be {.cls numeric}, check input" = is.numeric(y)
) |> try()

# Formulas can be used to take advantage of tidyselect features
# on the lhs, with functions/additional formulas required on
# the rhs:
schema(li,
  "multiple columns: {.pkg tidyselect}" = c(x, y) ~ is.integer
) |> try()

# Formulas can also be used with `cast()`, `recycle()`, and
# `coerce()` on the rhs to safely cast or recycle named
# elements:
class(schema(li, x ~ cast(double()))$x) # x is now numeric
length(schema(li, x ~ recycle(5))$x) # x is now length 5
schema(
  li,
  y ~ coerce(type = factor(), size = 5)
)$y # y is now factor and length 5

# Multiple calls can be used with formulas by wrapping them
# in `list()`, with the names of list elements being
# preferentially chosen for error messaging and the error
# message also showing which formula/function/call caused the
# error:
schema(
  li,
  "generic message" = c(x, y, z) ~ list(
    Negate(is.null),
    "{.var specific} message" = Negate(is.function)
  )
) |> try()

# Changed elements are available immediately:
df <- data.frame(x = 1L, y = 1L)
lapply(schema(df, x ~ cast(double()), y ~ cast(x)), class)
# both now numeric

# `.names` and `.size` arguments can be used to check that given
# names are present and that the data has the desired size:
schema(li, .names = c("a", "x", "y", "b")) |> try()
schema(li, .size = 5) |> try()

# The `.error_call` argument can be used to specify where the
# error occurs, by default this is the caller environment:
myfunc <- function(x, ...) schema(x, ...)
myfunc(li, x > 4) |> try()

# rlang pronouns and injection can be used, but care must be
# taken when using `.env` and `enforce_schema()` as the
# caller environment may have changed:
msg <- "{.var injection} msg"
cols <- quote(c(x, y))
schema(li, !!msg := !!cols ~ is.integer) |> try()

x <- 1L
li <- schema(li, x == .env$x) # no error

x <- 2
enforce_schema(li) |>
  try() # error as the environmental variable has changed
}
