#' Gaussian Quadrature Points and Weights
#'
#' @description
#' Generate Gaussian quadrature points and corresponding normalized weights
#' based on the standard normal density over a symmetric interval.
#'
#' @param n Integer. Number of quadrature points (must be >= 2).
#' @param mm Numeric. Positive value giving the maximum absolute value of the
#'   quadrature nodes (range will be from -mm to +mm).
#'
#' @return A list with:
#' \describe{
#'   \item{nodes}{Quadrature nodes from -mm to +mm.}
#'   \item{weights}{Normalized weights proportional to the standard normal
#'   density at each node.}
#' }
#'
#' @export
#'
#' @examples
#' normal_quadra(41, 5)
normal_quadra <- function(n, mm) {

  # --- input checks -----------------------------------------------------------
  if (missing(n) || missing(mm)) {
    stop("Both `n` and `mm` must be provided.")
  }

  if (!is.numeric(n) || length(n) != 1L || n < 2) {
    stop("`n` must be a single numeric value >= 2.")
  }

  if (!is.numeric(mm) || length(mm) != 1L || mm <= 0) {
    stop("`mm` must be a single positive numeric value.")
  }

  n  <- as.integer(n)

  # --- quadrature nodes -------------------------------------------------------
  nodes <- seq(-mm, mm, length.out = n)

  # --- unnormalized weights (standard normal density) -------------------------
  w_unnormalized <- stats::dnorm(nodes)

  # --- normalize weights ------------------------------------------------------
  w_normalized <- w_unnormalized / sum(w_unnormalized)

  # --- return list ------------------------------------------------------------
  return(list(
    nodes   = nodes,
    weights = w_normalized
  ))
}
