

#' Ternary Diagram
#'
#' Generates a ternary diagram from compositional data, with options to center, scale,
#' and color the points by group. Optionally overlays principal components.
#'
#' @param X A numeric matrix or data frame of compositional data with exactly three columns.
#' @param group A factor or character vector indicating groups for color coding (optional).
#' @param center Logical. Should the data be centered before plotting? Default is FALSE.
#' @param scale Logical. Should the data be scaled to unit variance? Default is FALSE.
#' @param show_pc Logical. If TRUE, principal components are overlaid. Default is FALSE.
#'
#' @return A \code{ggplot} plot object (inherits from \code{ggplot}).
#'
#' X = milk_cows[,5:7]
#' group = milk_cows$group
#' ternary_diagram(X, group = group)
#'
#' @export
ternary_diagram = function(X, group = NULL,
                               center = FALSE, scale = FALSE,
                               show_pc = FALSE){


  composition = function(x, ...) suppressWarnings(coda.base::composition(x, ...))
  geom_point = function(x, ...) suppressWarnings(ggplot2::geom_point(x, ...))
  geom_path = function(x, ...) suppressWarnings(ggplot2::geom_path(x, ...))

  if(!is.matrix(X)){
    X = as.matrix(X)
  }

  GROUPED = !is.null(group)

  xyz_labs = colnames(X)
  if(ncol(X) != 3){
    stop("three columns needed")
  }
  if(center | scale){
    X = composition(scale(coordinates(X), center = center, scale = scale))
  }
  dplot = as.data.frame(matrix(X, ncol = ncol(X)))
  names(dplot) = c('c1', 'c2','c3')

  to_ternary = function(dplot){
    dplot = transform(dplot,
                      .A = c1 / (c1+c2+c3),
                      .B = c2 / (c1+c2+c3),
                      .C = c3 / (c1+c2+c3))

    transform(dplot,
              .x = .C + 0.5 * .A,
              .y = sqrt(3)/2 * .A)
  }
  dplot = to_ternary(dplot)

  if(GROUPED) dplot$group = group

  geom_ternary_outline <- function(...) {
    geom_path(data = data.frame(.x = c(0, 1, 0.5, 0),
                                .y = c(0, 0, sqrt(3)/2, 0)),
              aes(.x, .y), inherit.aes = FALSE, ...)
  }
  ternary_isolines_df <- function(var = c("A","B","C"), ticks = seq(0.1, 0.9, 0.1)) {
    var <- match.arg(var)
    out <- lapply(ticks, function(tk) {
      n <- 100
      if (var == "A") {
        Bv <- seq(0, 1 - tk, length.out = n); Cv <- (1 - tk) - Bv; Av <- rep(tk, n)
      } else if (var == "B") {
        Av <- seq(0, 1 - tk, length.out = n); Cv <- (1 - tk) - Av; Bv <- rep(tk, n)
      } else {
        Av <- seq(0, 1 - tk, length.out = n); Bv <- (1 - tk) - Av; Cv <- rep(tk, n)
      }
      df <- data.frame(c1 = Av, c2 = Bv, c3 = Cv, var = var, tick = tk)
      to_ternary(df)
    })
    d = do.call(rbind, out)
    transform(d, .xend = c(.x[-1], NA), .yend = c(.y[-1],NA))
  }
  geom_ternary_grid <- function(ticks = seq(0.1, 0.9, 0.1), alpha = 0.3, ...) {
    gA = ternary_isolines_df("A", ticks)
    gB <- ternary_isolines_df("B", ticks)
    gC <- ternary_isolines_df("C", ticks)
    gA = gA[seq(1,nrow(gA),2),]
    gB = gB[seq(1,nrow(gB),2),]
    gC = gC[seq(1,nrow(gC),2),]
    list(
      geom_segment(data = gA, aes(x = .x, y = .y, xend = .xend, yend = .yend),
                   inherit.aes = FALSE, alpha = alpha, ...),
      geom_segment(data = gB, aes(x = .x, y = .y, xend = .xend, yend = .yend),
                   inherit.aes = FALSE, alpha = alpha, ...),
      geom_segment(data = gC, aes(x = .x, y = .y, xend = .xend, yend = .yend),
                   inherit.aes = FALSE, alpha = alpha, ...)
    )
  }
  geom_ternary_corner_labels <- function(labels, ...) {
    labs <- data.frame(
      lab = unname(labels),
      .x  = c(0.5, 0, 1),
      .y  = c(sqrt(3)/2, 0, 0)
    )
    geom_text(data = labs, aes(.x, .y, label = lab),
              inherit.aes = FALSE, vjust = c(-0.4, 1.2, 1.2), ...)
  }
  theme_ternary <- function() {
    theme_void() + theme(plot.margin = margin(10,10,10,10))
  }
  p = ggplot2::ggplot(data = dplot) +
    geom_ternary_grid() +
    geom_ternary_outline() +
    geom_ternary_corner_labels(xyz_labs) +
    coord_equal() +
    theme_ternary()


  if(GROUPED){
    p = p + geom_point(aes(x=.x, y=.y, col=group)) +
      labs(color = '')
  }else{
    p = p + geom_point(aes(x=.x, y=.y))
  }

  if(show_pc){
    H = coordinates(X)
    if(GROUPED){

      for(group_k in unique(group)){

        H_k = H[group == group_k,]
        eig = eigen(cov(H_k))

        n_lims = 3/min(abs(ilr_basis(3) %*% (eig$values[2] * eig$vectors[,2])))

        l_X_pc = lapply(1:ncol(H), function(i){
          h_pc = t(colMeans(H_k) + sapply(seq(-n_lims, n_lims, length=500),  `*`, eig$values[i] * eig$vectors[,i]))
          composition(h_pc)
        })
        dplot1 = as.data.frame(l_X_pc[[1]])
        names(dplot1) = c('c1','c2','c3')
        dplot1 = to_ternary(dplot1)
        dplot1$group = group_k
        dplot2 = as.data.frame(l_X_pc[[2]])
        names(dplot2) = c('c1','c2','c3')
        dplot2 = to_ternary(dplot2)
        dplot2$group = group_k
        p = p +
          geom_path(data = dplot1, aes(x = .x, y = .y, col = group, linetype = 'Prin.Comp.1')) +
          geom_path(data = dplot2, aes(x = .x, y = .y, col = group, linetype = 'Prin.Comp.2'))

      }
      p = p  +
        scale_linetype_manual(values = c('dashed', 'dotted')) +
        labs(col = '', linetype = '') +
        guides(color  = guide_legend(order = 1),
               linetype = guide_legend(order = 2))
    }else{
      eig = eigen(cov(H))
      n_lims = 3/min(abs(ilr_basis(3) %*% (eig$values[2] * eig$vectors[,2])))
      l_X_pc = lapply(1:ncol(H), function(i){
        h_pc = t(colMeans(H) + sapply(seq(-n_lims, n_lims, length=500),  `*`, eig$values[i] * eig$vectors[,i]))
        composition(h_pc)
      })
      dplot1 = as.data.frame(l_X_pc[[1]])
      names(dplot1) = c('c1','c2','c3')
      dplot2 = as.data.frame(l_X_pc[[2]])
      names(dplot2) = c('c1','c2','c3')
      p = p +
        geom_path(data = to_ternary(dplot1), aes(x = .x, y = .y, linetype = 'Prin.Comp.1'),
                  col='blue') +
        geom_path(data = to_ternary(dplot2), aes(x = .x, y = .y, linetype = 'Prin.Comp.2'),
                  col = 'blue') +
        scale_linetype_manual(values = c('dashed', 'dotted')) +
        labs(linetype = '')

    }
  }
  p

}
