## ------------------------------------------------------------------------
# 
# "para.boot(X,func,rdist,param,B,...)" 
# 
# Parametric Bootstrap
#
# ------------------------------------------------------------------------
##

#' @aliases para.boot
#' @title Parametric Bootstrap for i.i.d. Data
#' @description 
#' This function performs a parametric bootstrap, a technique that resamples 
#' data based on an assumed distribution with estimated parameter,
#' rather than resampling the original data directly.
#' @param X A numeric vector representing the data.
#' @param func A function taking \code{X} as an argument,
#' representing the statistic of interest to be bootstrapped.
#'  It should returns a vector of size `p >= 1`.
#' @param rdist  A parametric distribution generator that produces bootstrap data based on the
#' data size and \code{param}. It should be a function with two arguments
#' \code{n} - the size of the bootstrap sample and \code{par} - a vector 
#' of the parameters.
#' @param param Numeric vector. Values of parameters used to generate the bootstrap data. 
#' These can be either the true parameter values for Monte Carlo approximation of the
#'  true distribution, or the estimated parameters, typically obtained by the 
#'  maximum likelihood method.
#' @param B A positive integer; the number of bootstrap replications. By default it is 999.
#' @param ... Optional additional arguments for the \code{func} function.
#' @details 
#' \code{para.boot} is a flexible function for bootstrapping a specified 
#' function \code{func} using a parametric distribution \code{rdist} generated 
#' with estimated or true parameters \code{param}. The function returns a \code{boodd}
#'  object containing the values of the function over all bootstrap samples and
#'   the statistic computed over the original sample.
#' 
#' 
#' @return A \code{boodd} object containing:
#'   - \code{s}: Matrix of bootstrap replicates of \code{func}, with dimensions \code{B x p}.
#'   - \code{Tn}: Vector of \code{func} evaluated on the original data \code{X}.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted..
#' 
#'   Efron, B., Tibshirani, R. (1993). 
#'   \emph{An Introduction to the Bootstrap}, Chapman and Hall.
#' @seealso \code{\link{bootglm}}, \code{\link{boots}}.
#' @keywords "Parametric bootstrap" "Monte Carlo simulation"
#' @export
#' @examples 
#' rn<-function(n,par) {rnorm(n,mean=par[1],sd=par[2])}
#' set.seed(5)
#' # Parametric bootstrap of the mean in a gaussian family
#' X=rnorm(n=100,mean=2,sd=1)
#' # simulate distribution with true parameter values (and a Monte-Carlo size 9999
#' true1<-para.boot(X,mean,rn,param=c(2,1),B=9999)
#' pb1<-para.boot(X,mean,rn,param=c(mean(X),sd(X)))
#' plot(pb1)
#' lines(density(true1$s),col="red")
#' confint(true1,method="bperc")
#' confint(pb1, method="all")
#' 


para.boot<-function(X,func,rdist,param,B=999,...) {
  # Compute original function of X, of size p
  Tn=func(X)
  p=length(Tn)
  # define matrix of bootstrap results Bxp
  resb=matrix(0,B,p)
  n=length(X)
  for (i in (1:B)) {
    # generate distribution with estimated parameters
    Xb=rdist(n,par=param)
    resb[i,]=func(Xb)
  }
  resboot <- list(s = resb, Tn =Tn)
  class(resboot) <- "boodd"
  attr(resboot, "kind") <- "parametric"
  attr(resboot, "func") <- func
  return(resboot)
}