% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/helpers.R
\name{mm}
\alias{mm}
\title{Define a multiple-membership structure}
\usage{
mm(id, vars = NULL, fn = NULL, RE = NULL, ar = FALSE)
}
\arguments{
\item{id}{An \code{\link{id}} object specifying the member-level and group-level
identifiers: \code{id(mmid, mainid)} where \code{mmid} identifies members
and \code{mainid} identifies groups.}

\item{vars}{A \code{\link{vars}} object specifying member-level covariates to
aggregate, or \code{NULL} for random effects only. Supports interactions
(\code{*}, \code{:}) and transformations (\code{I()}). Variables are weighted
according to the function specified in \code{fn}.}

\item{fn}{A \code{\link{fn}} object specifying the weight function (default:
\code{fn(w ~ 1/n, c = TRUE)} for equal weights). Note: Weight functions do
NOT support interactions or \code{I()} - pre-create any needed transformed
variables in your data. See \code{\link{fn}} for details.}

\item{RE}{Logical; if \code{TRUE}, include random effects for member-level units.
Automatically set to \code{TRUE} if \code{vars = NULL} (random effects only).}

\item{ar}{Logical; if \code{TRUE}, random effects evolve autoregressively across
participations. Requires members to have sequential participation indicators
in the data. Default: \code{FALSE}.}
}
\value{
A \code{bml_mm} object containing the multiple-membership specification.
}
\description{
Specifies a multiple-membership level in the model where group-level units
(e.g., governments) are composed of multiple member-level units (e.g., political
parties). Unlike pure hierarchical nesting, members can belong to multiple groups,
and their contributions are aggregated using a user-specified weight function.
}
\details{
\strong{Multiple-Membership Models:}

In standard hierarchical models, each observation belongs to exactly one group.
Multiple-membership models relax this assumption, allowing groups to be composed
of multiple members, with flexible weighting of member contributions.

\strong{Model Structure:}

The contribution from mm block \eqn{k} to group \eqn{j} is:

\deqn{\mathrm{mm}_{kj} = \sum_{i \in group_j} w_{ki} (x_{ki}' \beta_k + \alpha_{ki})}

where:
\itemize{
  \item \eqn{w_{ki}}: Weight for member \eqn{i} in group \eqn{j} (from \code{fn})
  \item \eqn{x_{ki}}: Member-level covariates (from \code{vars})
  \item \eqn{\beta_k}: Regression coefficients (estimated)
  \item \eqn{\alpha_{ki}}: Member-level random effect (if \code{RE = TRUE})
}

\strong{Multiple mm() Blocks:}

You can specify multiple \code{mm()} blocks with different weight functions,
variables, or random effect specifications. This allows modeling different
aggregation mechanisms simultaneously.
}
\examples{
\donttest{
# Equal weights with variables
mm(
  id = id(pid, gid),
  vars = vars(rile + ipd),
  fn = fn(w ~ 1/n, c = TRUE),
  RE = FALSE
)

# Random effects only (no variables)
mm(
  id = id(pid, gid),
  vars = NULL,
  fn = fn(w ~ 1/n, c = TRUE),
  RE = TRUE  # Automatically TRUE when vars = NULL
)

# Flexible weights with parameter
mm(
  id = id(pid, gid),
  vars = vars(org_structure),
  fn = fn(w ~ ilogit(b0 + b1 * pseat), c = TRUE),
  RE = TRUE
)

# Autoregressive random effects
mm(
  id = id(pid, gid),
  vars = NULL,
  fn = fn(w ~ 1/n, c = TRUE),
  RE = TRUE,
  ar = TRUE  # Random effects evolve over participations
)

# Interactions and transformations in vars
mm(
  id = id(pid, gid),
  vars = vars(rile * ipd),  # Main effects plus interaction
  fn = fn(w ~ 1/n, c = TRUE),
  RE = FALSE
)

mm(
  id = id(pid, gid),
  vars = vars(rile + I(rile^2)),  # Quadratic term
  fn = fn(w ~ 1/n, c = TRUE),
  RE = FALSE
)
}

}
\references{
Rosche, B. (2026). A Multilevel Model for Theorizing and Estimating the
Micro-Macro Link. \emph{Political Analysis}.

Browne, W. J., Goldstein, H., & Rasbash, J. (2001). Multiple membership
multiple classification (MMMC) models. \emph{Statistical Modelling}, 1(2), 103-124.

Fielding, A., & Goldstein, H. (2006). \emph{Cross-classified and multiple
membership structures in multilevel models: An introduction and review}.
Research Report RR791, Department for Education and Skills.
}
\seealso{
\code{\link{bml}}, \code{\link{id}}, \code{\link{vars}}, \code{\link{fn}},
  \code{\link{hm}}
}
