% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functional_traits.R
\name{functional_traits}
\alias{functional_traits}
\title{Functional PCA traits for circadian rhythm data}
\usage{
functional_traits(
  smooth_out,
  time,
  meta = NULL,
  group_col = "Genotype",
  groups = NULL,
  derivative = c(0, 1, 2),
  n_pc = NULL,
  segments = NULL,
  centerfns = FALSE
)
}
\arguments{
\item{smooth_out}{An object of class \code{fdSmooth}, typically the output
of a smoothing function applied to circadian time-series data.}

\item{time}{A numeric vector giving the time points corresponding to the
functional observations.}

\item{meta}{An optional data frame containing sample metadata. Must include
a column named \code{Sample_id} and a grouping column specified by
\code{group_col}.}

\item{group_col}{Character string giving the name of the column in
\code{meta} that defines the grouping variable (e.g. genotype).
Default is \code{"Genotype"}.}

\item{groups}{Optional factor or vector defining group membership for each
sample. Used only if \code{meta} is not supplied. Length must match the
number of curves.}

\item{derivative}{Integer vector specifying which derivatives of the
functional data to analyse (e.g. 0 = original curve, 1 = first derivative).
Default is \code{c(0, 1, 2)}.}

\item{n_pc}{Integer giving the number of principal components to retain.
If \code{NULL}, the maximum number of PCs allowed by the data is used.}

\item{segments}{Optional named list defining time segments over which
traits should be summarised. Each element must be a numeric vector of
length two giving the start and end times (e.g.
\code{list(pre = c(0, 104), post = c(104, 240))}).
If \code{NULL}, the full time range is used.}

\item{centerfns}{Logical indicating whether to centre functional observations
before FPCA. Passed to \code{\link[fda]{pca.fd}}.}
}
\value{
A list with the following components:
\describe{
  \item{traits}{A tidy data frame of group-level FPCA-derived traits,
    including marginal PC summaries and FPCA-space dispersion measures.}
  \item{scores}{A data frame of FPCA scores for each sample, annotated with
    group, segment, and derivative information.}
  \item{fpca}{A named list of \code{\link[fda]{pca.fd}} objects, one per
    derivative.}
  \item{meta}{The metadata data frame aligned to the FPCA input order. A two-column data frame containing \code{Sample_id} and \code{Genotype}.}
}
}
\description{
Performs functional principal component analysis (FPCA) on smoothed
circadian time-series data and extracts group-level traits from the
resulting FPCA score space.
}
\details{
The analysis can optionally be stratified by time segments
(e.g. pre/post environmental shift) and by derivatives of the curves.
When segments are supplied, functional observations from all segments
are first re-evaluated onto a shared time grid and analysed jointly,
ensuring that FPCA scores are directly comparable across segments.

For each derivative, FPCA is performed once across all samples and
segments simultaneously. Segment- and group-specific traits are then
computed post hoc from the shared FPCA score space.

Extracted traits include:
\itemize{
  \item Mean and standard deviation of FPCA scores (per principal component)
  \item Mean and standard deviation of distance to the FPCA-space centroid
  \item Convex hull area in PC1--PC2 space (measure of within-group diversity)
}


FPCA is performed in a shared functional space for each derivative.
When segments are provided, all segment-specific curves are interpolated
onto a common time grid prior to FPCA, ensuring statistical comparability
of FPCA scores across segments. Group-level traits are computed after FPCA
and do not influence the decomposition itself.

Convex hull area is computed in PC1-PC2 space only and requires at least
three samples per group and segment.
}
\examples{

time_vec <- matrix(seq(0, 48, length.out = 100))

Genotype_A <- matrix(
sin(2 * pi * (1:100) / 24) +
matrix(rnorm(100 * 5, 0, 0.1), nrow = 100, ncol = 5),
nrow = 100,
ncol = 5,
dimnames = list(NULL, paste0("A", 1:5)))

Genotype_B <- matrix(
0.3 * sin(2 * pi * (1:100) / 24) +
matrix(rnorm(100 * 5, 0, 0.5), nrow = 100, ncol = 5),
nrow = 100,
ncol = 5,
dimnames = list(NULL, paste0("B", 1:5)))

raw_data <- cbind(Genotype_A, Genotype_B)
smooth_data <- smooth_fun(raw_data, time_vec)

meta_df <- data.frame(
Sample_id = colnames(raw_data),
Genotype = rep(c("A", "B"), each = 5))

fpca_res <- functional_traits(
smooth_out = smooth_data$smooth,
time = time_vec,
meta = meta_df,
n_pc = 2)

head(fpca_res$traits)

cols <- c(A = "red", B = "blue")
with(subset(fpca_res$scores, derivative == 0),
plot(PC1, PC2, pch = 19, xlab = "PC1", ylab = "PC2", col = cols[group]))
legend("topright", legend = names(cols), col = cols, pch = 19)

}
\seealso{
\code{\link[fda]{pca.fd}}, \code{\link[TimeTraits]{smooth_fun}},
  \code{\link[fda]{eval.fd}}
}
