#' Generates the unweighted and survey weighted principal component based normalized composite index
#'
#' @description The PcIndex function generates the survey weighted principal component based normalized composite index by assigning survey weights to each of the sample observations. Besides this, the function also generates the principal component based normalized composite index without incorporating survey weights. Further, the function provides composite index values for both the methods in descending order along with the corresponding sample id as well as rank of the sample observations. Here, the sample observations having highest and lowest composite index values are given first and last rank respectively.
#'Data Requirements: The input data must be in CSV format and include the following columns:
#'• Sample_id: A unique identifier for each sample observation.
#'• SW: The survey weight associated with each sample observation.
#'• Variables: Columns representing the variables used to construct the composite index, with each column corresponding to a distinct variable.
#'
#' @param data Input multivariate survey data with survey weights.
#'
#' @return A list with the following components:
#' \item{PC based composite index}{Ordered principal component based normalized composite index values in descending order along with the corresponding Sample_id as well as its rank of the sample observations.}
#' \item{Survey weighted PC based composite index}{Ordered survey weighted principal component based normalized composite index values in descending order along with the corresponding Sample_id as well as its rank of the sample observations.}
#'
#' @details This function returns the principal component based normalized composite index without incorporating survey weights as well as survey weighted principal component based normalized composite index.
#'
#' @references
#' Skinner, C. J., Holmes, D. J. and Smith, T. M. F. (1986). The Effect of Sample Design on Principal Component Analysis. \emph{Journal of the American Statistical Association} 81(395): 789-798. <DOI:10.1080/01621459.1986.10478336>
#'
#' Singh, D., Basak, P., Kumar, R. and Ahmad, T. (2023). On the Methodological Framework of Composite index under complex Surveys and its application in development of food consumption index for India. \emph{Frontiers in Applied Mathematics and Statistics} 9:1274530. <DOI:10.3389/fams.2023.1274530>
#'
#' Singh, D., Basak, P., Ahmad, T., Kumar, R. and Rai, A. (2024). Development of Survey Weighted Composite Indices under Complex Surveys.  \emph{Statistics and Applications} 22(1): 73-82.
#'
#' @examples
#' #  Sample data
#' SERIAL_NO<- c(1:12)
#' SW<- c(20,10,40,50,13,25,23,26,14,10,07,05)
#' x <- c(840,204,944,1009,745,811,883,593,254,215,172,169)
#' y <- c(150,104,293,331,113,52,164,201,124,108,95,90)
#' z  <- c(350,214,693,731,413,352,364,301,214,208,186,130)
#' # View the sample data
#' data<-cbind(SERIAL_NO, SW, x, y, z)
#' # Calculate composite indices using the sample data
#' PcIndex(data)
#' @export
#' @importFrom stats cov
#' @importFrom stats cov.wt
PcIndex=function(data)
{
  var=as.matrix(data[,-c(1:2)])
  n=dim(data)[1]
  sw=data[,2]
  swt=sw/sum(sw)
  #####PC based composite index
  u1=cov(var)
  eigenvalues0<- eigen(u1)$values
  loadingsLONG0<- eigen(u1)$vectors
  scoresLONG0<- var %*%  loadingsLONG0
  score0<- (scoresLONG0 %*% eigenvalues0)/sum(eigenvalues0) #PC based composite index
  a0=max(score0)
  b0=min(score0)
  uwpcindex<-(score0-b0)/(a0-b0)# PC based normalized composite index

  #####Survey weighted PC based composite index
  B1=cov.wt(var,wt = swt)
  B1=B1$cov
  eigenvalues1<- eigen(B1)$values
  loadingsLONG1<- eigen(B1)$vectors
  scoresLONG1<- var %*%  loadingsLONG1
  score1<- (scoresLONG1 %*% eigenvalues1)/sum(eigenvalues1)#Survey weighted PC based composite index
  a1=max(score1)
  b1=min(score1)
  swpcindex<-(score1-b1)/(a1-b1)#Survey weighted PC based normalized composite index
  uwpcindex=cbind(order(-uwpcindex),uwpcindex[order(-uwpcindex)], c(1:n))
  colnames(uwpcindex)=c("Sample_id", "Composite index","Rank")
  swpcindex=cbind(order(-swpcindex),swpcindex[order(-swpcindex)], c(1:n))
  colnames(swpcindex)=c("Sample_id", "Composite index","Rank")
  # Output results as a list
  output <- list("PC based composite index"=uwpcindex, "Survey weighted PC based composite index"=swpcindex)
  return(output)
}
