#' Scalable MCMC for Dynamic GLMs
#'
#' Implements Algorithm 1 of the reference paper with three mutation-rate
#' strategies (ScaI/II/III) and three move types (mutation/crossover/exchange).
#'
#' @param y Numeric response vector (length n).
#' @param X n x p design matrix.
#' @param family Character: "poisson", "pareto", or "gamma".
#' @param method Character: "ScaI", "ScaII", or "ScaIII".
#' @param N_chain Integer >= 2, number of parallel chains.
#' @param iter Integer, total MCMC iterations per chain.
#' @param burn Integer, burn-in length (default = iter/2).
#' @param thin Integer, thinning interval (default = 1).
#' @param T_max Numeric > 1, hottest temperature (default = 10).
#' @param beta_init Optional matrix (N_chain x p) of initial coefficients.
#'                 If NULL, random starts are generated.
#' @param verbose Logical, print progress bar (default = TRUE).
#'
#' @return List with components:
#' \item{beta_chain}{3-D array (iter/thin x p x N_chain) of posterior samples.}
#' \item{family}{Character, distribution family used.}
#' \item{method}{Character, mutation-rate strategy used.}
#'
#' @examples
#' set.seed(1)
#' X <- matrix(rnorm(200 * 3), 200, 3)
#' beta <- c(0.5, -0.2, 0.1)
#' y <- rgamma(200, shape = 2, rate = exp(X %*% beta))
#' fit <- sca_mcmc(y, X, family = "gamma", method = "ScaII",
#'                 N_chain = 6, iter = 1000, burn = 500)
#'
#' @export
sca_mcmc <- function(y, X,
                     family = c("poisson", "pareto", "gamma"),
                     method = c("ScaI", "ScaII", "ScaIII"),
                     N_chain = 6L,
                     iter    = 10000L,
                     burn    = NULL,
                     thin    = 1L,
                     T_max   = 10,
                     beta_init = NULL,
                     verbose = TRUE) {
  
  # Input validation
  family <- match.arg(family)
  method <- match.arg(method)
  n <- length(y)
  p <- ncol(X)
  
  if (is.null(burn)) burn <- iter %/% 2
  if (iter <= burn) stop("iter must be greater than burn")
  if (thin < 1) stop("thin must be >= 1")
  if (N_chain < 2) stop("N_chain must be >= 2")
  
  # Geometric temperature ladder
  temp <- geoTemp(N_chain, T1 = 1, TN = T_max)
  
  # Initialize coefficients
  if (is.null(beta_init)) {
    beta_init <- matrix(stats::rnorm(N_chain * p), N_chain, p)
  } else {
    if (!is.matrix(beta_init) || dim(beta_init)[1] != N_chain || dim(beta_init)[2] != p) {
      stop("beta_init must be a matrix with dimensions N_chain x p")
    }
  }
  beta <- beta_init
  
  # Mutation-rate vector
  Qvec <- compute_mutation_rate(method,
                                beta_star = rep(1, p), # pseudo target
                                beta_init = rep(0, p),
                                L = p,
                                N_chain = N_chain)$Q
  
  # Storage arrays
  n_save <- (iter - burn) %/% thin
  beta_chain <- array(NA, dim = c(n_save, p, N_chain))
  idx <- 1L
  
  # Main MCMC loop
  if (verbose) {
    pb <- utils::txtProgressBar(max = iter, style = 3)
  }
  
  for (t in 1:iter) {
    # Mutation or Crossover move
    for (k in 1:N_chain) {
      if (stats::runif(1) < sum(Qvec)) {
        # Mutation: single-chain Metropolis
        prop <- beta[k, ] + stats::rnorm(p, 0, 0.05)
        ll_old <- sum(dglm_likelihood(y, X, beta[k, ], family))
        ll_new <- sum(dglm_likelihood(y, X, prop, family))
        if (log(stats::runif(1)) < (ll_new - ll_old) / temp[k]) {
          beta[k, ] <- prop
        }
      } else {
        # Crossover: swap segments between two chains
        k2 <- sample(setdiff(1:N_chain, k), 1)
        cpoint <- sample(1:p, 1)
        prop1 <- beta[k, ]
        prop2 <- beta[k2, ]
        prop1[cpoint:p] <- beta[k2, cpoint:p]
        prop2[cpoint:p] <- beta[k, cpoint:p]
        ll1_old <- sum(dglm_likelihood(y, X, beta[k, ], family))
        ll2_old <- sum(dglm_likelihood(y, X, beta[k2, ], family))
        ll1_new <- sum(dglm_likelihood(y, X, prop1, family))
        ll2_new <- sum(dglm_likelihood(y, X, prop2, family))
        if (log(stats::runif(1)) < ((ll1_new + ll2_new) - (ll1_old + ll2_old)) / temp[k]) {
          beta[k, ]  <- prop1
          beta[k2, ] <- prop2
        }
      }
    }
    
    # Exchange move (parallel tempering)
    for (k in 1:(N_chain - 1)) {
      ll_k  <- sum(dglm_likelihood(y, X, beta[k, ], family))
      ll_k1 <- sum(dglm_likelihood(y, X, beta[k + 1, ], family))
      log_alpha <- (ll_k - ll_k1) * (1 / temp[k] - 1 / temp[k + 1])
      if (log(stats::runif(1)) < log_alpha) {
        tmp           <- beta[k, ]
        beta[k, ]     <- beta[k + 1, ]
        beta[k + 1, ] <- tmp
      }
    }
    
    # Post-burn-in storage
    if (t > burn && (t - burn) %% thin == 0) {
      beta_chain[idx, , ] <- t(beta)
      idx <- idx + 1L
    }
    
    if (verbose) {
      utils::setTxtProgressBar(pb, t)
    }
  }
  
  if (verbose) {
    close(pb)
  }
  
  # Return results
  structure(list(
    beta_chain = beta_chain,
    family = family,
    method = method
  ), class = "SDGLM")
}
