% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/quantify.outliers.R
\name{quantify.outliers}
\alias{quantify.outliers}
\title{Compute quantities for outlier detection}
\usage{
quantify.outliers(
  x,
  method = "mean",
  trim = 0,
  nstart = 1,
  exclude.zero = FALSE
)
}
\arguments{
\item{x}{A numeric vector.}

\item{method}{A string indicating the quantities to be computed.  Possible values are
\itemize{
\item \code{'mean'} : z-scores based on mean and standard deviation or trimmed mean and trimmed standard deviation if \code{trim > 0},
\item \code{'median'} : z-scores based on median and median absolute deviation, or
\item \code{'kmeans'} : cluster assignment from k-means with two clusters.
The default is z-scores based on the mean and standard deviation.
}}

\item{trim}{A number, the fraction of observations to be trimmed from each end of \code{x}.  Default is no trimming.}

\item{nstart}{A number, for k-means clustering, the number of random initial centers for the clusters.  Default is \code{1}.  See \code{\link[stats:kmeans]{stats::kmeans()}} for further information.}

\item{exclude.zero}{A logical, whether zeros should be excluded (\code{TRUE}) or not excluded (\code{FALSE}, the default) from computations.  For \code{method = 'mean'} and \code{method = 'median'}, this means zeros will not be included in computing the summary statistics; for \code{method = 'kmeans'}, this means zeros will be placed in their own cluster, coded \code{0}.}
}
\value{
A numeric vector the same size as \code{x} whose values are the requested quantities computed on the corresponding elements of \code{x}.
}
\description{
Compute quantities for use in the detection of outliers.  Specifically, compute z-scores based on the mean / standard deviation, the trimmed mean / trimmed standard deviation, or the median / median absolute deviation, or the cluster assignment from k-means with two clusters.
}
\examples{
# Generate fake data.
set.seed(1234);
x <- rgamma(
    n = 20,
    shape = 2,
    scale = 2
    );
# Add missing values and zeros for demonstration.  Missing values are
# ignored, and zeros can be ignored with `exclude.zeros = TRUE`.
x[1:5] <- NA;
x[6:10] <- 0;

# Compute z-scores based on mean and standard deviation.
quantify.outliers(
    x = x,
    method = 'mean',
    trim = 0
    );
# Exclude zeros from the calculation of the mean and standard
# deviation.
quantify.outliers(
    x = x,
    method = 'mean',
    trim = 0,
    exclude.zero = TRUE
    );

# Compute z-scores based on the 5\% trimmed mean and 5\% trimmed
# standard deviation.
quantify.outliers(
    x = x,
    method = 'mean',
    trim = 0.05
    );

# Compute z-scores based on the median and median absolute deviation.
quantify.outliers(
    x = x,
    method = 'median'
    );

# Compute cluster assignments using k-means with k = 2.
quantify.outliers(
    x = x,
    method = 'kmeans'
    );
# Try different initial cluster assignments.
quantify.outliers(
    x = x,
    method = 'kmeans',
    nstart = 10
    );
# Assign zeros to their own cluster.
quantify.outliers(
    x = x,
    method = 'kmeans',
    exclude.zero = TRUE
    );
}
