# PoissonInvLomax.R
#' Poisson Inverse Lomax Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Poisson Inverse Lomax distribution.
#'
#' The Poisson Inverse Lomax distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Poisson Inverse Lomax distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad \frac{1}{\left(1-e^{-\lambda}\right)}
#' \left[1-\exp \left\{-\lambda(1+\beta / x)^{-\alpha}\right\}\right]  \quad ;\;x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The functions available are listed below:
#' \itemize{
#'   \item \code{dpois.inv.lomax()} — Density function
#'   \item \code{ppois.inv.lomax()} — Distribution function
#'   \item \code{qpois.inv.lomax()} — Quantile function
#'   \item \code{rpois.inv.lomax()} — Random generation
#'   \item \code{hpois.inv.lomax()} — Hazard function
#' }
#'
#' @usage
#' dpois.inv.lomax(x, alpha, beta, lambda, log = FALSE)
#' ppois.inv.lomax(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qpois.inv.lomax(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rpois.inv.lomax(n, alpha, beta, lambda)
#' hpois.inv.lomax(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dpois.inv.lomax}: numeric vector of (log-)densities
#'   \item \code{ppois.inv.lomax}: numeric vector of probabilities
#'   \item \code{qpois.inv.lomax}: numeric vector of quantiles
#'   \item \code{rpois.inv.lomax}: numeric vector of random variates
#'   \item \code{hpois.inv.lomax}: numeric vector of hazard values
#' }
#'
#' @references
#' Joshi, R.K., & Kumar, V. (2021).
#' Poisson Inverted Lomax Distribution: Properties and Applications. 
#' \emph{International Journal of Research in Engineering 
#' and Science (IJRES)}, \bold{9(1)}, 48--57.    
#'
#' Chaudhary, A. K., & Kumar, V.(2021). 
#' The ArcTan Lomax Distribution with Properties and Applications. 
#' \emph{International Journal of Scientific Research in Science, Engineering and 
#' Technology(IJSRSET)}, \bold{8(1)}, 117--125. \doi{10.32628/IJSRSET218117}
#'
#' @examples
#' x <- seq(0.1, 10, 0.2)
#' dpois.inv.lomax(x, 2.0, 0.5, 0.2)
#' ppois.inv.lomax(x, 2.0, 0.5, 0.2)
#' qpois.inv.lomax(0.5, 2.0, 0.5, 0.2)
#' rpois.inv.lomax(10, 2.0, 0.5, 0.2)
#' hpois.inv.lomax(x, 2.0, 0.5, 0.2)
#' 
#' # Data
#' x <- stress
#' # ML estimates
#' params = list(alpha=4.1507, beta=5.4091, lambda=80.5762)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = ppois.inv.lomax, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qpois.inv.lomax, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dpois.inv.lomax, pfun=ppois.inv.lomax, plot=FALSE)
#' print.gofic(out)
#' 
#' @name PoissonInvLomax
#' @aliases dpois.inv.lomax ppois.inv.lomax qpois.inv.lomax rpois.inv.lomax hpois.inv.lomax
#' @export
NULL

#' @rdname PoissonInvLomax
#' @usage NULL
#' @export   
dpois.inv.lomax <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
	tmp <- 1/(1 - exp(-lambda))
	u <- (1+(beta/x[valid]))
    gx <- alpha * beta * lambda* (x[valid]^-2)* (u^-(alpha+1)) * exp(-lambda* (u^-alpha))	
    pdf[valid] <- tmp * gx	
	}
    if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname PoissonInvLomax
#' @usage NULL
#' @export  
ppois.inv.lomax <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	tmp <- 1/(1 - exp(-lambda))
	z <- (1+(beta/q[valid])) ^ -alpha
    cdf[valid] <- tmp * (1-exp(-lambda * z))
	}
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname PoissonInvLomax
#' @usage NULL
#' @export 
qpois.inv.lomax <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
    z <- -(1/lambda) * log(1-(1-exp(-lambda))*p[valid])	
    qtl[valid] <- beta /((z ^ (-1/alpha))-1) 	
	}
    return(qtl)   
 } 

#' @rdname PoissonInvLomax
#' @usage NULL
#' @export
rpois.inv.lomax <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qpois.inv.lomax(u, alpha, beta, lambda)
    return(q)
}

#' @rdname PoissonInvLomax
#' @usage NULL
#' @export
hpois.inv.lomax <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
   
    nume <- dpois.inv.lomax(x, alpha, beta, lambda)
    surv <- 1.0 - ppois.inv.lomax(x, alpha, beta, lambda)
    hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
} 
