# HLIW.R
#' Half-Logistic Inverted Weibull (HLIW) Distribution
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Half-Logistic Inverted Weibull distribution.
#'
#' The HLIW distribution is parameterized by shape parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Half-Logistic Inverted Weibull (HLIW) distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \frac{1-\left\{1-e^{-\alpha x^{-\beta}}\right\}^\lambda}
#' {1+\left\{1-e^{-\alpha x^{-\beta}}\right\}^\lambda} \, ; \quad   x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The implementation includes the following functions:
#' \itemize{
#'   \item \code{dHL.inv.weib()} — Density function
#'   \item \code{pHL.inv.weib()} — Distribution function
#'   \item \code{qHL.inv.weib()} — Quantile function
#'   \item \code{rHL.inv.weib()} — Random generation
#'   \item \code{hHL.inv.weib()} — Hazard function
#' }
#'
#' @usage
#' dHL.inv.weib(x, alpha, beta, lambda, log = FALSE)
#' pHL.inv.weib(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qHL.inv.weib(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rHL.inv.weib(n, alpha, beta, lambda)
#' hHL.inv.weib(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric shape parameter
#' @param beta positive numeric rate parameter
#' @param lambda positive numeric shape parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dHL.inv.weib}: numeric vector of (log-)densities
#'   \item \code{pHL.inv.weib}: numeric vector of probabilities
#'   \item \code{qHL.inv.weib}: numeric vector of quantiles
#'   \item \code{rHL.inv.weib}: numeric vector of random variates
#'   \item \code{hHL.inv.weib}: numeric vector of hazard values
#' }
#'
#' @references
#'
#' Elgarhy, M., ul Haq, M.A. & Perveen, I. (2019).  
#' Type II Half Logistic Exponential Distribution with Applications. 
#' \emph{Ann. Data. Sci.}, \bold{6}, 245--257 \doi{10.1007/s40745-018-0175-y}
#' 
#' Chaudhary, A. K., & Kumar, V. (2020). 
#' Half Logistic Exponential Extension Distribution with Properties and Applications. 
#' \emph{International Journal of Recent Technology and Engineering (IJRTE)}, 
#' \bold{8(3)}, 506--512. \doi{10.35940/ijrte.C4625.099320} 
#'
#' Dhungana, G.P. & Kumar, V.(2022). 
#' Half Logistic Inverted Weibull Distribution: Properties and Applications.
#' \emph{J. Stat. Appl. Pro. Lett.}, \bold{9(3)}, 161--178. \doi{10.18576/jsapl/090306}
#'
#' @examples
#' x <- seq(0.1, 5, 0.1)
#' dHL.inv.weib(x, 1.5, 0.8, 2)
#' pHL.inv.weib(x, 1.5, 0.8, 2)
#' qHL.inv.weib(0.5, 1.5, 0.8, 2)
#' rHL.inv.weib(10, 1.5, 0.8, 2)
#' hHL.inv.weib(x, 1.5, 0.8, 2)
#'
#' #Data
#' x <- survtimes
#' gofic(x,
#'       params = list(alpha=31.1650, beta=0.4213, lambda=45.5485),
#'       dfun = dHL.inv.weib, pfun = pHL.inv.weib, plot=TRUE, verbose = TRUE)
#' 
#' pp.plot(x,
#'         params = list(alpha=31.1650, beta=0.4213, lambda=45.5485),
#'         pfun = pHL.inv.weib, fit.line=TRUE)
#' 
#' qq.plot(x,
#'         params = list(alpha=31.1650, beta=0.4213, lambda=45.5485),
#'         qfun = qHL.inv.weib, fit.line=TRUE)
#' 
#' @name HLIW
#' @aliases dHL.inv.weib pHL.inv.weib qHL.inv.weib rHL.inv.weib hHL.inv.weib
#' @export
NULL

#' @rdname HLIW
#' @usage NULL
#' @export
dHL.inv.weib <- function(x, alpha, beta, lambda, log = FALSE) 
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {	
	u <- exp(-alpha * (x[valid] ^ -beta))
	z <- (1 - u) ^ (lambda-1)
	w <- (1+(1 - u)^ lambda) 
	pdf[valid] <- 2*alpha * beta * lambda * u * z * (x[valid]^-(beta+1))* (w ^ -2)
  }
  if (log) 
	pdf <- log(pdf)
  return(pdf)
}

#' @rdname HLIW
#' @usage NULL
#' @export
pHL.inv.weib <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE) 
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	u <- exp(-alpha * (q[valid] ^ -beta))
	z <- (1 - u) ^ lambda
	cdf[valid] <- (1 - z)/(1 + z)
	}
	if (!lower.tail) 
		cdf <- 1 - cdf
	if (log.p) 
		cdf <- log(cdf)
  return(cdf)
}

#' @rdname HLIW
#' @usage NULL
#' @export
qHL.inv.weib <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE) 
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
	z <- ((1- p[valid])/(1+ p[valid])) ^ (1/lambda)
	qtl[valid] <- (-(1 / alpha) * log(1 - z)) ^ (-1 / beta)
	}
  return(qtl)
}

#' @rdname HLIW
#' @usage NULL
#' @export
rHL.inv.weib <- function(n, alpha, beta, lambda) 
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
  u <- runif(n)
  q <- qHL.inv.weib(u, alpha, beta, lambda)
  return(q)
  
}

#' @rdname HLIW
#' @usage NULL
#' @export
hHL.inv.weib <- function(x, alpha, beta, lambda) {
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")  
		
	nume <- dHL.inv.weib(x, alpha, beta, lambda)
	surv <- 1 - pHL.inv.weib(x, alpha, beta, lambda)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
  return(hrf)
}
