conditionalreturnsHMMgev <- function(x, HMM, return_periods = c(50, 100, 200, 500),
                                     varcov = NULL, B = 10000, level = 0.95,
                                     time_structure = NULL,
                                     plot_title = "Return Levels Over Time",
                                     save_plot = FALSE, filename = NULL,
                                     width = 12, height = 8, dpi = 300,
                                     verbose = TRUE) {

  # Suppress scientific notation for cleaner output
  old_scipen <- options(scipen = 999)
  on.exit(options(old_scipen), add = TRUE)

  nk <- length(HMM$estimate$delta)

  # Extract GEV parameters with log transformation for scale parameter
  gev.param <- c(
    HMM$estimate$loc,
    log(HMM$estimate$scale),
    HMM$estimate$shape
  )

  # Compute or use provided variance-covariance matrix
  if (is.null(varcov)) {
    if (verbose) message("Computing numerical variance-covariance matrix...")
    gev.varc <- HMMVarianceMatrix(
      x = x, HMM = HMM, obsdist = "gev", verbose=verbose
    )
  } else {
    gev.varc <- varcov
  }

  # Extract observation parameters and decode state sequence
  obspar <- HMM$estimate[setdiff(names(HMM$estimate), c("Pi", "delta"))]
  vky <- globaldecodeHMM(
    x = x, HMM=HMM, obsdist = "gev"
  )

  n_periods <- length(return_periods)

  # Convert return periods to probabilities (assuming weekly data)
  return_probs <- 1 / (return_periods * 365.25/7)

  if (verbose) message("Computing return levels by state...")

  # Initialize return level storage for each return period
  rl_states <- list()
  for (i in 1:n_periods) {
    rl_states[[i]] <- matrix(0, 3, nk)  # [estimate, lower CI, upper CI] x states
  }

  # Compute return levels for each state
  for (jj in 1:nk) {
    if (verbose && (jj == 1 || jj %% max(1, nk %/% 5) == 0)) {
      message(sprintf("Processing state %d / %d", jj, nk))
    }

    for (rp_idx in 1:n_periods) {
      rp <- return_periods[rp_idx]
      p <- return_probs[rp_idx]

      # Define function to find return level by solving P(X > x) = p
      rl_function <- function(x_val) {
        loc.bt <- gev.param[1:nk]
        scale.bt <- exp(gev.param[(nk+1):(2*nk)])
        shape.bt <- gev.param[(2*nk+1):(3*nk)]

        rl.fun <- pevd(x_val, loc = loc.bt[jj], scale = scale.bt[jj], shape = shape.bt[jj],
                       type = c("GEV"), npy = 365.25/7)
        (rl.fun - 1 + p)
      }

      # Set search range based on return period
      search_range <- if (rp <= 100) c(0, 30000) else if (rp <= 200) c(0, 50000) else c(0, 100000)
      rl_states[[rp_idx]][1, jj] <- uniroot(rl_function, search_range)$root

      # Bootstrap confidence intervals
      ret_levels_bt <- rep(0, B)
      b <- 1

      while (b <= B) {
        if (verbose && b %% max(1, B %/% 10) == 0) {
          message(sprintf("State %d: Bootstrap sample %d / %d", jj, b, B))
        }

        # Generate bootstrap parameter sample
        gev.param.bt <- rmnorm(n = 1, mean = gev.param, varcov = gev.varc)
        loc.bt <- gev.param.bt[1:nk]
        scale.bt <- exp(gev.param.bt[(nk+1):(2*nk)])
        shape.bt <- gev.param.bt[(2*nk+1):(3*nk)]

        # Define bootstrap return level function
        rl_bt_function <- function(x_val) {
          loc.bt <- gev.param.bt[1:nk]
          scale.bt <- exp(gev.param.bt[(nk+1):(2*nk)])
          shape.bt <- gev.param.bt[(2*nk+1):(3*nk)]

          rl.fun <- pevd(x_val, loc = loc.bt[jj], scale = scale.bt[jj], shape = shape.bt[jj],
                         type = c("GEV"), npy = 365.25/7)
          (rl.fun - 1 + p)
        }

        # Try to find the root, if it fails, generate new bootstrap sample
        tryCatch({
          ret_levels_bt[b] <- uniroot(rl_bt_function, search_range)$root
          b <- b + 1  # Only increment if successful
        }, error = function(e) {
          # Don't increment b, just try again with a new bootstrap sample
          NULL
        })
      }

      # Store confidence interval bounds
      rl_states[[rp_idx]][2:3, jj] <- quantile(ret_levels_bt, probs = c(0.025, 0.975))
    }
  }

  # Map state-specific return levels to time series
  rl_all <- list()
  for (rp_idx in 1:n_periods) {
    rl_all[[rp_idx]] <- matrix(0, 3, length(x))
    for (k in 1:length(x)) {
      rl_all[[rp_idx]][, k] <- rl_states[[rp_idx]][, vky[k]]
    }
  }

  # Create time information for plotting using flexible time structure
  time_info <- createTimeInfo(length(x), time_structure)

  # Set up plot output
  if (save_plot) {
    if (is.null(filename)) {
      stop("filename must be specified when save_plot = TRUE")
    }

    if (grepl("\\.eps$", filename)) {
      postscript(filename, paper = "special",
                 height = 2.25 * 4.5 / 1.25,
                 width = 4 * 4.5 / 1.25)
    } else if (grepl("\\.png$", filename)) {
      png(filename, width = width, height = height, units = "in", res = dpi)
    } else {
      pdf(filename, width = width, height = height)
    }
    on.exit(dev.off(), add = TRUE)
  }

  oldpar <- par(no.readonly = TRUE)
  on.exit(par(oldpar), add = TRUE)

  # Create 2x2 panel layout
  par(mfrow = c(2, 2))
  par(mar = c(4.5, 7, 1.5, 1))

  # Plot return levels for up to 4 return periods
  for (rp_idx in 1:min(4, n_periods)) {
    rp <- return_periods[rp_idx]

    # Create log-scale plot with return level estimate
    plot(time_info$labels, rl_all[[rp_idx]][1,], type = "l",
         ylim = c(min(rl_all[[rp_idx]]) * 0.8, max(rl_all[[rp_idx]]) * 7.0),
         xlab = time_info$x_label, ylab = "Return level (nT/min)",
         axes = FALSE, log = "y", cex.lab = 1.3)

    box()

    # Add x-axis
    axis(1, cex.axis = 1.2)

    # Add y-axis with horizontal labels and no scientific notation
    y_ticks <- axTicks(2)
    y_labels <- formatC(y_ticks, format = "f", digits = 0, big.mark = "")
    axis(2, at = y_ticks, labels = y_labels, cex.axis = 0.7, las = 1)

    # Add additional y-axis tick for high return levels
    if (rp >= 200 && max(rl_all[[rp_idx]]) >= 15000) {
      axis(2, at = 20000, labels = "20000", cex.axis = 0.7, las = 1)
    }

    # Add confidence interval lines
    lines(time_info$labels, rl_all[[rp_idx]][2,], lty = "dashed", col = "black")
    lines(time_info$labels, rl_all[[rp_idx]][3,], lty = "dashed", col = "black")

    # Add panel label with return period positioned in top-left corner
    legend_text <- paste0("(", letters[rp_idx], ") HMM ", rp, "-year")

    legend("topleft", legend = legend_text, bty = "n", cex = 0.9,
           x.intersp = 0.1, y.intersp = 0.1, inset = c(0.005, 0.005))
  }

  if (save_plot && verbose) {
    message("Plot saved to: ", filename)
  }

  invisible(list(
    return_levels_states = rl_states,
    return_levels_time = rl_all,
    return_periods = return_periods,
    time_info = time_info
  ))
}


createTimeInfo <- function(n, time_structure = NULL) {

  # Default to simple observation numbering if no structure provided
  if (is.null(time_structure)) {
    return(list(
      labels = 1:n,
      unit = "observation",
      conversion_factor = 1,
      unit_name = "observation",
      x_label = "Time"
    ))
  }

  # Validate required fields based on time structure type
  required_fields <- c("unit", "observations_per_unit")
  if (time_structure$unit == "custom") {
    required_fields <- c("conversion_factor", "unit_name")
  }

  missing_fields <- setdiff(required_fields, names(time_structure))
  if (length(missing_fields) > 0) {
    stop("Missing required fields in time_structure: ", paste(missing_fields, collapse = ", "))
  }

  # Set conversion factor and unit names
  if (time_structure$unit == "custom") {
    conversion_factor <- time_structure$conversion_factor
    unit_name <- time_structure$unit_name
    x_label <- paste("Time (", unit_name, "s)", sep = "")
  } else {
    conversion_factor <- time_structure$observations_per_unit
    unit_name <- time_structure$unit

    # Create appropriate axis labels for common units
    if (unit_name == "year") {
      x_label <- "Time (years)"
    } else if (unit_name == "day") {
      x_label <- "Time (days)"
    } else if (unit_name == "hour") {
      x_label <- "Time (hours)"
    } else {
      x_label <- paste("Time (", unit_name, "s)", sep = "")
    }
  }

  # Calculate time duration and handle start/end points
  duration <- n / conversion_factor
  has_start <- !is.null(time_structure$start_point)
  has_end <- !is.null(time_structure$end_point)

  if (has_start && has_end) {
    start_point <- time_structure$start_point
    end_point <- time_structure$end_point
    time_labels <- seq(start_point, end_point, length.out = n)
  } else if (has_start && !has_end) {
    start_point <- time_structure$start_point
    end_point <- start_point + duration
    time_labels <- seq(start_point, end_point, length.out = n)
  } else if (!has_start && has_end) {
    end_point <- time_structure$end_point
    start_point <- end_point - duration
    time_labels <- seq(start_point, end_point, length.out = n)
  } else {
    start_point <- 0
    end_point <- duration
    time_labels <- seq(start_point, end_point, length.out = n)
  }

  return(list(
    labels = time_labels,
    unit = ifelse(time_structure$unit == "custom", "custom", time_structure$unit),
    conversion_factor = conversion_factor,
    unit_name = unit_name,
    x_label = x_label,
    duration = duration,
    start_point = start_point,
    end_point = end_point
  ))
}
