#' Experimental Differential Evolution - ExpDE
#'
#' This function is an interface to call the main function \code{\link{ExpDE}}, 
#' usinga method specification generated by \code{\link{gen_methods}} and a 
#' problem specification generated by \code{\link{gen_problems}}. It is used to 
#' provide an easy data-generating process to study the experimental comparisons of 
#' algorithms. See the documentation of \code{\link{ExpDE}} for details of the 
#' underlying algorithm.
#' 
#' @param method.name A specific method key name, generated by \code{\link{gen_methods}}. 
#'        This must be a character string in the format "DE.[integer].[single letter]".
#' @param problem.name A specific problem key name, generated by \code{\link{gen_problems}}.
#'        This must be a character string in the format "Prob.[integer].[integer]".
#' @param seed positive integer, seed for the random number generator.
#' 
#' @return A single scalar value representing the final value returned by 
#' running the DE method specified by `method.name` on the problem specified 
#' by `problem.name`.
#'
#' @examples
#' library(ExpDE)
#' myprobs <- gen_problems(1234567, nprobs=10)
#' myalgos <- gen_methods(1234567)
#' onerun  <- ExpDE2(method.name  = "DE.1234567.A", 
#'                   problem.name = "Prob.1234567.6")
#' 
#' @export




ExpDE2 <- function(method.name, problem.name, seed = NULL){
  
  #  ========== Error catching and default value definitions 
  if (!is.null(seed)) {
    assertthat::assert_that(assertthat::is.count(seed))
    set.seed(seed)               # set PRNG seed
  }
  
  assertthat::assert_that(is.character(method.name),
                          length(method.name) == 1,
                          is.character(problem.name),
                          length(problem.name) == 1, 
                          msg = "both method.name and problem.name must be a character string. See ?ExpDE2")
  
  method.split <- strsplit(method.name, split = "\\.")[[1]]
  assertthat::assert_that(method.split[[1]] == "DE",
                          !is.na(as.numeric(method.split[[2]])),
                          nchar(method.split[[3]]) == 1,
                          msg = "argument method.name is not in the correct format. See ?ExpDE2")
  
  prob.split <- strsplit(problem.name, split = "\\.")[[1]]
  assertthat::assert_that(prob.split[[1]] == "Prob",
                          !is.na(as.numeric(prob.split[[2]])),
                          !is.na(as.numeric(prob.split[[3]])), 
                          msg = "argument problem.name is not in the correct format. See ?ExpDE2")
  
  #  ==========
                            
  method.id  <- as.numeric(method.split[[2]])
  method.idx <- which(LETTERS == method.split[3])
  prob.id    <- as.numeric(prob.split[[2]])
  prob.idx   <- as.numeric(prob.split[[3]])
  
  algpars    <- gen_methods(method.id, echo = FALSE)[[method.idx]]
  probpars   <- gen_problems(prob.id, prob.idx, echo = FALSE)[[prob.idx]]
  probpars$ProblemName <- NULL
  
  run <- ExpDE(popsize  = algpars$popsize, 
               mutpars  = algpars$mutpars,
               recpars  = algpars$recpars, 
               selpars  = algpars$selpars, 
               stopcrit = algpars$stopcrit, 
               probpars = probpars,
               showpars = algpars$showpars)
  
  return(run$Fbest)
  
}