#' Run Model Executor from Pirana
#'
#' Used by Pirana internally to launch the Model Executor Shiny GUI.
#' 
#' @param metamodelFile Path to existing metamodel file \code{.mmdl}.
#' @param hostsfile Path to hosts definitions file \code{.json}. Note,
#' this file is automatically generated by Pirana given the user provided
#' NLME host setup in Pirana settings. 
#'
#' @return Deploys a Shiny app to execute a \code{Certara.RsNLME} model. Returns \code{NULL} if assigned to an object.
#' @examples
#' if (interactive()) {
#' # Get existing mmdl file
#' mmdl_file <- system.file("vignettesdata/OneCpt_IVInfusion.mmdl",
#'   package = "Certara.RsNLME")
#'
#' # Create hosts file json. Note, hosts file is automatically created by Pirana.
#' hosts_file <- tempfile(pattern = "hosts", fileext = ".json")
#' jsonlite::write_json(
#'   list(
#'     list(profile_name = "examplehost", cores_number = 1, os = "Windows", parallel_mode="None")),
#'   auto_unbox = TRUE, 
#'   path = hosts_file
#'   ) 
#' execute_mmdlModel(
#'   mmdl_file, 
#'   hosts_file
#'   )
#' }
#' @export
execute_mmdlModel <- function(metamodelFile, hostsfile) {
  
  metamodelFile <- gsub("\\", "/", metamodelFile, fixed = TRUE)
  if(!file.exists(metamodelFile)){
      stop("Cannot find specified metamodel file ", metamodelFile)
  }
  
  if(!file.exists(hostsfile)){
    stop("Cannot find specified hosts file ", hostsfile)
  }
  
  if(!requireNamespace("Certara.RsNLME", quietly = TRUE)) {
    stop("Cannot find RsNLME package")
  } else {
    
    INSTALLDIR <- Sys.getenv("INSTALLDIR")
    
    if(!all(Certara.NLME8::checkInstallDir(INSTALLDIR), Certara.NLME8::checkLicenseFile(INSTALLDIR))) {
      stop("Cannot use NLME without valid NLME executables")
    }
    
    host <- parse_NLMEHosts(hostsfile)
    
    fitmodel_args <- create_model_from_metamodel(metamodelFile, directoryToRun = dirname(metamodelFile))
    
    workDir <- if (dirname(metamodelFile) == ".") normalizePath(".", winslash = "/", mustWork = FALSE) else dirname(metamodelFile)
    
    modelExecutorUI(fitmodel_args$model, hosts = c(host), wd = workDir, hostsfile, metamodelFile, fromPirana = TRUE)
  }
}