/*
 * Decompiled with CFR 0.152.
 */
package org.torproject.descriptor.index;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.torproject.descriptor.DescriptorCollector;
import org.torproject.descriptor.index.FileNode;
import org.torproject.descriptor.index.IndexNode;

public class DescriptorIndexCollector
implements DescriptorCollector {
    private static Logger log = LoggerFactory.getLogger(DescriptorIndexCollector.class);

    @Override
    public void collectDescriptors(String collecTorIndexUrlString, String[] remoteDirectories, long minLastModified, File localDirectory, boolean deleteExtraneousLocalFiles) {
        SortedMap<String, FileNode> remoteFiles;
        IndexNode index;
        log.info("Starting descriptor collection.");
        if (minLastModified < 0L) {
            throw new IllegalArgumentException("A negative minimum last-modified time is not permitted.");
        }
        if (null == collecTorIndexUrlString || null == remoteDirectories || null == localDirectory) {
            throw new IllegalArgumentException("Null values are not permitted for CollecTor base URL, remote directories, or local directory.");
        }
        if (localDirectory.isFile()) {
            throw new IllegalArgumentException("A non-directory file exists at {} which is supposed to be the local directory for storing remotely fetched files.  Move this file away or delete it.  Aborting descriptor collection.");
        }
        log.info("Indexing local directory {}.", (Object)localDirectory.getAbsolutePath());
        SortedMap<String, Long> localFiles = DescriptorIndexCollector.statLocalDirectory(localDirectory);
        String indexUrlString = "";
        try {
            URL indexUrl = new URL(collecTorIndexUrlString);
            indexUrlString = indexUrl.toString();
            if (indexUrl.getPath().isEmpty()) {
                indexUrlString = indexUrlString + "/index/index.json";
            }
            log.info("Fetching remote index file {}.", (Object)indexUrlString);
            index = IndexNode.fetchIndex(indexUrlString);
            remoteFiles = index.retrieveFilesIn(remoteDirectories);
        }
        catch (Exception ex) {
            log.warn("Cannot fetch index file {} and hence cannot determine which remote files to fetch.  Aborting descriptor collection.", (Object)indexUrlString, (Object)ex);
            return;
        }
        log.info("Fetching remote files from {}.", (Object)index.path);
        if (!this.fetchRemoteFiles(index.path, remoteFiles, minLastModified, localDirectory, localFiles)) {
            return;
        }
        if (deleteExtraneousLocalFiles) {
            log.info("Deleting extraneous files from local directory {}.", (Object)localDirectory);
            DescriptorIndexCollector.deleteExtraneousLocalFiles(remoteDirectories, remoteFiles, localDirectory, localFiles);
        }
        log.info("Finished descriptor collection.");
    }

    boolean fetchRemoteFiles(String baseUrl, SortedMap<String, FileNode> remotes, long minLastModified, File localDir, SortedMap<String, Long> locals) {
        for (Map.Entry<String, FileNode> entry : remotes.entrySet()) {
            String filepathname = entry.getKey();
            String filename = entry.getValue().path;
            File filepath = new File(localDir, filepathname.replace(filename, ""));
            long lastModifiedMillis = entry.getValue().lastModifiedMillis();
            if (lastModifiedMillis < minLastModified || locals.containsKey(filepathname) && (Long)locals.get(filepathname) >= lastModifiedMillis) continue;
            if (!filepath.exists() && !filepath.mkdirs()) {
                log.warn("Cannot create local directory {} to store remote file {}.  Aborting descriptor collection.", (Object)filepath, (Object)filename);
                return false;
            }
            File destinationFile = new File(filepath, filename);
            File tempDestinationFile = new File(filepath, "." + filename);
            log.debug("Fetching remote file {} with expected size of {} bytes from {}, storing locally to temporary file {}, then renaming to {}.", filepathname, entry.getValue().size, baseUrl, tempDestinationFile.getAbsolutePath(), destinationFile.getAbsolutePath());
            try {
                InputStream is = new URL(baseUrl + "/" + filepathname).openStream();
                try {
                    Files.copy(is, tempDestinationFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
                    if (tempDestinationFile.length() == entry.getValue().size) {
                        tempDestinationFile.renameTo(destinationFile);
                        destinationFile.setLastModified(lastModifiedMillis);
                        continue;
                    }
                    log.warn("Fetched remote file {} from {} has a size of {} bytes which is different from the expected {} bytes.  Not storing this file.", filename, baseUrl, tempDestinationFile.length(), entry.getValue().size);
                }
                finally {
                    if (is == null) continue;
                    is.close();
                }
            }
            catch (IOException e) {
                log.warn("Cannot fetch remote file {} from {}.  Skipping that file.", filename, baseUrl, e);
            }
        }
        return true;
    }

    static void deleteExtraneousLocalFiles(String[] remoteDirectories, SortedMap<String, FileNode> remoteFiles, File localDir, SortedMap<String, Long> locals) {
        for (String localPath : locals.keySet()) {
            for (String remoteDirectory : remoteDirectories) {
                String remDir;
                String string = remDir = remoteDirectory.charAt(0) == '/' ? remoteDirectory.substring(1) : remoteDirectory;
                if (!localPath.startsWith(remDir) || remoteFiles.containsKey(localPath)) continue;
                File extraneousLocalFile = new File(localDir, localPath);
                log.debug("Deleting extraneous local file {}.", (Object)extraneousLocalFile.getAbsolutePath());
                extraneousLocalFile.delete();
            }
        }
    }

    static SortedMap<String, Long> statLocalDirectory(final File localDir) {
        final TreeMap<String, Long> locals = new TreeMap<String, Long>();
        if (!localDir.exists()) {
            return locals;
        }
        try {
            Files.walkFileTree(localDir.toPath(), (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path path, BasicFileAttributes bfa) {
                    locals.put(path.toFile().getAbsolutePath().replace(localDir.getAbsolutePath() + "/", ""), path.toFile().lastModified());
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (IOException ioe) {
            log.warn("Cannot index local directory {} to skip any remote files that already exist locally.  Continuing with an either empty or incomplete index of local files.", (Object)localDir, (Object)ioe);
        }
        return locals;
    }
}

