/**
 * ./src/classes/ovary.d/luks-helpers.ts
 * penguins-eggs v.25.10.x / ecmascript 2020
 * author: Piero Proietti
 * email: piero.proietti@gmail.com
 * license: MIT
 */
import { spawn } from 'node:child_process';
import Utils from '../utils.js';
const noop = () => { };
/**
 * Funzione helper per eseguire comandi esterni in modo asincrono,
 * gestendo lo standard input per passare le password.
 * Restituisce una Promise che si risolve al successo o si rigetta in caso di errore.
 */
export function luksExecuteCommand(command, args, stdinData) {
    if (!this.hidden) {
        Utils.info(`${command} ${args.join(' ')}`);
    }
    return new Promise((resolve, reject) => {
        // Se passiamo dati a stdin, dobbiamo usare 'pipe'. Altrimenti, 'inherit'.
        const stdioConfig = stdinData ? ['pipe', 'inherit', 'inherit'] : 'inherit';
        const process = spawn(command, args, { stdio: stdioConfig });
        // Se fornito, scriviamo i dati (es. la password) nello stdin del processo.
        if (stdinData && process.stdin) {
            process.stdin.write(stdinData);
            process.stdin.end();
        }
        process.on('error', (err) => {
            reject(new Error(`Error starting command "${command}": ${err.message}`));
        });
        process.on('close', (code) => {
            if (code === 0) {
                resolve(); // Success
            }
            else {
                reject(new Error(`Command "${command} ${args.join(' ')}" ended with error code ${code}`));
            }
        });
    });
}
/**
 * buildLuksFormatArgs
 */
export function buildLuksFormatArgs(config, luksFile) {
    const args = [
        '--batch-mode', // Per saltare la conferma "YES"
        'luksFormat',
        '--type', 'luks2',
        // Parametri base
        '--cipher', config.cipher,
        '--key-size', config['key-size'].toString(),
        '--hash', config.hash,
        '--sector-size', config['sector-size'].toString(),
        '--pbkdf', config.pbkdf,
    ];
    // Aggiungi i parametri condizionali del PBKDF
    switch (config.pbkdf) {
        case 'argon2id':
        case 'argon2i':
            const argonConfig = config;
            args.push('--pbkdf-memory', argonConfig['pbkdf-memory (KiB)'].toString());
            args.push('--pbkdf-parallel', argonConfig['pbkdf-parallel (threads)'].toString());
            break;
        case 'pbkdf2':
            const pbkdf2Config = config;
            args.push('--iter-time', pbkdf2Config['iter-time (ms)'].toString());
            break;
    }
    // Aggiungi il file di destinazione
    args.push(luksFile);
    return args;
}
