#
# TCL Library for tkcvs
#

#
# $Id: workdir.tcl,v 1.140 2004/03/14 03:44:08 dorothyr Exp $
#
# Current working directory display.  Handles all of the functions
# concerned with navigating about the current directory on the main
# window.
#

proc workdir_setup {} {
  global cwd
  global module_dir
  global cvscfg
  global cvsglb
  global current_tagname
  global logclass
  global tcl_platform

  gen_log:log T "ENTER"
  set cwd [pwd]
  set pid [pid]

  if {[winfo exists .workdir]} {
    wm deiconify .workdir
    raise .workdir
    return
  }

  # Make a new toplevel and unmap . so that the working directory browser
  # the module browser are equal
  toplevel .workdir
  wm title .workdir "TkCVS Working Directory"
  wm iconname .workdir "TkCVS"
  if {$tcl_platform(platform) != "windows"} {
    wm iconbitmap .workdir @$cvscfg(bitmapdir)/tkcvs48.xbm
  }
  wm minsize .workdir 430 300
  wm protocol .workdir WM_DELETE_WINDOW { .workdir.close invoke }
  wm withdraw .

  if {[catch "image type Conflict"]} {
    workdir_images
  }
  # Ignore the dimensions, because they depend on the content.  Use
  # the remembered position though.
  if {[info exists cvscfg(workgeom)]} {
    #regsub {\d+x\d+} $cvscfg(workgeom) {} winloc
    #wm geometry .workdir $winloc
    wm geometry .workdir $cvscfg(workgeom)
  }

  workdir_menus

  #
  # Top section - where we are, where the module is
  #
  frame .workdir.top -relief groove -border 2
  pack .workdir.top -side top -fill x

  ::picklist::entry .workdir.top.tcwd cwd directory
  ::picklist::bind .workdir.top.tcwd <Return> {change_dir "$cwd"}

  button .workdir.top.updir_btn -image updir \
    -command {change_dir ..}

  label .workdir.top.lmodule -text "Module"
  label .workdir.top.tmodule -textvariable module_dir -anchor w -relief groove

  label .workdir.top.ltagname -text "Tag"
  label .workdir.top.ttagname -textvariable current_tagname -anchor w -relief groove

  # Make the Module Browser button prominent
  button .workdir.top.bmodbrowse -image Modules \
     -command modbrowse_run

  label .workdir.top.lcvsroot -text "CVSROOT"
  label .workdir.top.tcvsroot -textvariable cvscfg(cvsroot) -anchor w -relief groove


  grid columnconf .workdir.top 1 -weight 1
  grid rowconf .workdir.top 3 -weight 1
  grid .workdir.top.updir_btn -column 0 -row 0 -sticky s
  grid .workdir.top.tcwd -column 1 -row 0 -columnspan 2 -sticky sew -padx 4 -pady 1
  grid .workdir.top.lmodule -column 0 -row 1 -sticky nw
  grid .workdir.top.tmodule -column 1 -row 1 -columnspan 2 -padx 4 -pady 1 -sticky new
  grid .workdir.top.bmodbrowse -column 2 -row 2 -rowspan 2 -sticky w
  grid .workdir.top.ltagname -column 0 -row 2 -sticky nw
  grid .workdir.top.ttagname -column 1 -row 2 -padx 4 -pady 1 -sticky new
  grid .workdir.top.lcvsroot -column 0 -row 3 -sticky nw
  grid .workdir.top.tcvsroot -column 1 -row 3 -padx 4 -pady 1 -sticky new


  # Pack the bottom before the middle so it doesnt disappear if
  # the window is resized smaller
  #frame .workdir.bottom -relief groove -border 2 -height 128
  frame .workdir.bottom
  frame .workdir.bottom.filters -relief raised
  pack .workdir.bottom -side bottom -fill x
  pack .workdir.bottom.filters -side top -fill x

  label .workdir.bottom.filters.showlbl -text "Show:" -anchor w
  entry .workdir.bottom.filters.showentry -textvariable cvscfg(file_filter) -width 12
  label .workdir.bottom.filters.hidelbl -text "   Hide:" -anchor w
  entry .workdir.bottom.filters.hideentry -width 12 \
     -textvariable cvsglb(default_ignore_filter)
  label .workdir.bottom.filters.space -text "    "
  button .workdir.bottom.filters.cleanbutton -text "Clean:" \
     -pady 0 -highlightthickness 0 \
     -command workdir_cleanup
  entry .workdir.bottom.filters.cleanentry -width 12 \
     -textvariable cvscfg(clean_these)
  bind .workdir.bottom.filters.showentry <Return> {setup_dir}
  bind .workdir.bottom.filters.hideentry <Return> {
     set cvsglb(default_ignore_filter) [.workdir.bottom.filters.hideentry get]
     setup_dir}
  bind .workdir.bottom.filters.cleanentry <Return> {workdir_cleanup}
  pack .workdir.bottom.filters.showlbl -side left
  pack .workdir.bottom.filters.showentry -side left
  pack .workdir.bottom.filters.hidelbl -side left
  pack .workdir.bottom.filters.hideentry -side left
  pack .workdir.bottom.filters.space -side left
  pack .workdir.bottom.filters.cleanbutton -side left -ipadx 2 -ipady 0
  pack .workdir.bottom.filters.cleanentry -side left

  frame .workdir.bottom.buttons -relief groove -bd 2
  frame .workdir.bottom.buttons.funcs -relief groove -bd 2
  frame .workdir.bottom.buttons.cvsfuncs -relief groove -bd 2
  frame .workdir.bottom.buttons.close -relief groove -bd 2
  pack .workdir.bottom.buttons -side top -fill x -expand yes
  pack .workdir.bottom.buttons.close -side right -padx 10
  pack .workdir.bottom.buttons.funcs -side left
  pack .workdir.bottom.buttons.cvsfuncs -side left -expand yes

  #
  # Action buttons along the bottom of the screen.
  #
  button .workdir.bottom.buttons.cvsfuncs.bcheckdir -image Check \
     -command { cvs_check [workdir_list_files] }
  button .workdir.bottom.buttons.funcs.bedit_files -image Fileedit \
     -command { workdir_edit_file [workdir_list_files] }
  button .workdir.bottom.buttons.funcs.bview_files -image Fileview \
     -command { workdir_view_file [workdir_list_files] }
  button .workdir.bottom.buttons.funcs.bdelete_file -image Delete \
     -command { workdir_delete_file [workdir_list_files] }
  button .workdir.bottom.buttons.funcs.bmkdir -image Dir_new \
     -command { file_input_and_do "New Directory" workdir_newdir}
  button .workdir.bottom.buttons.funcs.brefresh -image Refresh \
     -command { change_dir [pwd] }
  button .workdir.bottom.buttons.cvsfuncs.bconflict -image Conflict \
     -command { cvs_merge_conflict [workdir_list_files] }

  button .workdir.bottom.buttons.cvsfuncs.blogfile -image Branches \
     -command { cvs_logcanvas [pwd] [workdir_list_files] }
  button .workdir.bottom.buttons.cvsfuncs.bjoin -image DirBranches \
     -command { cvs_joincanvas }
  button .workdir.bottom.buttons.cvsfuncs.btag -image Tag \
     -command { file_tag_dialog "no" }
  button .workdir.bottom.buttons.cvsfuncs.bbranchtag -image Branchtag \
     -command { file_tag_dialog "yes" }
  button .workdir.bottom.buttons.cvsfuncs.badd_files -image Add \
     -command { add_dialog [workdir_list_files] }
  button .workdir.bottom.buttons.cvsfuncs.bremove -image Remove \
     -command { subtract_dialog [workdir_list_files] }
  button .workdir.bottom.buttons.cvsfuncs.bdiff -image Diff \
     -command { cvs_diff [workdir_list_files] }
  button .workdir.bottom.buttons.cvsfuncs.bcheckin -image Checkin \
     -command commit_run
  button .workdir.bottom.buttons.cvsfuncs.bupdate -image Checkout \
     -command { cvs_update "BASE" "Normal" \
        "Remove" "No" " " [workdir_list_files] }
  button .workdir.bottom.buttons.cvsfuncs.bannotate -image Annotate \
     -command { cvs_annotate $current_tagname [workdir_list_files] }
  button .workdir.bottom.buttons.cvsfuncs.bupdateopts -image CheckoutOpts \
     -command { update_run }
  button .workdir.bottom.buttons.cvsfuncs.bcvsedit_files -image Edit \
     -command { cvs_edit [workdir_list_files] }
  button .workdir.bottom.buttons.cvsfuncs.bunedit_files -image Unedit \
     -command { cvs_unedit [workdir_list_files] }
  button .workdir.close -text "Close" \
      -command {
        global cvscfg
        set cvscfg(workgeom) [wm geometry .workdir]
        destroy .workdir
        exit_cleanup 0
      }

  # These buttons work in any directory
  grid columnconf .workdir.bottom.buttons.funcs 2 -weight 1
  grid rowconf .workdir.bottom.buttons.funcs 1 -weight 1
  grid .workdir.bottom.buttons.funcs.bdelete_file -column 0 -row 0
  grid .workdir.bottom.buttons.funcs.bmkdir -column 0 -row 1
  grid .workdir.bottom.buttons.funcs.bedit_files -column 1 -row 0 \
     -ipadx 2
  grid .workdir.bottom.buttons.funcs.bview_files -column 1 -row 1 \
     -ipadx 2
  grid .workdir.bottom.buttons.funcs.brefresh -column 2 -row 0 \
     -rowspan 2 -ipadx 2 -ipady 2 -sticky ns

  # These buttons are only good in a CVS directory
  grid columnconf .workdir.bottom.buttons.cvsfuncs 9 -weight 1
  grid rowconf .workdir.bottom.buttons.cvsfuncs 1 -weight 1
  grid .workdir.bottom.buttons.cvsfuncs.bcheckdir -column 1 -row 0 \
     -rowspan 2 -ipadx 2 -ipady 2 -sticky ns
  grid .workdir.bottom.buttons.cvsfuncs.blogfile -column 2 -row 0
  grid .workdir.bottom.buttons.cvsfuncs.bjoin -column 2 -row 1
  grid .workdir.bottom.buttons.cvsfuncs.bdiff -column 3 -row 0
  grid .workdir.bottom.buttons.cvsfuncs.bconflict -column 3 -row 1
  grid .workdir.bottom.buttons.cvsfuncs.bupdate -column 4 -row 0 \
    -ipadx 8
  grid .workdir.bottom.buttons.cvsfuncs.bcheckin -column 4 -row 1 \
    -ipadx 8
  grid .workdir.bottom.buttons.cvsfuncs.bannotate -column 5 -row 1
  grid .workdir.bottom.buttons.cvsfuncs.bupdateopts -column 5 -row 0
  grid .workdir.bottom.buttons.cvsfuncs.badd_files -column 6 -row 0
  grid .workdir.bottom.buttons.cvsfuncs.bremove -column 6 -row 1
  grid .workdir.bottom.buttons.cvsfuncs.btag -column 7 -row 0
  grid .workdir.bottom.buttons.cvsfuncs.bbranchtag -column 7 -row 1
  grid .workdir.bottom.buttons.cvsfuncs.bcvsedit_files -column 8 -row 0 \
     -sticky w
  grid .workdir.bottom.buttons.cvsfuncs.bunedit_files -column 8 -row 1 \
     -sticky w

  pack .workdir.close -in .workdir.bottom.buttons.close \
    -side right -fill both -expand yes

  set_tooltips .workdir.top.updir_btn \
     {"Go up (..)"}
  set_tooltips .workdir.bottom.buttons.funcs.bedit_files \
     {"Edit the selected files"}
  set_tooltips .workdir.bottom.buttons.funcs.bview_files \
     {"View the selected files"}
  set_tooltips .workdir.bottom.buttons.funcs.bdelete_file \
     {"Delete the selected files from the current directory"}
  set_tooltips .workdir.bottom.buttons.funcs.bmkdir \
     {"Make a new directory"}
  set_tooltips .workdir.bottom.buttons.funcs.brefresh \
     {"Re-read the current directory"}
  set_tooltips .workdir.bottom.buttons.cvsfuncs.bconflict \
     {"Merge Conflicts using TkDiff"}
  set_tooltips .workdir.bottom.buttons.cvsfuncs.blogfile \
     {"Revision Log and Branch Diagram of the selected files"}
  set_tooltips .workdir.bottom.buttons.cvsfuncs.bjoin \
     {"Directory Branch Diagram and Merge Tool"}
  set_tooltips .workdir.bottom.buttons.cvsfuncs.badd_files \
     {"Add the selected files to the repository"}
  set_tooltips .workdir.bottom.buttons.cvsfuncs.btag \
     {"Tag the selected files"}
  set_tooltips .workdir.bottom.buttons.cvsfuncs.bbranchtag \
     {"Branch the selected files"}
  set_tooltips .workdir.bottom.buttons.cvsfuncs.bremove \
     {"Remove the selected files from the repository"}
  set_tooltips .workdir.bottom.buttons.cvsfuncs.bcheckin \
     {"Check in (commit) the selected files to the repository"}
  set_tooltips .workdir.bottom.buttons.cvsfuncs.bcheckdir \
     {"Check the status of the directory"}
  set_tooltips .workdir.bottom.buttons.cvsfuncs.bupdate \
     {"Update (checkout, patch) the selected files from the repository"}
  set_tooltips .workdir.bottom.buttons.cvsfuncs.bupdateopts \
     {"Update with options (-A, -r, -f, -d, -kb)"}
  set_tooltips .workdir.bottom.buttons.cvsfuncs.bannotate \
     {"View revision where each line was modified"}
  set_tooltips .workdir.bottom.buttons.cvsfuncs.bcvsedit_files \
     {"Set the Edit flag on the selected files"}
  set_tooltips .workdir.bottom.buttons.cvsfuncs.bunedit_files \
     {"Reset the Edit flag on the selected files"}
  set_tooltips .workdir.bottom.buttons.cvsfuncs.bdiff \
     {"Compare the selected files with the repository version"}
  set_tooltips .workdir.top.bmodbrowse \
     {"Open the Module Browser"}
  set_tooltips .workdir.close \
     {"Close the Working Directory Browser"}


  frame .workdir.main
  pack .workdir.main -side bottom -fill both -expand 1 -fill both

  if {! [winfo ismapped .workdir]} {
    wm deiconify .workdir
  }

  change_dir "[pwd]"
  gen_log:log T "LEAVE"
}

proc workdir_images {} {
  global cvscfg

  image create photo arr_up \
    -format gif -file [file join $cvscfg(bitmapdir) arrow_up.gif]
  image create photo arr_dn \
    -format gif -file [file join $cvscfg(bitmapdir) arrow_dn.gif]
  image create photo arh_up \
    -format gif -file [file join $cvscfg(bitmapdir) arrow_hl_up.gif]
  image create photo arh_dn \
    -format gif -file [file join $cvscfg(bitmapdir) arrow_hl_dn.gif]
  image create photo updir \
    -format gif -file [file join $cvscfg(bitmapdir) updir.gif]
  image create photo Check \
    -format gif -file [file join $cvscfg(bitmapdir) check.gif]
  image create photo Fileview \
    -format gif -file [file join $cvscfg(bitmapdir) fileview.gif]
  image create photo Fileedit \
    -format gif -file [file join $cvscfg(bitmapdir) fileedit.gif]
  image create photo Annotate \
    -format gif -file [file join $cvscfg(bitmapdir) annotate.gif]
  image create photo Delete \
    -format gif -file [file join $cvscfg(bitmapdir) delete.gif]
  image create photo Dir_new \
    -format gif -file [file join $cvscfg(bitmapdir) dir_new.gif]
  image create photo Refresh \
    -format gif -file [file join $cvscfg(bitmapdir) loop-glasses.gif]
  image create photo Branches \
    -format gif -file [file join $cvscfg(bitmapdir) branch.gif]
  image create photo DirBranches \
    -format gif -file [file join $cvscfg(bitmapdir) dirbranch.gif]
  image create photo Add \
    -format gif -file [file join $cvscfg(bitmapdir) add.gif]
  image create photo Remove \
    -format gif -file [file join $cvscfg(bitmapdir) remove.gif]
  image create photo Diff \
    -format gif -file [file join $cvscfg(bitmapdir) diff.gif]
  image create photo Checkin \
    -format gif -file [file join $cvscfg(bitmapdir) checkin.gif]
  image create photo Edit \
    -format gif -file [file join $cvscfg(bitmapdir) edit.gif]
  image create photo Unedit \
    -format gif -file [file join $cvscfg(bitmapdir) unedit.gif]
  image create photo Modules \
    -format gif -file [file join $cvscfg(bitmapdir) modbrowse.gif]

  image create photo Tags \
    -format gif -file [file join $cvscfg(bitmapdir) tags.gif]
  image create photo Mergebranch \
    -format gif -file [file join $cvscfg(bitmapdir) merge.gif]
  image create photo Mergediff \
    -format gif -file [file join $cvscfg(bitmapdir) merge_changes.gif]
  image create photo Conflict \
    -format gif -file [file join $cvscfg(bitmapdir) conflict.gif]

  image create photo Man \
    -format gif -file [file join $cvscfg(bitmapdir) man.gif]
}

proc workdir_menus {} {
  global cvscfg
  global cvsmenu
  global usermenu
  global execmenu
  global bookmarks

  gen_log:log T "ENTER"
  set startdir "[pwd]"

  .workdir configure -menu .workdir.menubar
  menu .workdir.menubar

  #
  # Create the Menu bar
  #
  .workdir.menubar add cascade -label "File" -menu .workdir.menubar.file -underline 0
  menu .workdir.menubar.file
  .workdir.menubar add cascade -label "CVS" -menu .workdir.menubar.cvs -underline 0
  menu .workdir.menubar.cvs
  .workdir.menubar add cascade -label "Reports" -menu .workdir.menubar.reports -underline 0
  menu .workdir.menubar.reports
  .workdir.menubar add cascade -label "Options" -menu .workdir.menubar.options -underline 0
  menu .workdir.menubar.options

  if { [info exists cvsmenu] || \
       [info exists usermenu] || \
       [info exists execmenu]} {
    .workdir.menubar add cascade -label "User Defined" -menu .workdir.menubar.user -underline 0
    menu .workdir.menubar.user
    gen_log:log T "Adding user defined menu"
  }
  .workdir.menubar add cascade -label "Go" -menu .workdir.menubar.goto -underline 0
  menu .workdir.menubar.goto

  menu_std_help .workdir.menubar

  #
  # Create the Menus
  #
  .workdir.menubar.file add command -label "Open" -underline 0 \
     -command { workdir_edit_file [workdir_list_files] }
  .workdir.menubar.file add command -label "Print" -underline 0 \
     -command { workdir_print_file  [workdir_list_files ] }
  .workdir.menubar.file add separator
  .workdir.menubar.file add command -label "Join (Merge) Directory" -underline 0 \
     -command { cvs_joincanvas }
  .workdir.menubar.file add command -label "Browse Modules" -underline 0 \
     -command modbrowse_run
  .workdir.menubar.file add command -label "Cleanup" -underline 4 \
     -command workdir_cleanup
  .workdir.menubar.file add separator
  .workdir.menubar.file add command -label "Shell window" -underline 0 \
     -command {eval exec $cvscfg(shell) >& $cvscfg(null) &}
  .workdir.menubar.file add separator
  .workdir.menubar.file add command -label Exit -underline 1 \
     -command { exit_cleanup 1 }

  .workdir.menubar.cvs add command -label "Update" -underline 0 \
     -command { \
        cvs_update {BASE} {Normal} {Remove} {No} { } [workdir_list_files] }
  .workdir.menubar.cvs add command -label "Update with Options" -underline 7 \
     -command update_run
  .workdir.menubar.cvs add command -label "Commit" -underline 0 \
     -command commit_run
  .workdir.menubar.cvs add command -label "Add Files" -underline 0 \
     -command { add_dialog [workdir_list_files] }
  .workdir.menubar.cvs add command -label "Add Recursively" \
     -command { addir_dialog [workdir_list_files] }
  .workdir.menubar.cvs add command -label "Remove Files" -underline 0 \
     -command { subtract_dialog [workdir_list_files] }
  .workdir.menubar.cvs add command -label "Remove Recursively" \
     -command { subtractdir_dialog [workdir_list_files] }
  .workdir.menubar.cvs add command -label "Set Edit Flag (Edit)" -underline 15 \
     -command { cvs_edit [workdir_list_files] }
  .workdir.menubar.cvs add command -label "Reset Edit Flag (Unedit)" -underline 11 \
     -command { cvs_unedit [workdir_list_files] }
  .workdir.menubar.cvs add command -label "Tag Files" -underline 0 \
     -command { file_tag_dialog "no" }
  .workdir.menubar.cvs add command -label "Revision Tree" \
     -command { cvs_logcanvas [pwd] [workdir_list_files] }
  .workdir.menubar.cvs add command -label "Resolve Conflicts" \
     -command { cvs_merge_conflict [workdir_list_files] }
  .workdir.menubar.cvs add command -label "Release" \
     -command { release_dialog [workdir_list_files] }
  .workdir.menubar.cvs add command -label "Import WD into Repository" -underline 0 \
     -command import_run

  .workdir.menubar.reports add command -label "Check Directory" -underline 0 \
     -command { cvs_check [workdir_list_files] }
  .workdir.menubar.reports add command -label "CVS status" -underline 4 \
     -command { cvs_status [workdir_list_files] }
  .workdir.menubar.reports add command -label "Sticky status" -underline 4 \
     -command { cvs_tag_status [workdir_list_files] }
  .workdir.menubar.reports add command -label "CVS diff" -underline 4 \
     -command { cvs_diff [workdir_list_files] }
  .workdir.menubar.reports add command -label "CVS log" -underline 4 \
     -command { cvs_log [workdir_list_files] }
  .workdir.menubar.reports add command -label "CVS annotate" -underline 4 \
     -command { cvs_annotate $current_tagname [workdir_list_files] }

  set selcolor [option get .workdir selectColor selectColor]
  .workdir.menubar.options add checkbutton -label "Show hidden files" \
     -variable cvscfg(allfiles) -onvalue true -offvalue false \
     -selectcolor $selcolor -command setup_dir
  .workdir.menubar.options add checkbutton -label "Automatic directory status" \
     -variable cvscfg(auto_status) -onvalue true -offvalue false \
     -selectcolor $selcolor
  .workdir.menubar.options add checkbutton -label "Confirmation Dialogs" \
     -variable cvscfg(confirm_prompt) -onvalue true -offvalue false \
     -selectcolor $selcolor
  .workdir.menubar.options add separator
  .workdir.menubar.options add checkbutton -label "Editors Column" \
     -variable cvscfg(econtrol) -onvalue true -offvalue false \
     -selectcolor $selcolor \
     -command { if {($incvs || $inrcs) && $cvscfg(econtrol)} {
                  DirCanvas:map_column .workdir.main editcol
                } else {
                  pack forget .workdir.main.editcol
                }
              }
  .workdir.menubar.options add checkbutton -label "Status Column" \
     -variable cvscfg(showstatcol) -onvalue true -offvalue false \
     -selectcolor $selcolor \
     -command { if {($incvs || $inrcs) && $cvscfg(showstatcol)} {
                  DirCanvas:map_column .workdir.main statcol
                } else {
                  pack forget .workdir.main.statcol
                }
              }
  .workdir.menubar.options add checkbutton -label "Date Column" \
     -variable cvscfg(showdatecol) -onvalue true -offvalue false \
     -selectcolor $selcolor \
     -command { if {$cvscfg(showdatecol)} {
                  DirCanvas:map_column .workdir.main datecol
                } else {
                  pack forget .workdir.main.datecol
                }
              }
  .workdir.menubar.options add separator
  .workdir.menubar.options add checkbutton -label "Report->Check Shows Unknown Files" \
     -variable cvscfg(status_filter) -onvalue false -offvalue true \
     -selectcolor $selcolor
  .workdir.menubar.options add checkbutton -label "Report->Check is Recursive" \
     -variable cvscfg(checkrecursive) -onvalue {} -offvalue -l \
     -selectcolor $selcolor
  .workdir.menubar.options add checkbutton -label "Report->Status is Recursive" \
     -variable cvscfg(recurse) -onvalue true -offvalue false \
     -selectcolor $selcolor
  .workdir.menubar.options add cascade -label "CVS Status Detail" \
     -menu .workdir.menubar.options.report_detail
  .workdir.menubar.options add cascade -label "CVS Log Detail" \
     -menu .workdir.menubar.options.logfile_detail
  .workdir.menubar.options add separator
  .workdir.menubar.options add checkbutton -label "Tracing On/Off" \
     -variable cvscfg(logging) -onvalue true -offvalue false \
     -selectcolor $selcolor -command log_toggle
  .workdir.menubar.options add cascade -label "Trace Level" \
     -menu .workdir.menubar.options.loglevel
  .workdir.menubar.options add separator
  .workdir.menubar.options add command -label "Save Options" -underline 0 \
     -command save_options

  menu .workdir.menubar.options.loglevel
  .workdir.menubar.options.loglevel add checkbutton -label "CVS commands (C)" \
     -variable logclass(C) -onvalue "C" -offvalue "" \
     -selectcolor $selcolor -command gen_log:changeclass
  .workdir.menubar.options.loglevel add checkbutton -label "CVS stderr (E)" \
     -variable logclass(E) -onvalue "E" -offvalue "" \
     -selectcolor $selcolor -command gen_log:changeclass
  .workdir.menubar.options.loglevel add checkbutton -label "File creation/deletion (F)"\
     -variable logclass(F) -onvalue "F" -offvalue "" \
     -selectcolor $selcolor -command gen_log:changeclass
  .workdir.menubar.options.loglevel add checkbutton -label "Function entry/exit (T)" \
     -variable logclass(T) -onvalue "T" -offvalue "" \
     -selectcolor $selcolor -command gen_log:changeclass
  .workdir.menubar.options.loglevel add checkbutton -label "Debugging (D)" \
     -variable logclass(D) -onvalue "D" -offvalue "" \
     -selectcolor $selcolor -command gen_log:changeclass

  menu .workdir.menubar.options.report_detail
  .workdir.menubar.options.report_detail add radiobutton -label "Verbose" \
     -variable cvscfg(rdetail) -value "verbose" -selectcolor $selcolor
  .workdir.menubar.options.report_detail add radiobutton -label "Summary" \
     -variable cvscfg(rdetail) -value "summary" -selectcolor $selcolor
  .workdir.menubar.options.report_detail add radiobutton -label "Terse" \
     -variable cvscfg(rdetail) -value "terse" -selectcolor $selcolor

  menu .workdir.menubar.options.logfile_detail
  .workdir.menubar.options.logfile_detail add radiobutton -label "Latest" \
     -variable cvscfg(ldetail) -value "latest" -selectcolor $selcolor
  .workdir.menubar.options.logfile_detail add radiobutton -label "Summary" \
     -variable cvscfg(ldetail) -value "summary" -selectcolor $selcolor
  .workdir.menubar.options.logfile_detail add radiobutton -label "Verbose" \
     -variable cvscfg(ldetail) -value "verbose" -selectcolor $selcolor

  .workdir.menubar.goto add command -label "Go Home" \
     -command {change_dir $cvscfg(home)}
     .workdir.menubar.goto add command -label "Add Bookmark" \
         -command add_bookmark
      .workdir.menubar.goto add command -label "Delete Bookmark" \
         -command delete_bookmark_dialog
      .workdir.menubar.goto add separator
      foreach mark [array names bookmarks] {
        .workdir.menubar.goto add command -label "$mark" \
           -command "change_dir \"$mark\""
      }


  #
  # Add user commands to the menu.
  #
  if {[info exists cvsmenu]} {
    foreach item [array names cvsmenu] {
      .workdir.menubar.user add command -label $item \
         -command "eval cvs_usercmd $cvsmenu($item) \[workdir_list_files\]"
    }
  }
  if {[info exists usermenu]} {
    .workdir.menubar.user add separator
    foreach item [array names usermenu] {
      .workdir.menubar.user add command -label $item \
         -command "eval cvs_catchcmd $usermenu($item) \[workdir_list_files\]"
    }
  }
  if {[info exists execmenu]} {
    .workdir.menubar.user add separator
    foreach item [array names execmenu] {
      .workdir.menubar.user add command -label $item \
         -command "eval cvs_execcmd $execmenu($item) \[workdir_list_files\]"
    }
  }
  gen_log:log T "LEAVE"
}

proc menu_std_help { w } {
  $w add cascade -label "Help" -menu $w.help -underline 0
  menu $w.help
  $w.help add command -label "About TkCVS" -underline 0 \
     -command aboutbox
  $w.help add command -label "About CVS" -underline 6 \
     -command cvs_version
  $w.help add command -label "About Wish" -underline 6 \
     -command wish_version
  $w.help add separator
  $w.help add command -label "Current Directory Display" \
     -command current_directory
  $w.help add command -label "Log Browser" \
     -command log_browser
  $w.help add command -label "Merge Tool" \
     -command directory_branch_viewer
  $w.help add separator
  $w.help add command -label "Module Browser" \
     -command module_browser
  $w.help add command -label "Importing New Modules" \
     -command importing_new_modules
  $w.help add command -label "Importing To An Existing Module" \
     -command importing_to_existing_module
  $w.help add command -label "Vendor Merge" \
     -command vendor_merge
  $w.help add separator
  $w.help add command -label "Configuration Files" \
     -command configuration_files
  $w.help add command -label "Environment Variables" \
     -command environment_variables
  $w.help add command -label "Command Line Options" \
     -command cli_options
  $w.help add command -label "User Defined Menu" \
     -command user_defined_menu
  $w.help add command -label "CVS modules File" \
     -command cvs_modules_file
}

proc workdir_list_files {} {
  global cvscfg
  global cvsglb

  gen_log:log T "ENTER (cvsglb(current_selection) = $cvsglb(current_selection))"

  for {set i 0} {$i < [llength $cvsglb(current_selection)]} {incr i} {
    set item [lindex $cvsglb(current_selection) $i]
    regsub {^no file } $item "" item
    # regsub here causes file isfile to return 0.  You have to do it in each
    # proc, just before the cvs command, after file tests have been done.
    #regsub -all {\$} $item {\$} item
    set cvsglb(current_selection) [lreplace $cvsglb(current_selection) $i $i $item]
  }
  gen_log:log T "LEAVE -- ($cvsglb(current_selection))"
  return $cvsglb(current_selection)
}

proc workdir_edit_command {file} {
  global cvscfg

  gen_log:log T "ENTER ($file)"
  if {[info exists cvscfg(editors)]} {
    foreach {editor pattern} $cvscfg(editors) {
      if {[string match $pattern $file]} {
        return "$editor \"$file\""
      }
    }
  }
  return "\"$cvscfg(editor)\" $cvscfg(editorargs) \"$file\""
}

proc workdir_newdir {file} {
  global cvscfg

  gen_log:log T "ENTER ($file)"

  file mkdir $file

  if {$cvscfg(auto_status)} {
    setup_dir
  }

  gen_log:log T "LEAVE"
}

proc workdir_edit_file {args} {
  global cvscfg
  global cwd

  gen_log:log T "ENTER ($args)"

  set filelist [join $args]
  if {$filelist == ""} {
    file_input_and_do "Edit File" workdir_edit_file
    return
  }

  gen_log:log D "$filelist"
  foreach file $filelist {
    if {[file isdirectory $file]} {
      change_dir "$file"
    } else {
      if {![file exists $file] || [file isfile $file]} {
        # If the file doesn't exist it's tempting to touch the file and
        # trigger a reread, but is an empty file of this type valid?
        regsub -all {\$} $file {\$} file
        set commandline "[workdir_edit_command $file] >& $cvscfg(null) &"
        gen_log:log C "$commandline"
        eval "exec $commandline"
      } else {
        cvsfail "$file is not a plain file"
      }
    }
  }
  gen_log:log T "LEAVE"
}

proc workdir_view_file {args} {
  global cvscfg
  global cwd

  gen_log:log T "ENTER ($args)"

  set filelist [join $args]
  if {$filelist == ""} {
    cvsfail "Please select some files to view first!"
    return
  }

  gen_log:log D "$filelist"
  foreach file $filelist {
    set filelog ""
    if {[file isfile $file]} {
      #regsub -all {\$} $file {\$} file
      gen_log:log F "OPEN $file"
      set f [open $file]
      while { [eof $f] == 0 } {
        append filelog [gets $f]
        append filelog "\n"
      }
      view_output::new "$file" $filelog
    } else {
      cvsfail "$file is not a plain file"
    }
  }
  gen_log:log T "LEAVE"
}

# Let the user mark directories they visit often
proc add_bookmark { } {
  variable w
  global bookmarks

  set dir [pwd]
  regsub -all {\$} $dir {\$} dir
  set bookmarks($dir) t
  .workdir.menubar.goto add command -label "$dir" \
     -command "change_dir \"$dir\""
}

# A listbox to choose a bookmark to delete
proc delete_bookmark_dialog { } {
   variable w
   global cvscfg
   global bookmarks

   set maxlbl 0
   foreach mark [array names bookmarks] {
     set len [string length $mark]
     if {$len > $maxlbl} {
        set maxlbl $len
     }
   }

   set mname .workdir.bookmarkedit
   toplevel $mname
   grab set $mname
   wm title $mname "Delete Bookmarks"
   listbox $mname.lbx -selectmode multiple \
     -font $cvscfg(listboxfont) -width $maxlbl
   pack $mname.lbx -ipadx 10 -ipady 10 -expand y -fill both
   foreach mark [array names bookmarks] {
     $mname.lbx insert end $mark
   }
   frame $mname.buttons
   pack $mname.buttons -side top -fill x
   button $mname.delete -text "Delete" \
     -command [namespace code "delete_bookmark $mname"]
        
   button $mname.close -text "Done" \
     -command [namespace code "
       grab release $mname
       destroy $mname
       exit_cleanup 0"]
   pack $mname.delete $mname.close -in $mname.buttons \
     -side right -ipadx 2 -ipady 2 -padx 4 -pady 4 \
     -expand y
}

# Do the actual deletion of the bookmark
proc delete_bookmark {mname} {
  global bookmarks
  variable w

  set items [$mname.lbx curselection]
  foreach item $items {
    set itemstring [$mname.lbx get $item]
    #gen_log:log D "selection is \"$itemstring\""
    unset bookmarks($itemstring)
    $mname.lbx delete $item
    .workdir.menubar.goto delete "$itemstring"
  }
}

proc change_dir {new_dir} {
  global cwd

  gen_log:log T "ENTER ($new_dir)"
  if {![file exists $new_dir]} {
    set cwd [pwd]
    cvsfail "Directory $new_dir doesn\'t exist!"
    return
  }
  set cwd $new_dir
  setup_dir

  gen_log:log T "LEAVE"
}

proc setup_dir { } {
  #
  # Call this when entering a directory.  It puts all of the file names
  # in the listbox, and reads the CVS or directory.
  #
  global cwd
  global module_dir
  global incvs
  global inrcs
  global cvscfg
  global current_tagname
  global cvsglb

  gen_log:log T "ENTER"

  if { ! [winfo exists .workdir.main] } {
    workdir_setup
    return
  } else {
    DirCanvas:deltree .workdir.main
  }

  if {![file isdirectory $cwd]} {
    gen_log:log D "$cwd is not a directory"
    gen_log:log T "LEAVE -- $cwd is not a directory"
    return
  }

  cd $cwd
  gen_log:log F "CD [pwd]"

  set module_dir ""
  set current_tagname ""
  ::picklist::used directory [pwd]

  foreach {incvs inrcs module_dir} [cvsroot_check [pwd]] { break }

  set bstate [expr {$incvs ? {normal} : {disabled}}]
  foreach widget [grid slaves .workdir.bottom.buttons.cvsfuncs ] {
    $widget config -state $bstate
  }
  if {$inrcs} {
    .workdir.bottom.buttons.cvsfuncs.bdiff config -state normal
    .workdir.bottom.buttons.cvsfuncs.blogfile config -state normal \
      -command { rcs_filelog [workdir_list_files] }
  } else {
    .workdir.bottom.buttons.cvsfuncs.blogfile config \
      -command { cvs_logcanvas [pwd] [workdir_list_files] }
  }

  DirCanvas:create .workdir.main \
    -relief flat -bd 0 -highlightthickness 0 \
    -width 100 -height 350

  set cvsglb(current_selection) {}

  set cvscfg(ignore_file_filter) $cvsglb(default_ignore_filter)

  if { [ file exists ".cvsignore" ] } {
    set fileId [ open ".cvsignore" "r" ]
    while { [ eof $fileId ] == 0 } {
      gets $fileId line
      append cvscfg(ignore_file_filter) " $line"
    }
    close $fileId
  }

  set filelist [ getFiles ]
  directory_list $filelist

  gen_log:log T "LEAVE"
}

proc directory_list { filenames } {
  global module_dir
  global incvs
  global inrcs
  global cvs
  global cwd
  global cvscfg
  global cvsglb
  global cmd
  global Filelist

  gen_log:log T "ENTER ($filenames)"

  if {[info exists Filelist]} {
    unset Filelist
  }

  busy_start .workdir.main

  #gen_log:log F "processing files in the local directory"
  set cwd [pwd]
  set my_cwd $cwd

  # If we have commands running they were for a different directory
  # and won't be needed now. (i.e. this is a recursive invocation
  # triggered by a button click)
  if {[info exists cmd(cvs_status)]} {
    catch {$cmd(cvs_status)\::abort}
    unset cmd(cvs_status)
  }
  if {[info exists cmd(cvs_editors)]} {
    catch {$cmd(cvs_editors)\::abort}
    unset cmd(cvs_editors)
  }

  if {$incvs} {
    set cmd(cvs_status) [exec::new "$cvs -n -q status -l"]
    set status_lines [split [$cmd(cvs_status)\::output] "\n"]
    if {$cvscfg(econtrol)} {
      set cmd(cvs_editors) [exec::new "$cvs -n -q editors -l"]
      set editors_lines [split [$cmd(cvs_editors)\::output] "\n"]
    }
  }

  # Select from those files only the ones we want (e.g., no CVS dirs)
  foreach i $filenames {
    if { $i == "."  || $i == ".."} {
      gen_log:log D "SKIPPING $i"
      continue
    }
    if {[file isdirectory $i]} {
      if {[isCmDirectory $i]} {
        # Read the CVS files but don't list the directory
        if {$i == "CVS"} {
          read_cvs_dir [file join $cwd $i]
          continue
        }
        if {$i == "RCS"} {
          continue
        }
      }
      if {[file exists [file join $i "CVS"]]} {
        set Filelist($i:status) "<directory:CVS>"
      } elseif {[file exists [file join $i "RCS"]]} {
        set Filelist($i:status) "<directory:RCS>"
      } else {
        set Filelist($i:status) "<directory>"
      }
    } else {
      if {$incvs} {
        set Filelist($i:status) "?"
      } else {
        set Filelist($i:status) "<file>"
      }
    }
    set Filelist($i:wrev) ""
    set Filelist($i:stickytag) ""
    set Filelist($i:option) ""
    if {[string match "<direc*" $Filelist($i:status)]} {
      set Filelist($i:date) ""
    } else {
      # Prepending ./ to the filename prevents tilde expansion
      catch {set Filelist($i:date) \
         [clock format [file mtime ./$i] -format $cvscfg(dateformat)]}
    }
  }

  gen_log:log D "inrcs = $inrcs"
  if {[info exists cmd(cvs_status)]} {
    #gen_log:log F "processing files that CVS knows about"
    # gets cvs status in current directory only, pulling out lines that include
    # Status: or Sticky Tag:, putting each file's info (name, status, and tag)
    # into an array.


    # If the cwd has changed under us we are bogus
    if {$cwd != $my_cwd} {
      gen_log:log T "LEAVE (Directory changed)"
      return
    }

    unset cmd(cvs_status)
    foreach logline $status_lines {
      if {[string match "File:*" $logline]} {
        regsub -all {\t+} $logline "\t" logline
        set line [split [string trim $logline] "\t"]
        gen_log:log D "$line"
        # Should be able to do these regsubs in one expression
        regsub {File: } [lindex $line 0] "" filename
        regsub {\s*$} $filename "" filename
        if {[string match "no file *" $filename]} {
          regsub {^no file } $filename "" filename
        }
        regsub {Status: } [lindex $line 1] "" status
        set Filelist($filename:status) $status
        # Don't set editors to null because we'll use its presence
        # or absence to see if we need to re-read the repository when
        # we ask to map the editors column
        #set Filelist($filename:editors) ""
      } elseif {[string match "*Working revision:*" $logline]} {
        regsub -all {\t+} $logline "\t" logline
        set line [split [string trim $logline] "\t"]
        gen_log:log D "$line"
        set revision [lindex $line 1]
        regsub {New .*} $revision "New" revision
        set date [lindex $line 2]
        # The date field is not supplied to remote clients.
        if {$date == "" || [string match "New *" $date ] || \
            [string match "Result *" $date]} {
          ; # Leave as is
        } else {
          set juliandate [clock scan $date -gmt yes]
          set date [clock format $juliandate -format $cvscfg(dateformat)]
          set Filelist($filename:date) $date
        }
        set Filelist($filename:wrev) $revision
        set Filelist($filename:status) $status
      } elseif {[string match "*Sticky Tag:*" $logline]} {
        regsub -all {\t+} $logline "\t" logline
        set line [split [string trim $logline] "\t"]
        gen_log:log D "$line"
        set tagline [lindex $line 1]
        set t0 [lindex $tagline 0]
        set t1 [lrange $tagline 1 end]
        set stickytag ""
        if { $t0 == "(none)" } {
          set stickytag " on trunk"
        } elseif {[string match "(branch:*" $t1 ]} {
          regsub {\(branch: (.*)\)} $t1 {\1} t1
          set stickytag " on $t0  branch"
        } elseif {[string match "(revision:*" $t1 ]} {
          set stickytag " $t0"
        }
        set Filelist($filename:stickytag) "$revision $stickytag"
      } elseif {[string match "*Sticky Options:*" $logline]} {
        regsub -all {\t+} $logline "\t" logline
        set line [split [string trim $logline] "\t"]
        gen_log:log D "$line"
        set option [lindex $line 1]
        set Filelist($filename:option) $option
      }
    }
  }

  if {[info exists cmd(cvs_editors)]} {
    set filename {}
    # If the cwd has changed under us we are bogus
    if {$cwd != $my_cwd} {
      gen_log:log T "LEAVE (Directory changed)"
      return
    }
    unset cmd(cvs_editors)
    foreach logline $editors_lines {
      set line [split $logline "\t"]
      gen_log:log D "$line"
      set ell [llength $line]
      # ? files will show up in cvs editors output under certain conditions
      if {$ell < 5} {
        continue
      }
      #if there is no filename, then this is a continuation line
      set f [lindex $line 0]
      if {$f == {}} {
        append editors ",[lindex $line 1]"
      } else {
        if {$filename != {}} {
          set Filelist($filename:editors) $editors
        }
        set filename $f
        set editors [lindex $line 1]
      }
      gen_log:log D " $filename   $editors"
    }
    if {$filename != {}} {
      set Filelist($filename:editors) $editors
    }
  }

  if {$inrcs} {
      set rcsfiles [glob -nocomplain -- RCS/* RCS/.??* *,v .??*,v]
      set command "rlog -h $rcsfiles"
      gen_log:log C "$command"
      set ret [catch {eval "exec $command"} raw_rcs_log]
      #gen_log:log D "$raw_rcs_log"

      set rlog_lines [split $raw_rcs_log "\n"]
      set logstate "working"
      set lockers ""
      foreach rlogline $rlog_lines {
        gen_log:log D "$rlogline"
        gen_log:log D "  logstate $logstate"
        # Found one!
        switch -exact -- $logstate {
          "working" {
            if {[string match "Working file:*" $rlogline]} {
              regsub {Working file: } $rlogline "" filename
              regsub {\s*$} $filename "" filename
              gen_log:log D "RCS file $filename"
              set Filelist($filename:wrev) ""
              set Filelist($filename:stickytag) ""
              set Filelist($filename:option) ""
              if {[file exists $filename]} {
                set Filelist($filename:status) "RCS"
                # Do rcsdiff to see if it's changed
                set command "rcsdiff -q \"$filename\" > $cvscfg(null)"
                gen_log:log C "$command"
                set ret [catch {eval "exec $command"}]
                if {$ret == 1} {
                  set Filelist($filename:status) "RCS Modified"
                }
              } else {
                set Filelist($filename:status) "RCS Needs Checkout"
              }
              set who ""
              set lockers ""
              set logstate "head"
              continue
            }
          }
          "head" {
            if {[string match "head:*" $rlogline]} {
              regsub {head: } $rlogline "" revnum
              set Filelist($filename:wrev) "$revnum"
              set Filelist($filename:stickytag) "$revnum on trunk"
              gen_log:log D "  Rev \"$revnum\""
              set logstate "branch"
              continue
            }
          }
          "branch" {
            if {[string match "branch:*" $rlogline]} {
              regsub {branch: *} $rlogline "" revnum
              if {[string length $revnum] > 0} {
                set Filelist($filename:wrev) "$revnum"
                set Filelist($filename:stickytag) "$revnum on branch"
                gen_log:log D "  Branch rev \"$revnum\""
              }
              set logstate "locks"
              continue
            }
          }
          "locks" {
            if { [string index $rlogline 0] == "\t" } {
               set splitline [split $rlogline]
               gen_log:log F "\"[lindex $splitline 1]\""
               gen_log:log F "\"[lindex $splitline 2]\""
               set who [lindex $splitline 1]
               set who [string trimright $who ":"]
               gen_log:log F " who $who"
               append lockers ",$who"
               gen_log:log F " lockers $lockers"
            } else {
              if {[string match "access list:*" $rlogline]} {
                set lockers [string trimleft $lockers ","]
                set Filelist($filename:editors) $lockers
                # No more tags after this point
                set logstate "working"
                continue
              }
            }
          }  
        }
      }
  }

  gen_log:log D "Sending all files to the canvas"
  foreach i [array names Filelist *:status] {
    regsub {:status$} $i "" j
    # If it's locally removed or missing, it may not have
    # gotten a date especially on a remote client.
    if {! [info exists Filelist($j:date)]} {
      set Filelist($j:date) ""
    }
    DirCanvas:newitem .workdir.main "$j"
  }

  busy_done .workdir.main

  gen_log:log T "LEAVE"
}

proc read_cvs_dir {dirname} {
#
# Reads a CVS directory
#
  global module_dir
  global cvscfg
  global current_tagname

  gen_log:log T "ENTER ($dirname)"
  if {[file isdirectory $dirname]} {
    if {[file isfile [file join $dirname Repository]]} {
      gen_log:log F "OPEN CVS/Repository"
      set f [open [file join $dirname Repository] r]
      gets $f module_dir
      close $f
      gen_log:log D "  MODULE $module_dir"
      if {[file isfile [file join $dirname Root]]} {
        gen_log:log F "OPEN CVS/Root"
        set f [open [file join $dirname Root] r]
        gets $f cvscfg(cvsroot)
        close $f
        # On a PC, the cvsroot can be like C:\DosRepository.
        # This makes that workable.
        regsub -all {\\} $cvscfg(cvsroot) {\\\\} cvscfg(cvsroot)
        gen_log:log D " cvsroot: $cvscfg(cvsroot)"
      }
      if {[file isfile [file join $dirname Tag]]} {
        gen_log:log F "OPEN CVS/Tag"
        set f [open [file join $dirname Tag] r]
        gets $f current_tagname
        close $f
        # T = branch tag, N = non-branch, D = sticky date
        set current_tagname [string range $current_tagname 1 end]
        gen_log:log D "  BRANCH TAG $current_tagname"
      }
    } else {
      cvsfail "Repository file not found in $dirname"
    }
  } else {
    cvsfail "$dirname is not a directory"
  }
  gen_log:log T "LEAVE"
}

proc workdir_cleanup {} {
  global cvscfg

  gen_log:log T "ENTER"
  set rmitem ""
  set list [ split $cvscfg(clean_these) " " ]
  foreach pattern $list {
    gen_log:log D "pattern $pattern"
    if { $pattern != "" } {
      set items [lsort [glob -nocomplain $pattern]]
      gen_log:log D "$items"
      if {[llength $items] != 0} {
        append rmitem " [concat $items]"
      }
    }
  }

  if {$rmitem != ""} {
    if { [ are_you_sure "You are about to delete:\n" $rmitem] == 1 } {
      gen_log:log F "DELETE $rmitem"
      eval file delete -force -- $rmitem
    }
  } else {
    gen_log:log F "No files to delete"
    cvsok "Nothing matched $cvscfg(clean_these)"
    return
  }
  setup_dir
  gen_log:log T "LEAVE"
}

proc workdir_delete_file {args} {
  global cvscfg

  gen_log:log T "ENTER ($args)"

  set filelist [join $args]
  if {$filelist == ""} {
    cvsfail "Please select some files to delete first!"
    return
  }

  if { [ are_you_sure "This will delete these files from your local, working directory:\n" $filelist ] == 1 } {
    gen_log:log F "DELETE $filelist"
    eval file delete -force -- $filelist
    setup_dir
  }
  gen_log:log T "LEAVE"
}

proc are_you_sure {mess args} {
#
# General posting message
#
  global cvscfg

  gen_log:log T "ENTER ($mess $args)"

  set filelist [join $args]
  if {$cvscfg(confirm_prompt)} {
    append mess "\n"
    set indent "      "

    foreach item $filelist {
      if { $item != {} } {
        append mess " $indent"
        append mess " $item\n"
      }
    }
    append mess "\nAre you sure?"
    if {[cvsconfirm $mess] == 1} {
      gen_log:log T "LEAVE 0"
      return 0
    }
  }
  gen_log:log T "LEAVE 1"
  return 1
}

proc busy_start {w} {

  foreach widget [winfo children $w] {
    catch {$widget config -cursor watch}
  }
  update idletasks
}

proc busy_done {w} {

  foreach widget [winfo children $w] {
    catch {$widget config -cursor ""}
  }
}

proc workdir_print_file {args} {
  global cvscfg

  gen_log:log T "ENTER ($args)"

  set filelist [join $args]
  if {$filelist == ""} {
    cvsfail "Please select some files to print first!"
    return
  }

  set mess "This will print these files:\n\n"
  foreach file $filelist {
    append mess "   $file\n"
  }
  append mess "\nUsing $cvscfg(print_cmd)\n"
  append mess "\nAre you sure?"
  if {[cvsconfirm $mess] == 0} {
    set final_result ""
    foreach file $filelist {
      gen_log:log C "$cvscfg(print_cmd) \"$file\""
      catch { eval exec $cvscfg(print_cmd) \"$file\" } file_result
      if { $file_result != "" } {
        set final_result "$final_result\n$file_result"
      }
    }
    if { $final_result != "" } {
      view_output::new "Print" $final_result
    }
  }
  gen_log:log T "LEAVE"
}

proc cvsroot_check { dir } {
  global module_dir

  gen_log:log T "ENTER ($dir)"

  foreach {incvs inrcs module_dir} {0 0 {}} { break }

  if {[file isfile [file join $dir CVS Root]]} {
    set incvs 1
  } else {
    set module_dir [file join $dir RCS]
    if {[file exists $module_dir]} {
      set inrcs 1
    } elseif {[llength [glob -nocomplain -dir $dir *,v]] > 0} {
      set inrcs 1
      set module_dir $dir
    } else {
      set module_dir ""
    }
  }
  if {$inrcs} {
    # Make sure we have rcs, and bag this (silently) if we don't   
    set command "rcs -V"
    gen_log:log C "$command"
    set ret [catch {eval "exec $command"} raw_rcs_log]
    if {$ret} {
       gen_log:log D "$raw_rcs_log"
       set inrcs 0
    }
  }

  gen_log:log T "LEAVE ($incvs $inrcs $module_dir)"
  return [list $incvs $inrcs $module_dir]
}

proc nop {} {}

proc disabled {} {
  cvsok "Command disabled."
}

proc isCmDirectory { file } {
  switch -- $file  {
    "CVS"  -
    "RCS"  -
    "SCCS" { set value 1 }
    default { set value 0 }
  }
  return $value
}

# Get the files in the current working directory.  Use the file_filter
# values Add hidden files if desired by the user.  Sort them to match
# the ordering that will be returned by cvs commands (this matches the
# default ls ordering.).
proc getFiles { } {
  global cvscfg
  global cvsglb

  gen_log:log T "ENTER"
  set filelist ""

  # make sure the file filter is at least set to "*".
  if { $cvscfg(file_filter) == "" } {
    set cvscfg(file_filter) "*"
  }

  # get the initial file list, including hidden if requested
  if {$cvscfg(allfiles)} {
    # get hidden as well
    foreach item $cvscfg(file_filter) {
      set filelist [ concat [ glob -nocomplain .$item $item ] $filelist ]
    }
  } else {
    foreach item $cvscfg(file_filter) {
      set filelist [ concat [ glob -nocomplain $item ] $filelist ]
    }
  }
  #gen_log:log D "filelist ($filelist)"

  # ignore files if requested
  if { $cvscfg(ignore_file_filter) != "" } {
    foreach item $cvscfg(ignore_file_filter) {
      # for each pattern
      if { $item != "*" } {
        # if not "*"
        while { [set idx [lsearch $filelist $item]] != -1 } {
          # for each occurence, delete
          catch { set filelist [ lreplace $filelist $idx $idx ] }
        }
      }
    }
  }

  # make sure "." is always in the list for 'cd' purposes
  if { ( [ lsearch -exact $filelist "." ] == -1 ) } {
    set filelist [ concat "." $filelist ]
  }

  # make sure ".." is always in the list for 'cd' purposes
  if { ( [ lsearch -exact $filelist ".." ] == -1 ) } {
    set filelist [ concat ".." $filelist ]
  }

  # sort it
  set filelist [ lsort $filelist ]

  # if this directory is under CVS and CVS is not in the list, add it. Its
  # presence is needed for later processing
  if { ( [ file exists "CVS" ] ) &&
       ( [ lsearch -exact $filelist "CVS" ] == -1 ) } {
    #puts "********* added CVS"
    catch { set filelist [ concat "CVS" $filelist ] }
  }

  set cvscfg(ignore_file_filter) $cvsglb(default_ignore_filter)
  gen_log:log T "return ($filelist)"
  return $filelist
}

proc log_toggle { } {
  global cvscfg

  if {$cvscfg(logging)} {
    gen_log:init
  } else {
    gen_log:quit
  }
}

proc exit_cleanup { force } {
  global cvscfg

  # Count the number of toplevels that are currently interacting
  # with the user (i.e. exist and are not withdrawn)
  set wlist {}
  foreach w [winfo children .] {
    if {[wm state $w] != {withdrawn}} {
      lappend wlist $w
    }
  }

  if {$force == 0 && [llength $wlist] != 0 \
    && $wlist != {.trace} && $wlist != {.bgerrorTrace}} {
      return
  }

  # If toplevel windows exist ask them to close gracefully if possible
  foreach w $wlist {
    # Except .trace!
    if {$w != {.trace}} {
      catch {$w.close invoke}
    } else {
      # Invoking trace's close turns off logging. We don't want that,
      # but we do want to save its geometry.
      if {[winfo exists .trace]} {
        set cvscfg(tracgeom) [wm geometry .trace]
      }
    }
  }

  save_options
  set pid [pid]
  gen_log:log F "DELETE $cvscfg(tmpdir)/cvstmpdir.$pid"
  catch {file delete -force [file join $cvscfg(tmpdir) cvstmpdir.$pid]}
  exit
}

proc save_options { } {
#
# Save the options which are configurable from the GUI
#
  global cvscfg
  global logcfg
  global bookmarks

  gen_log:log T "ENTER"

  # There are two kinds of options we can set
  set BOOLopts { allfiles auto_status confirm_prompt econtrol \
                 showstatcol showdatecol auto_tag \
                 status_filter checkrecursive recurse logging }
  set STRGopts { file_filter ignore_file_filter clean_these \
                 printer rdetail ldetail log_classes lastdir \
                 workgeom modgeom loggeom tracgeom}

  # Plus the logcanvas options
  set LOGopts [concat [array names logcfg show_*] scale]

  # set this to current directory, so we'll add it to the menu next time
  if ([catch pwd]) {
    return
  }
  set cvscfg(lastdir) [pwd]

  # Save the list so we can keep track of what we've done
  set BOOLset $BOOLopts
  set STRGset $STRGopts
  set LOGset $LOGopts

  set optfile [file join $cvscfg(home) .tkcvs]
  set bakfile [file join $cvscfg(home) .tkcvs.bak]
  # Save the old .tkcvs file
  gen_log:log F "MOVE $optfile $bakfile"
  catch {file rename -force $optfile $bakfile}

  gen_log:log F "OPEN $optfile"
  if {[catch {set fo [open $optfile w]}]} {
    cvsfail "Cannot open $optfile for writing"
    return
  }
  gen_log:log F "OPEN $bakfile"

  if {! [catch {set fi [open $bakfile r]}]} {
    while { [eof $fi] == 0 } {
      gets $fi line
      set match 0
      if {[regexp {^#} $line]} {
        # Don't try to scan comments.
        #gen_log:log D "PASSING \"$line\""
        puts $fo "$line"
        continue
      } elseif {[string match "*set *bookmarks*" $line]} {
        # Discard old bookmarks
        continue
      } else {
        foreach opt $BOOLopts {
          if {! [info exists cvscfg($opt)]} { continue }
          if {[string match "*set *cvscfg($opt)*" $line]} {
            # Print it and remove it from the list
            gen_log:log D "REPLACING $line  w/ set cvscfg($opt) $cvscfg($opt)"
            puts $fo "set cvscfg($opt) $cvscfg($opt)"
            set idx [lsearch $BOOLset $opt]
            set BOOLset [lreplace $BOOLset $idx $idx]
            set match 1
            break
          }
        }
        foreach opt $STRGopts {
          if {! [info exists cvscfg($opt)]} { continue }
          if {[string match "*set *cvscfg($opt)*" $line]} {
            # Print it and remove it from the list
            gen_log:log D "REPLACING $line  w/ set cvscfg($opt) $cvscfg($opt)"
            puts $fo "set cvscfg($opt) \"$cvscfg($opt)\""
            set idx [lsearch $STRGset $opt]
            set STRGset [lreplace $STRGset $idx $idx]
            set match 1
            break
          }
        }
        foreach opt $LOGopts {
          if {! [info exists logcfg($opt)]} { continue }
          if {[string match "*set *logcfg($opt)*" $line]} {
            # Print it and remove it from the list
            gen_log:log D "REPLACING \"$line\"  w/ set logcfg($opt) \"$logcfg($opt)\""
            puts $fo "set logcfg($opt) \"$logcfg($opt)\""
            set idx [lsearch $LOGset $opt]
            set LOGset [lreplace $LOGset $idx $idx]
            set match 1
            break
          }
        }
        if {$match == 0} {
          # We didn't do a replacement
          gen_log:log D "PASSING \"$line\""
          # If we don't check this, we get an extra blank line every time
          # we save the file.  Messy.
          if {[eof $fi] == 1} { break }
          puts $fo "$line"
        }
      }
    }
    foreach mark [array names bookmarks] {
      gen_log:log D "Adding bookmark \"$mark\""
      puts $fo "set \"bookmarks($mark)\" t"
    }

    close $fi
  }

  # Print what's left over
  foreach opt $BOOLset {
    if {! [info exists cvscfg($opt)]} { continue }
    gen_log:log D "ADDING cvscfg($opt) $cvscfg($opt)"
    puts $fo "set cvscfg($opt) $cvscfg($opt)"
  }

  foreach opt $STRGset {
    if {! [info exists cvscfg($opt)]} { continue }
    gen_log:log D "ADDING cvscfg($opt) \"$cvscfg($opt)\""
    puts $fo "set cvscfg($opt) \"$cvscfg($opt)\""
  }

  foreach opt $LOGset {
    if {! [info exists logcfg($opt)]} { continue }
    gen_log:log D "ADDING logcfg($opt) \"$logcfg($opt)\""
    puts $fo "set logcfg($opt) \"$logcfg($opt)\""
  }

  close $fo
  ::picklist::save
  gen_log:log T "LEAVE"
}

