#!/usr/bin/env python3
import io
import os
import sys
from importlib import import_module
from importlib.util import module_from_spec, spec_from_file_location

from setuptools import setup
from setuptools.command.build_py import build_py
from setuptools.command.sdist import sdist


# The goal is to get the version without doing a regular/full import of
# stgit. Executing the stgit._version module will obtain the version
# without importing anything else from the stgit package.
def _get_version():
    spec = spec_from_file_location('version', os.path.join('stgit', '_version.py'))
    module = module_from_spec(spec)
    spec.loader.exec_module(module)
    return module.__version__


version = _get_version()


def _write_version(base_dir):
    _version_path = os.path.join(base_dir, 'stgit', '_version.py')
    try:
        # This could be a hard link, so try to delete it first.
        os.remove(_version_path)
    except OSError:
        os.makedirs(os.path.join(base_dir, 'stgit'), exist_ok=True)
    with open(_version_path, "w") as f:
        f.write("# Generated by setup.py.\n\n__version__ = '{}'\n".format(version))


class _stgit_build_py(build_py):
    def run(self):
        super().run()
        _write_version(self.build_lib)


class _stgit_sdist(sdist):
    def make_release_tree(self, base_dir, files):
        super().make_release_tree(base_dir, files)
        _write_version(base_dir)


# Override setuptools build_py and sdist commands to rewrite the
# _version.py file into those commands' destination trees.
# N.B. the _version.py in the worktree is never overwritten.
cmdclass = dict(sdist=_stgit_sdist, build_py=_stgit_build_py)


# Carefully perform code generation. First check if generated code
# exists. Only write-out generated code if the content is different.
# This accommodates several setup.py contexts, including:
#  - When running from an sdist where code is already generated
#  - During pip installation where the stgit package is not in sys.path
#  - In a git worktree where the generated code may change frequently
#  - When running from a read-only directory/filesystem
def _maybe_generate_code():
    base = os.path.abspath(os.path.dirname(__file__))
    paths = dict(
        cmds=os.path.join(base, 'stgit', 'commands', 'cmdlist.py'),
        bash=os.path.join(base, 'completion', 'stgit.bash'),
        fish=os.path.join(base, 'completion', 'stg.fish'),
    )

    existing_content = dict()
    for k, path in paths.items():
        if os.path.exists(path):
            with open(path, 'r') as f:
                existing_content[k] = f.read()
        else:
            existing_content[k] = None

    sys.path.insert(0, base)
    try:
        try:
            gen_funcs = dict(
                cmds=import_module('stgit.commands').write_cmdlist_py,
                bash=import_module('stgit.completion.bash').write_bash_completion,
                fish=import_module('stgit.completion.fish').write_fish_completion,
            )
        except ImportError:
            if all(existing_content.values()):
                return  # Okay, all generated content exists
            else:
                raise RuntimeError('Cannot perform code generation')
    finally:
        sys.path.pop(0)

    for k, gen_func in gen_funcs.items():
        with io.StringIO() as f:
            gen_func(f)
            new_content = f.getvalue()
        if existing_content[k] != new_content:
            with open(paths[k], 'w') as f:
                f.write(new_content)


_maybe_generate_code()


def _long_description():
    base = os.path.abspath(os.path.dirname(__file__))
    parts = []
    for fn in ['README.md', 'CHANGELOG.md']:
        with open(os.path.join(base, fn)) as f:
            parts.append(f.read())
    return '\n\n'.join(parts)


setup(
    name='stgit',
    version=version,
    license='GPLv2',
    author='Catalin Marinas',
    author_email='catalin.marinas@gmail.com',
    maintainer='Peter Grayson',
    maintainer_email='pete@jpgrayson.net',
    url='http://stacked-git.github.io',
    download_url='https://github.com/stacked-git/stgit.git',
    description='Stacked Git',
    long_description=_long_description(),
    long_description_content_type='text/markdown',
    python_requires='>=3.5',
    zip_safe=False,
    include_package_data=True,
    entry_points=dict(console_scripts=['stg = stgit.main:main']),
    cmdclass=cmdclass,
    packages=[
        'stgit',
        'stgit.commands',
        'stgit.completion',
        'stgit.lib',
        'stgit.lib.git',
    ],
    classifiers=[
        'Development Status :: 5 - Production/Stable',
        'Environment :: Console',
        'Intended Audience :: Developers',
        'License :: OSI Approved :: GNU General Public License v2 (GPLv2)',
        'Natural Language :: English',
        'Operating System :: OS Independent',
        'Programming Language :: Python',
        'Programming Language :: Python :: 3',
        'Programming Language :: Python :: 3.5',
        'Programming Language :: Python :: 3.6',
        'Programming Language :: Python :: 3.7',
        'Programming Language :: Python :: 3.8',
        'Programming Language :: Python :: 3.9',
        'Programming Language :: Python :: 3.10',
        'Programming Language :: Python :: Implementation :: CPython',
        'Programming Language :: Python :: Implementation :: PyPy',
        'Topic :: Software Development :: Version Control',
    ],
)
