###############################################################################
#                                                                             #
# Copyright (C) 2004-2014 Edward d'Auvergne                                   #
# Copyright (C) 2014 Troels E. Linnet                                         #
#                                                                             #
# This file is part of the program relax (http://www.nmr-relax.com).          #
#                                                                             #
# This program is free software: you can redistribute it and/or modify        #
# it under the terms of the GNU General Public License as published by        #
# the Free Software Foundation, either version 3 of the License, or           #
# (at your option) any later version.                                         #
#                                                                             #
# This program is distributed in the hope that it will be useful,             #
# but WITHOUT ANY WARRANTY; without even the implied warranty of              #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               #
# GNU General Public License for more details.                                #
#                                                                             #
# You should have received a copy of the GNU General Public License           #
# along with this program.  If not, see <http://www.gnu.org/licenses/>.       #
#                                                                             #
###############################################################################

# Module docstring.
"""The R1 and R2 exponential relaxation curve fitting optimisation functions."""

# Python module imports.
from numpy import array, float64, ndarray, nan_to_num

# relax module imports.
from dep_check import C_module_exp_fn
from specific_analyses.relax_fit.parameters import assemble_param_vector

# C modules.
if C_module_exp_fn:
    from target_functions.relax_fit import setup, func, dfunc, d2func, back_calc_I


def back_calc(spin=None, relax_time_id=None):
    """Back-calculation of peak intensity for the given relaxation time.

    @keyword spin:              The spin container.
    @type spin:                 SpinContainer instance
    @keyword relax_time_id:     The ID string for the desired relaxation time.
    @type relax_time_id:        str
    @return:                    The peak intensity for the desired relaxation time.
    @rtype:                     float
    """

    # Create the initial parameter vector.
    param_vector = assemble_param_vector(spin=spin)

    # The keys.
    keys = list(spin.peak_intensity.keys())

    # The peak intensities and times.
    values = []
    errors = []
    times = []
    for key in keys:
        values.append(spin.peak_intensity[key])
        errors.append(spin.peak_intensity_err[key])
        times.append(cdp.relax_times[key])

    # A fake scaling matrix in a diagonalised list form.
    scaling_list = []
    for i in range(len(param_vector)):
        scaling_list.append(1.0)

    # Initialise the relaxation fit functions.
    setup(num_params=len(spin.params), num_times=len(cdp.relax_times), values=values, sd=errors, relax_times=times, scaling_matrix=scaling_list)

    # Make a single function call.  This will cause back calculation and the data will be stored in the C module.
    func_wrapper(param_vector)

    # Get the data back.
    results = back_calc_I()

    # Return the correct peak height.
    return results[keys.index(relax_time_id)]


def func_wrapper(params):
    """Wrapper function for the C module, for converting numpy arrays.

    @param params:  The parameter array from the minimisation code.
    @type params:   numpy array
    @return:        The function value generated by the C module.
    @rtype:         float
    """

    # Convert if necessary.
    if isinstance(params, ndarray):
        params = params.tolist()

    # Call the C code.
    chi2 = func(params)

    # Return the chi2 value.
    return nan_to_num(chi2)


def dfunc_wrapper(params):
    """Wrapper function for the C module, for converting numpy arrays.

    @param params:  The parameter array from the minimisation code.
    @type params:   numpy array
    @return:        The gradient generated by the C module converted to numpy format.
    @rtype:         numpy float64 array
    """

    # Convert if necessary.
    if isinstance(params, ndarray):
        params = params.tolist()

    # Call the C code.
    dchi2 = dfunc(params)

    # Return the chi2 gradient as a numpy array.
    return array(dchi2, float64)


def d2func_wrapper(params):
    """Wrapper function for the C module, for converting numpy arrays.

    @param params:  The parameter array from the minimisation code.
    @type params:   numpy array
    @return:        The Hessian generated by the C module converted to numpy format.
    @rtype:         numpy float64 rank-2 array
    """

    # Convert if necessary.
    if isinstance(params, ndarray):
        params = params.tolist()

    # Call the C code.
    d2chi2 = d2func(params)

    # Return the chi2 Hessian as a numpy array.
    return array(d2chi2, float64)
