###############################################################################
#                                                                             #
# Copyright (C) 2003-2014 Edward d'Auvergne                                   #
# Copyright (C) 2014 Troels E. Linnet                                         #
#                                                                             #
# This file is part of the program relax (http://www.nmr-relax.com).          #
#                                                                             #
# This program is free software: you can redistribute it and/or modify        #
# it under the terms of the GNU General Public License as published by        #
# the Free Software Foundation, either version 3 of the License, or           #
# (at your option) any later version.                                         #
#                                                                             #
# This program is distributed in the hope that it will be useful,             #
# but WITHOUT ANY WARRANTY; without even the implied warranty of              #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               #
# GNU General Public License for more details.                                #
#                                                                             #
# You should have received a copy of the GNU General Public License           #
# along with this program.  If not, see <http://www.gnu.org/licenses/>.       #
#                                                                             #
###############################################################################

# Module docstring.
"""Module for generating OpenDX files."""


# relax module imports.
from lib.io import open_write_file


def write_config(file_prefix=None, dir=None, date=None):
    """Creating the OpenDX .cfg program configuration file.

    @keyword file_prefix:   The base part of the file name without extension.
    @type file_prefix:      str
    @keyword dir:           The directory to place the file in.
    @type dir:              str or None
    @keyword date:          The date string to include in the configuration.
    @type date:             str
    """

    # Print out.
    print("\nCreating the OpenDX .cfg program configuration file.")

    # Open the file.
    file = open_write_file(file_name=file_prefix+".cfg", dir=dir, force=True)

    # Generate the text.
    file.write("//\n")
    file.write("//\n")
    file.write("// time: %s\n" % date)
    file.write("//\n")
    file.write("// version: 3.2.0 (format), 4.3.2 (DX)\n")
    file.write("//\n")
    file.write("//\n")
    file.write("// panel[0]: position = (0.0164,0.0000), size = 0.2521x0.1933, startup = 1, devstyle = 1\n")
    file.write("// title: value = Control Panel\n")
    file.write("//\n")
    file.write("// workspace: width = 251, height = 142\n")
    file.write("// layout: snap = 0, width = 50, height = 50, align = NN\n")
    file.write("//\n")
    file.write("// interactor Selector[1]: num_components = 1, value = 1 \n")
    file.write("// selections: maximum = 2, current = 0 \n")
    file.write("// option[0]: name = \"Colour\", value = 1\n")
    file.write("// option[1]: name = \"Grey\", value = 2\n")
    file.write("// instance: panel = 0, x = 81, y = 6, style = Scrolled List, vertical = 1, size = 170x136\n")
    file.write("// label: value = Colour Selector\n")
    file.write("//\n")
    file.write("// node Image[3]:\n")
    file.write("// title: value = Surface\n")
    file.write("// depth: value = 24\n")
    file.write("// window: position = (0.0000,0.0400), size = 0.9929x0.9276\n")

    # Close the file.
    file.close()


def write_general(file_prefix=None, dir=None, inc=None):
    """Create the OpenDX .general file for a 3D map.

    @keyword file_prefix:   The base part of the file name without extension.
    @type file_prefix:      str
    @keyword dir:           The directory to place the file in.
    @type dir:              str or None
    @keyword inc:           The number of increments in the 3D map.
    @type inc:              int
    """

    # Printout.
    print("\nCreating the OpenDX .general file.")

    # Open the file.
    file = open_write_file(file_name=file_prefix+".general", dir=dir, force=True)

    # Write out the text.
    file.write("file = %s\n" % file_prefix)
    file.write("grid = %s x %s x %s\n" % (inc+1, inc+1, inc+1))
    file.write("format = ascii\n")
    file.write("interleaving = field\n")
    file.write("majority = row\n")
    file.write("field = data\n")
    file.write("structure = scalar\n")
    file.write("type = float\n")
    file.write("dependency = positions\n")
    file.write("positions = regular, regular, regular, 0, 1, 0, 1, 0, 1\n\n")
    file.write("end\n")

    # Close the file.
    file.close()


def write_point(file_prefix=None, dir=None, inc=None, point=None, num_points=None, bounds=None, N=None):
    """Creating a sphere at a given position within the map.

    The formula used to calculate the coordinate position is::

                        V - L
        coord =   Inc * -----
                        U - L

    where:
        - V is the coordinate or parameter value.
        - L is the lower bound value.
        - U is the upper bound value.
        - Inc is the number of increments.

    Both a data file and .general file will be created.


    @keyword file_prefix:   The base part of the file name without extension.
    @type file_prefix:      str
    @keyword dir:           The directory to place the file in.
    @type dir:              str or None
    @keyword inc:           The number of increments in the map.
    @type inc:              int
    @keyword point:         The list of coordinates of the point.
    @type point:            numpy rank-1 float array or list of numpy rank-1 float array.
    @keyword num_points:    The number of points.
    @type num_points:       int
    @keyword bounds:        The bounds of the map.
    @type bounds:           numpy rank-2 float array
    @keyword N:             The dimension of the map.
    @type N:                int
    """

    # Print out.
    print("\nCreating the OpenDX .general and data files for the point %s." % point)

    # Open the files.
    point_file = open_write_file(file_name=file_prefix, dir=dir, force=True)
    point_file_general = open_write_file(file_name=file_prefix+".general", dir=dir, force=True)

    # Calculate the coordinate values.
    if num_points > 1:
        for i in range(num_points):
            i_point = point[i]

            coords = inc * (i_point - bounds[:, 0]) / (bounds[:, 1] - bounds[:, 0])
            for i in range(N):
                point_file.write("%-15.5g" % coords[i])
            point_file.write("1\n")
    else:
        i_point = point

        coords = inc * (i_point - bounds[:, 0]) / (bounds[:, 1] - bounds[:, 0])
        for i in range(N):
            point_file.write("%-15.5g" % coords[i])
        point_file.write("1\n")

    # The text of the point .general file.
    point_file_general.write("file = %s\n" % file_prefix)
    point_file_general.write("points = %s\n" % num_points)
    point_file_general.write("format = ascii\n")
    point_file_general.write("interleaving = field\n")
    point_file_general.write("field = locations, field0\n")
    point_file_general.write("structure = 3-vector, scalar\n")
    point_file_general.write("type = float, float\n\n")
    point_file_general.write("end\n")

    # Close the data and .general files.
    point_file.close()
    point_file_general.close()


def write_program(file_prefix=None, point_file=None, dir=None, inc=None, N=None, num_points=None, labels=None, tick_locations=None, tick_values=None, date=None, chi_surface=[7.0, 20.0, 100.0, 500.0]):
    """Create the OpenDX .net program file.

    @keyword file_prefix:       The base part of the file name without extension.
    @type file_prefix:          str
    @keyword point_file:        The name of the point file to be displayed by the program.
    @type point_file:           str or None
    @keyword dir:               The directory to place the file in.
    @type dir:                  str or None
    @keyword inc:               The number of increments in the map.
    @type inc:                  int
    @keyword N:                 The dimension of the map.
    @type N:                    int
    @keyword num_points:        The number of points to be displayed by the program.
    @type num_points:           int
    @keyword labels:            The strings associated with the map axes, as an OpenDX string.
    @type labels:               str
    @keyword tick_locations:    The locations of the tick marks, as an OpenDX string.
    @type tick_locations:       str
    @keyword tick_values:       The values of the tick marks, as an OpenDX string.
    @type tick_values:          str
    @keyword date:              The date string to include in the configuration.
    @type date:                 str
    @keyword chi_surface:       Chi2 surface level for the Innermost, Inner, Middle and Outer Isosurface.
    @type chi_surface:          list of 4 floats
    """

    # Print out.
    print("\nCreating the OpenDX .net program file.")

    # Open the file.
    file = open_write_file(file_name=file_prefix+".net", dir=dir, force=True)

    # Corners.
    corners = "{[0"
    for i in range(N - 1):
        corners += " 0"
    corners += "] [%s" % inc
    for i in range(N - 1):
        corners += " %s" % inc
    corners += "]}"

    # Sphere size.
    sphere_size = repr(0.025 * (inc + 1.0))

    # Image setup.
    image_array1 = "[%s %s %s]" % (0.6 * (inc + 1.0), 0.3 * (inc + 1.0), 0.6 * (inc + 1.0))
    image_array2 = "[%s %s %s]" % (0.6 * (inc + 1.0), 0.3 * (inc + 1.0), 6.0 * (inc + 1.0))
    image_val = repr(3.0 * (inc + 1.0))

    # Unpacking the values for the 4 isosurfaces.
    innermost_isosurface_value, inner_isosurface_value, middle_isosurface_value, outer_isosurface_value  = chi_surface

    # Generate the text of the program.
    ###################################

    file.write("//\n")
    file.write("// time: %s\n" % date)
    file.write("//\n")
    file.write("// version: 3.2.0 (format), 4.3.2 (DX)\n")
    file.write("//\n")
    file.write("//\n")
    file.write("// MODULE main\n")

    # Include the sphere.
    if num_points > 0:
        file.write("\n")
        file.write("// page assignment: Colour Space\torder=3, windowed=0, showing=0\n")
        file.write("// page assignment: ColourScene\t\torder=5, windowed=0, showing=0\n")
        file.write("// page assignment: Glyph\t\torder=2, windowed=0, showing=0\n")
        file.write("// page assignment: Grey Space\t\torder=4, windowed=0, showing=0\n")
        file.write("// page assignment: GreyScene\t\torder=6, windowed=0, showing=0\n")
        file.write("// page assignment: Image\t\t\torder=7, windowed=0, showing=0\n")
        file.write("// page assignment: Isosurfaces\t\torder=1, windowed=0, showing=1\n")

    # No sphere.
    else:
        file.write("\n")
        file.write("// page assignment: Colour Space\torder=4, windowed=0, showing=0\n")
        file.write("// page assignment: ColourScene\t\torder=6, windowed=0, showing=0\n")
        file.write("// page assignment: Grey Space\t\torder=5, windowed=0, showing=0\n")
        file.write("// page assignment: GreyScene\t\torder=7, windowed=0, showing=0\n")
        file.write("// page assignment: Image\t\t\torder=8, windowed=0, showing=0\n")
        file.write("// page assignment: Isosurfaces\t\torder=2, windowed=0, showing=1\n")

    # Common code.
    file.write("\n")
    file.write("// workspace: width = 474, height = 354\n")
    file.write("// layout: snap = 0, width = 50, height = 50, align = NN\n")
    file.write("//\n")
    file.write("macro main(\n")
    file.write(") -> (\n")
    file.write(") {\n")
    file.write("    // \n")

    # Include the sphere.
    if num_points > 0:
        file.write("\n")
        file.write("    // node Import[4]: x = 177, y = 62, inputs = 6, label = %s\n" % point_file)
        file.write("    // input[1]: defaulting = 0, visible = 1, type = 32, value = \"%s.general\"\n" % point_file)
        file.write("    // input[3]: defaulting = 1, visible = 1, type = 32, value = \"general\"\n")
        file.write("    // page group: Glyph\n")
        file.write("    //\n")
        file.write("main_Import_4_out_1 = \n")
        file.write("    Import(\n")
        file.write("    main_Import_4_in_1,\n")
        file.write("    main_Import_4_in_2,\n")
        file.write("    main_Import_4_in_3,\n")
        file.write("    main_Import_4_in_4,\n")
        file.write("    main_Import_4_in_5,\n")
        file.write("    main_Import_4_in_6\n")
        file.write("    ) [instance: 4, cache: 1];\n")
        file.write("    // \n")
        file.write("    // node Glyph[2]: x = 201, y = 182, inputs = 7, label = Glyph\n")
        file.write("    // input[2]: defaulting = 0, visible = 1, type = 32, value = \"sphere\"\n")
        file.write("    // input[3]: defaulting = 1, visible = 1, type = 5, value = 10.0\n")
        file.write("    // input[4]: defaulting = 0, visible = 1, type = 5, value = %s\n" % sphere_size)
        file.write("    // input[5]: defaulting = 0, visible = 1, type = 5, value = 0.0\n")
        file.write("    // page group: Glyph\n")
        file.write("    //\n")
        file.write("main_Glyph_2_out_1 = \n")
        file.write("    Glyph(\n")
        file.write("    main_Import_4_out_1,\n")
        file.write("    main_Glyph_2_in_2,\n")
        file.write("    main_Glyph_2_in_3,\n")
        file.write("    main_Glyph_2_in_4,\n")
        file.write("    main_Glyph_2_in_5,\n")
        file.write("    main_Glyph_2_in_6,\n")
        file.write("    main_Glyph_2_in_7\n")
        file.write("    ) [instance: 2, cache: 1];\n")
        file.write("    // \n")
        file.write("    // node Color[10]: x = 357, y = 278, inputs = 5, label = Color\n")
        file.write("    // input[2]: defaulting = 0, visible = 1, type = 8, value = [0 0 0]\n")
        file.write("    // input[3]: defaulting = 0, visible = 1, type = 5, value = 1.0\n")
        file.write("    // page group: Glyph\n")
        file.write("    //\n")
        file.write("main_Color_10_out_1 = \n")
        file.write("    Color(\n")
        file.write("    main_Glyph_2_out_1,\n")
        file.write("    main_Color_10_in_2,\n")
        file.write("    main_Color_10_in_3,\n")
        file.write("    main_Color_10_in_4,\n")
        file.write("    main_Color_10_in_5\n")
        file.write("    ) [instance: 10, cache: 1];\n")
        file.write("    // \n")
        file.write("    // node Transmitter[1]: x = 352, y = 386, inputs = 1, label = GreySphere\n")
        file.write("    // page group: Glyph\n")
        file.write("    //\n")
        file.write("GreySphere = main_Color_10_out_1;\n")
        file.write("    // \n")
        file.write("    // node Receiver[2]: x = 190, y = 350, inputs = 1, label = GreySphere\n")
        file.write("    // page group: Grey Space\n")
        file.write("    //\n")
        file.write("main_Receiver_2_out_1[cache: 0] = GreySphere;\n")

    # Common code.
    file.write("\n")
    file.write("    // \n")
    file.write("    // node Import[3]: x = 225, y = 84, inputs = 6, label = %s\n" % file_prefix)
    file.write("    // input[1]: defaulting = 0, visible = 1, type = 32, value = \"%s.general\"\n" % file_prefix)
    file.write("    // input[3]: defaulting = 1, visible = 1, type = 32, value = \"general\"\n")
    file.write("    // page group: Isosurfaces\n")
    file.write("    //\n")
    file.write("main_Import_3_out_1 = \n")
    file.write("    Import(\n")
    file.write("    main_Import_3_in_1,\n")
    file.write("    main_Import_3_in_2,\n")
    file.write("    main_Import_3_in_3,\n")
    file.write("    main_Import_3_in_4,\n")
    file.write("    main_Import_3_in_5,\n")
    file.write("    main_Import_3_in_6\n")
    file.write("    ) [instance: 3, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Isosurface[5]: x = 102, y = 191, inputs = 6, label = Outer Isosurface\n")
    file.write("    // input[2]: defaulting = 0, visible = 1, type = 5, value = %.1f\n"%(outer_isosurface_value))
    file.write("    // page group: Isosurfaces\n")
    file.write("    //\n")
    file.write("main_Isosurface_5_out_1 = \n")
    file.write("    Isosurface(\n")
    file.write("    main_Import_3_out_1,\n")
    file.write("    main_Isosurface_5_in_2,\n")
    file.write("    main_Isosurface_5_in_3,\n")
    file.write("    main_Isosurface_5_in_4,\n")
    file.write("    main_Isosurface_5_in_5,\n")
    file.write("    main_Isosurface_5_in_6\n")
    file.write("    ) [instance: 5, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Transmitter[7]: x = 110, y = 292, inputs = 1, label = Surface4\n")
    file.write("    // page group: Isosurfaces\n")
    file.write("    //\n")
    file.write("Surface4 = main_Isosurface_5_out_1;\n")
    file.write("    // \n")
    file.write("    // node Receiver[14]: x = 123, y = 51, inputs = 1, label = Surface4\n")
    file.write("    // page group: Grey Space\n")
    file.write("    //\n")
    file.write("main_Receiver_14_out_1[cache: 0] = Surface4;\n")
    file.write("    // \n")
    file.write("    // node Color[6]: x = 142, y = 145, inputs = 5, label = Color\n")
    file.write("    // input[2]: defaulting = 0, visible = 1, type = 8, value = [0 0 0]\n")
    file.write("    // input[3]: defaulting = 0, visible = 1, type = 5, value = 0.2\n")
    file.write("    // page group: Grey Space\n")
    file.write("    //\n")
    file.write("main_Color_6_out_1 = \n")
    file.write("    Color(\n")
    file.write("    main_Receiver_14_out_1,\n")
    file.write("    main_Color_6_in_2,\n")
    file.write("    main_Color_6_in_3,\n")
    file.write("    main_Color_6_in_4,\n")
    file.write("    main_Color_6_in_5\n")
    file.write("    ) [instance: 6, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Isosurface[6]: x = 200, y = 191, inputs = 6, label = Middle Isosurface\n")
    file.write("    // input[2]: defaulting = 0, visible = 1, type = 5, value = %.1f\n"%(middle_isosurface_value))
    file.write("    // page group: Isosurfaces\n")
    file.write("    //\n")
    file.write("main_Isosurface_6_out_1 = \n")
    file.write("    Isosurface(\n")
    file.write("    main_Import_3_out_1,\n")
    file.write("    main_Isosurface_6_in_2,\n")
    file.write("    main_Isosurface_6_in_3,\n")
    file.write("    main_Isosurface_6_in_4,\n")
    file.write("    main_Isosurface_6_in_5,\n")
    file.write("    main_Isosurface_6_in_6\n")
    file.write("    ) [instance: 6, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Transmitter[8]: x = 208, y = 292, inputs = 1, label = Surface3\n")
    file.write("    // page group: Isosurfaces\n")
    file.write("    //\n")
    file.write("Surface3 = main_Isosurface_6_out_1;\n")
    file.write("    // \n")
    file.write("    // node Receiver[13]: x = 227, y = 51, inputs = 1, label = Surface3\n")
    file.write("    // page group: Grey Space\n")
    file.write("    //\n")
    file.write("main_Receiver_13_out_1[cache: 0] = Surface3;\n")
    file.write("    // \n")
    file.write("    // node Color[7]: x = 246, y = 145, inputs = 5, label = Color\n")
    file.write("    // input[2]: defaulting = 0, visible = 1, type = 8, value = [0.2 0.2 0.2]\n")
    file.write("    // input[3]: defaulting = 0, visible = 1, type = 5, value = 0.4\n")
    file.write("    // page group: Grey Space\n")
    file.write("    //\n")
    file.write("main_Color_7_out_1 = \n")
    file.write("    Color(\n")
    file.write("    main_Receiver_13_out_1,\n")
    file.write("    main_Color_7_in_2,\n")
    file.write("    main_Color_7_in_3,\n")
    file.write("    main_Color_7_in_4,\n")
    file.write("    main_Color_7_in_5\n")
    file.write("    ) [instance: 7, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Collect[5]: x = 203, y = 236, inputs = 2, label = Collect\n")
    file.write("    // page group: Grey Space\n")
    file.write("    //\n")
    file.write("main_Collect_5_out_1 = \n")
    file.write("    Collect(\n")
    file.write("    main_Color_6_out_1,\n")
    file.write("    main_Color_7_out_1\n")
    file.write("    ) [instance: 5, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Isosurface[7]: x = 298, y = 191, inputs = 6, label = Inner Isosurface\n")
    file.write("    // input[2]: defaulting = 0, visible = 1, type = 5, value = %.1f\n"%(inner_isosurface_value))
    file.write("    // page group: Isosurfaces\n")
    file.write("    //\n")
    file.write("main_Isosurface_7_out_1 = \n")
    file.write("    Isosurface(\n")
    file.write("    main_Import_3_out_1,\n")
    file.write("    main_Isosurface_7_in_2,\n")
    file.write("    main_Isosurface_7_in_3,\n")
    file.write("    main_Isosurface_7_in_4,\n")
    file.write("    main_Isosurface_7_in_5,\n")
    file.write("    main_Isosurface_7_in_6\n")
    file.write("    ) [instance: 7, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Transmitter[9]: x = 306, y = 292, inputs = 1, label = Surface2\n")
    file.write("    // page group: Isosurfaces\n")
    file.write("    //\n")
    file.write("Surface2 = main_Isosurface_7_out_1;\n")
    file.write("    // \n")
    file.write("    // node Receiver[12]: x = 331, y = 51, inputs = 1, label = Surface2\n")
    file.write("    // page group: Grey Space\n")
    file.write("    //\n")
    file.write("main_Receiver_12_out_1[cache: 0] = Surface2;\n")
    file.write("    // \n")
    file.write("    // node Color[8]: x = 350, y = 145, inputs = 5, label = Color\n")
    file.write("    // input[2]: defaulting = 0, visible = 1, type = 8, value = [0.5 0.5 0.5]\n")
    file.write("    // input[3]: defaulting = 0, visible = 1, type = 5, value = 0.5\n")
    file.write("    // page group: Grey Space\n")
    file.write("    //\n")
    file.write("main_Color_8_out_1 = \n")
    file.write("    Color(\n")
    file.write("    main_Receiver_12_out_1,\n")
    file.write("    main_Color_8_in_2,\n")
    file.write("    main_Color_8_in_3,\n")
    file.write("    main_Color_8_in_4,\n")
    file.write("    main_Color_8_in_5\n")
    file.write("    ) [instance: 8, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Isosurface[8]: x = 396, y = 191, inputs = 6, label = Innermost Isosurface\n")
    file.write("    // input[2]: defaulting = 0, visible = 1, type = 5, value = %.1f\n"%(innermost_isosurface_value))
    file.write("    // page group: Isosurfaces\n")
    file.write("    //\n")
    file.write("main_Isosurface_8_out_1 = \n")
    file.write("    Isosurface(\n")
    file.write("    main_Import_3_out_1,\n")
    file.write("    main_Isosurface_8_in_2,\n")
    file.write("    main_Isosurface_8_in_3,\n")
    file.write("    main_Isosurface_8_in_4,\n")
    file.write("    main_Isosurface_8_in_5,\n")
    file.write("    main_Isosurface_8_in_6\n")
    file.write("    ) [instance: 8, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Transmitter[10]: x = 404, y = 292, inputs = 1, label = Surface1\n")
    file.write("    // page group: Isosurfaces\n")
    file.write("    //\n")
    file.write("Surface1 = main_Isosurface_8_out_1;\n")
    file.write("    // \n")
    file.write("    // node Receiver[11]: x = 434, y = 51, inputs = 1, label = Surface1\n")
    file.write("    // page group: Grey Space\n")
    file.write("    //\n")
    file.write("main_Receiver_11_out_1[cache: 0] = Surface1;\n")
    file.write("    // \n")
    file.write("    // node Color[9]: x = 453, y = 145, inputs = 5, label = Color\n")
    file.write("    // input[2]: defaulting = 0, visible = 1, type = 32, value = \"white\"\n")
    file.write("    // input[3]: defaulting = 0, visible = 1, type = 5, value = 0.7\n")
    file.write("    // page group: Grey Space\n")
    file.write("    //\n")
    file.write("main_Color_9_out_1 = \n")
    file.write("    Color(\n")
    file.write("    main_Receiver_11_out_1,\n")
    file.write("    main_Color_9_in_2,\n")
    file.write("    main_Color_9_in_3,\n")
    file.write("    main_Color_9_in_4,\n")
    file.write("    main_Color_9_in_5\n")
    file.write("    ) [instance: 9, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Collect[6]: x = 409, y = 236, inputs = 2, label = Collect\n")
    file.write("    // page group: Grey Space\n")
    file.write("    //\n")
    file.write("main_Collect_6_out_1 = \n")
    file.write("    Collect(\n")
    file.write("    main_Color_8_out_1,\n")
    file.write("    main_Color_9_out_1\n")
    file.write("    ) [instance: 6, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Collect[7]: x = 307, y = 327, inputs = 2, label = Collect\n")
    file.write("    // page group: Grey Space\n")
    file.write("    //\n")
    file.write("main_Collect_7_out_1 = \n")
    file.write("    Collect(\n")
    file.write("    main_Collect_5_out_1,\n")
    file.write("    main_Collect_6_out_1\n")
    file.write("    ) [instance: 7, cache: 1];\n")
    file.write("    // \n")

    # Include the sphere.
    if num_points > 0:
        file.write("\n")
        file.write("    // node Collect[8]: x = 293, y = 431, inputs = 2, label = Collect\n")
        file.write("    // page group: Grey Space\n")
        file.write("    //\n")
        file.write("main_Collect_8_out_1 = \n")
        file.write("    Collect(\n")
        file.write("    main_Receiver_2_out_1,\n")
        file.write("    main_Collect_7_out_1\n")
        file.write("    ) [instance: 8, cache: 1];\n")
        file.write("    // \n")
        file.write("    // node Transmitter[4]: x = 282, y = 517, inputs = 1, label = GreySpace\n")

    # No sphere.
    else:
        file.write("\n")
        file.write("    // node Transmitter[4]: x = 296, y = 439, inputs = 1, label = GreySpace\n")

    # Common code.
    file.write("\n")
    file.write("    // page group: Grey Space\n")
    file.write("    // \n")

    # Include the sphere.
    if num_points > 0:
        file.write("\n")
        file.write("GreySpace = main_Collect_8_out_1;\n")

    # No sphere.
    else:
        file.write("\n")
        file.write("GreySpace = main_Collect_7_out_1;\n")

    # Common code.
    file.write("\n")
    file.write("    // \n")
    file.write("    // node Receiver[3]: x = 137, y = 57, inputs = 1, label = GreySpace\n")
    file.write("    // page group: GreyScene\n")
    file.write("    //\n")
    file.write("main_Receiver_3_out_1[cache: 0] = GreySpace;\n")
    file.write("    // \n")
    file.write("    // node Scale[3]: x = 163, y = 159, inputs = 2, label = Scale\n")
    file.write("    // input[2]: defaulting = 0, visible = 1, type = 8, value = [1 1 1]\n")
    file.write("    // page group: GreyScene\n")
    file.write("    //\n")
    file.write("main_Scale_3_out_1 = \n")
    file.write("    Scale(\n")
    file.write("    main_Receiver_3_out_1,\n")
    file.write("    main_Scale_3_in_2\n")
    file.write("    ) [instance: 3, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node AutoCamera[2]: x = 273, y = 264, inputs = 9, label = AutoCamera\n")
    file.write("    // input[2]: defaulting = 0, visible = 1, type = 8, value = [1 -1 1]\n")
    file.write("    // input[3]: defaulting = 1, visible = 1, type = 5, value = 500.0\n")
    file.write("    // input[4]: defaulting = 0, visible = 0, type = 1, value = 640\n")
    file.write("    // input[5]: defaulting = 0, visible = 0, type = 5, value = .75\n")
    file.write("    // input[6]: defaulting = 0, visible = 0, type = 8, value = [-1 1 0 ]\n")
    file.write("    // input[7]: defaulting = 0, visible = 0, type = 3, value = 0\n")
    file.write("    // input[8]: defaulting = 0, visible = 0, type = 5, value = 30.0\n")
    file.write("    // input[9]: defaulting = 0, visible = 1, type = 32, value = \"white\"\n")
    file.write("    // page group: GreyScene\n")
    file.write("    //\n")
    file.write("main_AutoCamera_2_out_1 = \n")
    file.write("    AutoCamera(\n")
    file.write("    main_Scale_3_out_1,\n")
    file.write("    main_AutoCamera_2_in_2,\n")
    file.write("    main_AutoCamera_2_in_3,\n")
    file.write("    main_AutoCamera_2_in_4,\n")
    file.write("    main_AutoCamera_2_in_5,\n")
    file.write("    main_AutoCamera_2_in_6,\n")
    file.write("    main_AutoCamera_2_in_7,\n")
    file.write("    main_AutoCamera_2_in_8,\n")
    file.write("    main_AutoCamera_2_in_9\n")
    file.write("    ) [instance: 2, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node AutoAxes[2]: x = 175, y = 379, inputs = 19, label = AutoAxes\n")
    file.write("    // input[3]: defaulting = 0, visible = 1, type = 16777248, value = %s\n" % labels)
    file.write("    // input[4]: defaulting = 0, visible = 0, type = 1, value = 30\n")
    file.write("    // input[5]: defaulting = 0, visible = 1, type = 16777224, value = %s\n" % corners)
    file.write("    // input[6]: defaulting = 0, visible = 1, type = 3, value = 1\n")
    file.write("    // input[7]: defaulting = 1, visible = 0, type = 3, value = 1\n")
    file.write("    // input[9]: defaulting = 0, visible = 1, type = 3, value = 1\n")
    file.write("    // input[10]: defaulting = 0, visible = 1, type = 16777224, value = {[1 1 1] [0.1 0.1 0.1] [0 0 0] [0 0 0]}\n")
    file.write("    // input[11]: defaulting = 0, visible = 1, type = 16777248, value = {\"background\" \"grid\" \"labels\" \"ticks\"}\n")
    file.write("    // input[12]: defaulting = 1, visible = 0, type = 5, value = 0.4\n")
    file.write("    // input[13]: defaulting = 0, visible = 0, type = 32, value = \"area\"\n")
    file.write("    // input[14]: defaulting = 0, visible = 1, type = 16777221, value = %s\n" % tick_locations[0])
    file.write("    // input[15]: defaulting = 0, visible = 1, type = 16777221, value = %s\n" % tick_locations[1])
    file.write("    // input[16]: defaulting = 0, visible = 1, type = 16777221, value = %s\n" % tick_locations[2])
    file.write("    // input[17]: defaulting = 0, visible = 1, type = 16777248, value = %s\n" % tick_values[0])
    file.write("    // input[18]: defaulting = 0, visible = 1, type = 16777248, value = %s\n" % tick_values[1])
    file.write("    // input[19]: defaulting = 0, visible = 1, type = 16777248, value = %s\n" % tick_values[2])
    file.write("    // page group: GreyScene\n")
    file.write("    //\n")
    file.write("main_AutoAxes_2_out_1 = \n")
    file.write("    AutoAxes(\n")
    file.write("    main_Scale_3_out_1,\n")
    file.write("    main_AutoCamera_2_out_1,\n")
    file.write("    main_AutoAxes_2_in_3,\n")
    file.write("    main_AutoAxes_2_in_4,\n")
    file.write("    main_AutoAxes_2_in_5,\n")
    file.write("    main_AutoAxes_2_in_6,\n")
    file.write("    main_AutoAxes_2_in_7,\n")
    file.write("    main_AutoAxes_2_in_8,\n")
    file.write("    main_AutoAxes_2_in_9,\n")
    file.write("    main_AutoAxes_2_in_10,\n")
    file.write("    main_AutoAxes_2_in_11,\n")
    file.write("    main_AutoAxes_2_in_12,\n")
    file.write("    main_AutoAxes_2_in_13,\n")
    file.write("    main_AutoAxes_2_in_14,\n")
    file.write("    main_AutoAxes_2_in_15,\n")
    file.write("    main_AutoAxes_2_in_16,\n")
    file.write("    main_AutoAxes_2_in_17,\n")
    file.write("    main_AutoAxes_2_in_18,\n")
    file.write("    main_AutoAxes_2_in_19\n")
    file.write("    ) [instance: 2, cache: 1];\n")
    file.write("    // \n")

    # Include the sphere.
    if num_points > 0:
        file.write("\n")
        file.write("    // node Color[11]: x = 133, y = 278, inputs = 5, label = Color\n")
        file.write("    // input[2]: defaulting = 0, visible = 1, type = 8, value = [1 0 0]\n")
        file.write("    // input[3]: defaulting = 0, visible = 1, type = 5, value = 1.0\n")
        file.write("    // page group: Glyph\n")
        file.write("    //\n")
        file.write("main_Color_11_out_1 = \n")
        file.write("    Color(\n")
        file.write("    main_Glyph_2_out_1,\n")
        file.write("    main_Color_11_in_2,\n")
        file.write("    main_Color_11_in_3,\n")
        file.write("    main_Color_11_in_4,\n")
        file.write("    main_Color_11_in_5\n")
        file.write("    ) [instance: 11, cache: 1];\n")
        file.write("    // \n")
        file.write("    // node Transmitter[2]: x = 122, y = 386, inputs = 1, label = ColourSphere\n")
        file.write("    // page group: Glyph\n")
        file.write("    //\n")
        file.write("ColourSphere = main_Color_11_out_1;\n")
        file.write("    // \n")
        file.write("    // node Receiver[1]: x = 179, y = 349, inputs = 1, label = ColourSphere\n")
        file.write("    // page group: Colour Space\n")
        file.write("    //\n")
        file.write("main_Receiver_1_out_1[cache: 0] = ColourSphere;\n")

    # Common code.
    file.write("\n")
    file.write("    // \n")
    file.write("    // node Receiver[10]: x = 123, y = 51, inputs = 1, label = Surface4\n")
    file.write("    // page group: Colour Space\n")
    file.write("    //\n")
    file.write("main_Receiver_10_out_1[cache: 0] = Surface4;\n")
    file.write("    // \n")
    file.write("    // node Color[12]: x = 142, y = 145, inputs = 5, label = Color\n")
    file.write("    // input[2]: defaulting = 0, visible = 1, type = 8, value = [0 0 0.2]\n")
    file.write("    // input[3]: defaulting = 0, visible = 1, type = 5, value = 0.3\n")
    file.write("    // input[4]: defaulting = 1, visible = 0, type = 32, value = NULL\n")
    file.write("    // input[5]: defaulting = 1, visible = 0, type = 3, value = NULL\n")
    file.write("    // page group: Colour Space\n")
    file.write("    //\n")
    file.write("main_Color_12_out_1 = \n")
    file.write("    Color(\n")
    file.write("    main_Receiver_10_out_1,\n")
    file.write("    main_Color_12_in_2,\n")
    file.write("    main_Color_12_in_3,\n")
    file.write("    main_Color_12_in_4,\n")
    file.write("    main_Color_12_in_5\n")
    file.write("    ) [instance: 12, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Receiver[9]: x = 227, y = 51, inputs = 1, label = Surface3\n")
    file.write("    // page group: Colour Space\n")
    file.write("    //\n")
    file.write("main_Receiver_9_out_1[cache: 0] = Surface3;\n")
    file.write("    // \n")
    file.write("    // node Color[13]: x = 246, y = 145, inputs = 5, label = Color\n")
    file.write("    // input[2]: defaulting = 0, visible = 1, type = 32, value = \"blue\"\n")
    file.write("    // input[3]: defaulting = 0, visible = 1, type = 5, value = 0.3\n")
    file.write("    // page group: Colour Space\n")
    file.write("    //\n")
    file.write("main_Color_13_out_1 = \n")
    file.write("    Color(\n")
    file.write("    main_Receiver_9_out_1,\n")
    file.write("    main_Color_13_in_2,\n")
    file.write("    main_Color_13_in_3,\n")
    file.write("    main_Color_13_in_4,\n")
    file.write("    main_Color_13_in_5\n")
    file.write("    ) [instance: 13, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Collect[9]: x = 203, y = 236, inputs = 2, label = Collect\n")
    file.write("    // page group: Colour Space\n")
    file.write("    //\n")
    file.write("main_Collect_9_out_1 = \n")
    file.write("    Collect(\n")
    file.write("    main_Color_12_out_1,\n")
    file.write("    main_Color_13_out_1\n")
    file.write("    ) [instance: 9, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Receiver[8]: x = 331, y = 51, inputs = 1, label = Surface2\n")
    file.write("    // page group: Colour Space\n")
    file.write("    //\n")
    file.write("main_Receiver_8_out_1[cache: 0] = Surface2;\n")
    file.write("    // \n")
    file.write("    // node Color[14]: x = 350, y = 145, inputs = 5, label = Color\n")
    file.write("    // input[2]: defaulting = 0, visible = 1, type = 8, value = [0.5 0.5 1]\n")
    file.write("    // input[3]: defaulting = 0, visible = 1, type = 5, value = 0.2\n")
    file.write("    // page group: Colour Space\n")
    file.write("    //\n")
    file.write("main_Color_14_out_1 = \n")
    file.write("    Color(\n")
    file.write("    main_Receiver_8_out_1,\n")
    file.write("    main_Color_14_in_2,\n")
    file.write("    main_Color_14_in_3,\n")
    file.write("    main_Color_14_in_4,\n")
    file.write("    main_Color_14_in_5\n")
    file.write("    ) [instance: 14, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Receiver[7]: x = 434, y = 51, inputs = 1, label = Surface1\n")
    file.write("    // page group: Colour Space\n")
    file.write("    //\n")
    file.write("main_Receiver_7_out_1[cache: 0] = Surface1;\n")
    file.write("    // \n")
    file.write("    // node Color[15]: x = 453, y = 145, inputs = 5, label = Color\n")
    file.write("    // input[2]: defaulting = 0, visible = 1, type = 32, value = \"white\"\n")
    file.write("    // input[3]: defaulting = 0, visible = 1, type = 5, value = 0.55\n")
    file.write("    // input[4]: defaulting = 1, visible = 0, type = 32, value = \"positions\"\n")
    file.write("    // page group: Colour Space\n")
    file.write("    //\n")
    file.write("main_Color_15_out_1 = \n")
    file.write("    Color(\n")
    file.write("    main_Receiver_7_out_1,\n")
    file.write("    main_Color_15_in_2,\n")
    file.write("    main_Color_15_in_3,\n")
    file.write("    main_Color_15_in_4,\n")
    file.write("    main_Color_15_in_5\n")
    file.write("    ) [instance: 15, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Collect[10]: x = 409, y = 236, inputs = 2, label = Collect\n")
    file.write("    // page group: Colour Space\n")
    file.write("    //\n")
    file.write("main_Collect_10_out_1 = \n")
    file.write("    Collect(\n")
    file.write("    main_Color_14_out_1,\n")
    file.write("    main_Color_15_out_1\n")
    file.write("    ) [instance: 10, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Collect[11]: x = 307, y = 327, inputs = 2, label = Collect\n")
    file.write("    // page group: Colour Space\n")
    file.write("    //\n")
    file.write("main_Collect_11_out_1 = \n")
    file.write("    Collect(\n")
    file.write("    main_Collect_9_out_1,\n")
    file.write("    main_Collect_10_out_1\n")
    file.write("    ) [instance: 11, cache: 1];\n")
    file.write("    // \n")

    # Include the sphere.
    if num_points > 0:
        file.write("\n")
        file.write("    // node Collect[12]: x = 293, y = 431, inputs = 2, label = Collect\n")
        file.write("    // page group: Colour Space\n")
        file.write("    //\n")
        file.write("main_Collect_12_out_1 = \n")
        file.write("    Collect(\n")
        file.write("    main_Receiver_1_out_1,\n")
        file.write("    main_Collect_11_out_1\n")
        file.write("    ) [instance: 12, cache: 1];\n")
        file.write("    // \n")
        file.write("    // node Transmitter[3]: x = 276, y = 517, inputs = 1, label = ColourSpace\n")

    # No sphere.
    else:
        file.write("\n")
        file.write("    // node Transmitter[3]: x = 290, y = 440, inputs = 1, label = ColourSpace\n")

    # Common code.
    file.write("\n")
    file.write("    // page group: Colour Space\n")
    file.write("    // \n")

    # Include the sphere.
    if num_points > 0:
        file.write("\n")
        file.write("ColourSpace = main_Collect_12_out_1;\n")

    # No sphere.
    else:
        file.write("\n")
        file.write("ColourSpace = main_Collect_11_out_1;\n")

    # Common code.
    file.write("\n")
    file.write("    // \n")
    file.write("    // node Receiver[4]: x = 131, y = 58, inputs = 1, label = ColourSpace\n")
    file.write("    // page group: ColourScene\n")
    file.write("    //\n")
    file.write("main_Receiver_4_out_1[cache: 0] = ColourSpace;\n")
    file.write("    // \n")
    file.write("    // node Scale[5]: x = 163, y = 159, inputs = 2, label = Scale\n")
    file.write("    // input[2]: defaulting = 0, visible = 1, type = 8, value = [1 1 1]\n")
    file.write("    // page group: ColourScene\n")
    file.write("    //\n")
    file.write("main_Scale_5_out_1 = \n")
    file.write("    Scale(\n")
    file.write("    main_Receiver_4_out_1,\n")
    file.write("    main_Scale_5_in_2\n")
    file.write("    ) [instance: 5, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node AutoCamera[4]: x = 273, y = 264, inputs = 9, label = AutoCamera\n")
    file.write("    // input[2]: defaulting = 0, visible = 1, type = 8, value = [1 -1 1]\n")
    file.write("    // input[3]: defaulting = 1, visible = 1, type = 5, value = 500.0\n")
    file.write("    // input[5]: defaulting = 0, visible = 0, type = 5, value = .75\n")
    file.write("    // input[6]: defaulting = 0, visible = 0, type = 8, value = [-1 1 0 ]\n")
    file.write("    // input[7]: defaulting = 0, visible = 0, type = 3, value = 0\n")
    file.write("    // input[8]: defaulting = 0, visible = 0, type = 5, value = 30.0\n")
    file.write("    // input[9]: defaulting = 0, visible = 1, type = 32, value = \"black\"\n")
    file.write("    // page group: ColourScene\n")
    file.write("    //\n")
    file.write("main_AutoCamera_4_out_1 = \n")
    file.write("    AutoCamera(\n")
    file.write("    main_Scale_5_out_1,\n")
    file.write("    main_AutoCamera_4_in_2,\n")
    file.write("    main_AutoCamera_4_in_3,\n")
    file.write("    main_AutoCamera_4_in_4,\n")
    file.write("    main_AutoCamera_4_in_5,\n")
    file.write("    main_AutoCamera_4_in_6,\n")
    file.write("    main_AutoCamera_4_in_7,\n")
    file.write("    main_AutoCamera_4_in_8,\n")
    file.write("    main_AutoCamera_4_in_9\n")
    file.write("    ) [instance: 4, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node AutoAxes[4]: x = 175, y = 379, inputs = 19, label = AutoAxes\n")
    file.write("    // input[3]: defaulting = 0, visible = 1, type = 16777248, value = %s\n" % labels)
    file.write("    // input[4]: defaulting = 0, visible = 0, type = 1, value = 30\n")
    file.write("    // input[5]: defaulting = 0, visible = 1, type = 16777224, value = %s\n" % corners)
    file.write("    // input[6]: defaulting = 0, visible = 1, type = 3, value = 1\n")
    file.write("    // input[7]: defaulting = 1, visible = 0, type = 3, value = 1\n")
    file.write("    // input[9]: defaulting = 0, visible = 1, type = 3, value = 1\n")
    file.write("    // input[10]: defaulting = 0, visible = 1, type = 16777224, value = {[0.05 0.05 0.05] [0.3 0.3 0.3] [1 1 1] [1 1 0]}\n")
    file.write("    // input[11]: defaulting = 0, visible = 1, type = 16777248, value = {\"background\" \"grid\" \"labels\" \"ticks\"}\n")
    file.write("    // input[12]: defaulting = 1, visible = 0, type = 5, value = 0.4\n")
    file.write("    // input[13]: defaulting = 0, visible = 0, type = 32, value = \"area\"\n")
    file.write("    // input[14]: defaulting = 0, visible = 1, type = 16777221, value = %s\n" % tick_locations[0])
    file.write("    // input[15]: defaulting = 0, visible = 1, type = 16777221, value = %s\n" % tick_locations[1])
    file.write("    // input[16]: defaulting = 0, visible = 1, type = 16777221, value = %s\n" % tick_locations[2])
    file.write("    // input[17]: defaulting = 0, visible = 1, type = 16777248, value = %s\n" % tick_values[0])
    file.write("    // input[18]: defaulting = 0, visible = 1, type = 16777248, value = %s\n" % tick_values[1])
    file.write("    // input[19]: defaulting = 0, visible = 1, type = 16777248, value = %s\n" % tick_values[2])
    file.write("    // page group: ColourScene\n")
    file.write("    //\n")
    file.write("main_AutoAxes_4_out_1 = \n")
    file.write("    AutoAxes(\n")
    file.write("    main_Scale_5_out_1,\n")
    file.write("    main_AutoCamera_4_out_1,\n")
    file.write("    main_AutoAxes_4_in_3,\n")
    file.write("    main_AutoAxes_4_in_4,\n")
    file.write("    main_AutoAxes_4_in_5,\n")
    file.write("    main_AutoAxes_4_in_6,\n")
    file.write("    main_AutoAxes_4_in_7,\n")
    file.write("    main_AutoAxes_4_in_8,\n")
    file.write("    main_AutoAxes_4_in_9,\n")
    file.write("    main_AutoAxes_4_in_10,\n")
    file.write("    main_AutoAxes_4_in_11,\n")
    file.write("    main_AutoAxes_4_in_12,\n")
    file.write("    main_AutoAxes_4_in_13,\n")
    file.write("    main_AutoAxes_4_in_14,\n")
    file.write("    main_AutoAxes_4_in_15,\n")
    file.write("    main_AutoAxes_4_in_16,\n")
    file.write("    main_AutoAxes_4_in_17,\n")
    file.write("    main_AutoAxes_4_in_18,\n")
    file.write("    main_AutoAxes_4_in_19\n")
    file.write("    ) [instance: 4, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Selector[1]: x = 245, y = 66, inputs = 7, label = Selector\n")
    file.write("    // input[1]: defaulting = 0, visible = 0, type = 32, value = \"Selector_1\"\n")
    file.write("    // input[2]: defaulting = 0, visible = 0, type = 32, value = \"Colour\" \n")
    file.write("    // input[3]: defaulting = 0, visible = 0, type = 29, value = 1 \n")
    file.write("    // input[4]: defaulting = 1, visible = 1, type = 16777248, value = { \"Colour\" \"Grey\" }\n")
    file.write("    // input[5]: defaulting = 1, visible = 0, type = 16777245, value = { 1 2 }\n")
    file.write("    // output[1]: visible = 1, type = 29, value = 1 \n")
    file.write("    // output[2]: visible = 1, type = 32, value = \"Colour\" \n")
    file.write("    // page group: Image\n")
    file.write("    //\n")
    file.write("    // \n")
    file.write("    // node Transmitter[6]: x = 299, y = 487, inputs = 1, label = ColourImage\n")
    file.write("    // page group: ColourScene\n")
    file.write("    //\n")
    file.write("ColourImage = main_AutoAxes_4_out_1;\n")
    file.write("    // \n")
    file.write("    // node Receiver[5]: x = 76, y = 190, inputs = 1, label = ColourImage\n")
    file.write("    // page group: Image\n")
    file.write("    //\n")
    file.write("main_Receiver_5_out_1[cache: 0] = ColourImage;\n")
    file.write("    // \n")
    file.write("    // node Transmitter[5]: x = 305, y = 489, inputs = 1, label = GreyImage\n")
    file.write("    // page group: GreyScene\n")
    file.write("    //\n")
    file.write("GreyImage = main_AutoAxes_2_out_1;\n")
    file.write("    // \n")
    file.write("    // node Receiver[6]: x = 199, y = 190, inputs = 1, label = GreyImage\n")
    file.write("    // page group: Image\n")
    file.write("    //\n")
    file.write("main_Receiver_6_out_1[cache: 0] = GreyImage;\n")
    file.write("    // \n")
    file.write("    // node Switch[1]: x = 177, y = 293, inputs = 3, label = Switch\n")
    file.write("    // page group: Image\n")
    file.write("    //\n")
    file.write("main_Switch_1_out_1 = \n")
    file.write("    Switch(\n")
    file.write("    main_Selector_1_out_1,\n")
    file.write("    main_Receiver_5_out_1,\n")
    file.write("    main_Receiver_6_out_1\n")
    file.write("    ) [instance: 1, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Switch[14]: x = 325, y = 293, inputs = 3, label = Switch\n")
    file.write("    // input[2]: defaulting = 0, visible = 1, type = 67108863, value = \"black\"\n")
    file.write("    // input[3]: defaulting = 0, visible = 1, type = 67108863, value = \"white\"\n")
    file.write("    // page group: Image\n")
    file.write("    //\n")
    file.write("main_Switch_14_out_1 = \n")
    file.write("    Switch(\n")
    file.write("    main_Selector_1_out_1,\n")
    file.write("    main_Switch_14_in_2,\n")
    file.write("    main_Switch_14_in_3\n")
    file.write("    ) [instance: 14, cache: 1];\n")
    file.write("    // \n")
    file.write("    // node Image[3]: x = 252, y = 424, inputs = 49, label = Image\n")
    file.write("    // input[1]: defaulting = 0, visible = 0, type = 67108863, value = \"Image_3\"\n")
    file.write("    // input[4]: defaulting = 0, visible = 0, type = 1, value = 1\n")
    file.write("    // input[5]: defaulting = 0, visible = 0, type = 8, value = %s\n" % image_array1)
    file.write("    // input[6]: defaulting = 0, visible = 0, type = 8, value = %s\n" % image_array2)
    file.write("    // input[7]: defaulting = 0, visible = 0, type = 5, value = %s\n" % image_val)
    file.write("    // input[8]: defaulting = 0, visible = 0, type = 1, value = 1376\n")
    file.write("    // input[9]: defaulting = 0, visible = 0, type = 5, value = 0.678\n")
    file.write("    // input[10]: defaulting = 0, visible = 0, type = 8, value = [-0.109685 0.243133 0.963772]\n")
    file.write("    // input[11]: defaulting = 1, visible = 0, type = 5, value = 30.9877\n")
    file.write("    // input[12]: defaulting = 0, visible = 0, type = 1, value = 0\n")
    file.write("    // input[14]: defaulting = 0, visible = 0, type = 1, value = 1\n")
    file.write("    // input[15]: defaulting = 0, visible = 0, type = 32, value = \"none\"\n")
    file.write("    // input[16]: defaulting = 0, visible = 0, type = 32, value = \"none\"\n")
    file.write("    // input[17]: defaulting = 1, visible = 0, type = 1, value = 1\n")
    file.write("    // input[18]: defaulting = 1, visible = 0, type = 1, value = 1\n")
    file.write("    // input[19]: defaulting = 0, visible = 0, type = 1, value = 0\n")
    file.write("    // input[22]: defaulting = 1, visible = 1, type = 32, value = \"black\"\n")
    file.write("    // input[25]: defaulting = 0, visible = 0, type = 32, value = \"iso\"\n")
    file.write("    // input[26]: defaulting = 0, visible = 0, type = 32, value = \"tiff\"\n")
    file.write("    // input[29]: defaulting = 0, visible = 0, type = 3, value = 0\n")
    file.write("    // input[30]: defaulting = 1, visible = 0, type = 16777248, value = %s\n" % labels)
    file.write("    // input[32]: defaulting = 1, visible = 0, type = 16777224, value = %s\n" % corners)
    file.write("    // input[33]: defaulting = 0, visible = 0, type = 3, value = 1\n")
    file.write("    // input[34]: defaulting = 0, visible = 0, type = 3, value = 0\n")
    file.write("    // input[36]: defaulting = 0, visible = 0, type = 3, value = 1\n")
    file.write("    // input[41]: defaulting = 0, visible = 0, type = 32, value = \"rotate\"\n")
    file.write("    // input[42]: defaulting = 0, visible = 0, type = 32, value = \"Surface\"\n")
    file.write("    // page group: Image\n")
    file.write("    // title: value = Surface\n")
    file.write("    // depth: value = 24\n")
    file.write("    // window: position = (0.0000,0.0400), size = 0.9929x0.9276\n")
    file.write("    // internal caching: 1\n")
    file.write("    //\n")
    file.write("main_Image_3_out_1,\n")
    file.write("main_Image_3_out_2,\n")
    file.write("main_Image_3_out_3 = \n")
    file.write("    Image(\n")
    file.write("    main_Image_3_in_1,\n")
    file.write("    main_Switch_1_out_1,\n")
    file.write("    main_Image_3_in_3,\n")
    file.write("    main_Image_3_in_4,\n")
    file.write("    main_Image_3_in_5,\n")
    file.write("    main_Image_3_in_6,\n")
    file.write("    main_Image_3_in_7,\n")
    file.write("    main_Image_3_in_8,\n")
    file.write("    main_Image_3_in_9,\n")
    file.write("    main_Image_3_in_10,\n")
    file.write("    main_Image_3_in_11,\n")
    file.write("    main_Image_3_in_12,\n")
    file.write("    main_Image_3_in_13,\n")
    file.write("    main_Image_3_in_14,\n")
    file.write("    main_Image_3_in_15,\n")
    file.write("    main_Image_3_in_16,\n")
    file.write("    main_Image_3_in_17,\n")
    file.write("    main_Image_3_in_18,\n")
    file.write("    main_Image_3_in_19,\n")
    file.write("    main_Image_3_in_20,\n")
    file.write("    main_Image_3_in_21,\n")
    file.write("    main_Switch_14_out_1,\n")
    file.write("    main_Image_3_in_23,\n")
    file.write("    main_Image_3_in_24,\n")
    file.write("    main_Image_3_in_25,\n")
    file.write("    main_Image_3_in_26,\n")
    file.write("    main_Image_3_in_27,\n")
    file.write("    main_Image_3_in_28,\n")
    file.write("    main_Image_3_in_29,\n")
    file.write("    main_Image_3_in_30,\n")
    file.write("    main_Image_3_in_31,\n")
    file.write("    main_Image_3_in_32,\n")
    file.write("    main_Image_3_in_33,\n")
    file.write("    main_Image_3_in_34,\n")
    file.write("    main_Image_3_in_35,\n")
    file.write("    main_Image_3_in_36,\n")
    file.write("    main_Image_3_in_37,\n")
    file.write("    main_Image_3_in_38,\n")
    file.write("    main_Image_3_in_39,\n")
    file.write("    main_Image_3_in_40,\n")
    file.write("    main_Image_3_in_41,\n")
    file.write("    main_Image_3_in_42,\n")
    file.write("    main_Image_3_in_43,\n")
    file.write("    main_Image_3_in_44,\n")
    file.write("    main_Image_3_in_45,\n")
    file.write("    main_Image_3_in_46,\n")
    file.write("    main_Image_3_in_47,\n")
    file.write("    main_Image_3_in_48,\n")
    file.write("    main_Image_3_in_49\n")
    file.write("    ) [instance: 3, cache: 1];\n")
    file.write("// network: end of macro body\n")
    file.write("CacheScene(main_Image_3_in_1, main_Image_3_out_1, main_Image_3_out_2);\n")
    file.write("}\n")

    # Include the sphere.
    if num_points > 0:
        file.write("\n")
        file.write("main_Import_4_in_1 = \"%s.general\";\n" % point_file)
        file.write("main_Import_4_in_2 = NULL;\n")
        file.write("main_Import_4_in_3 = NULL;\n")
        file.write("main_Import_4_in_4 = NULL;\n")
        file.write("main_Import_4_in_5 = NULL;\n")
        file.write("main_Import_4_in_6 = NULL;\n")
        file.write("main_Import_4_out_1 = NULL;\n")
        file.write("main_Glyph_2_in_2 = \"sphere\";\n")
        file.write("main_Glyph_2_in_3 = NULL;\n")
        file.write("main_Glyph_2_in_4 = %s;\n" % sphere_size)
        file.write("main_Glyph_2_in_5 = 0.0;\n")
        file.write("main_Glyph_2_in_6 = NULL;\n")
        file.write("main_Glyph_2_in_7 = NULL;\n")
        file.write("main_Glyph_2_out_1 = NULL;\n")
        file.write("main_Color_10_in_2 = [0 0 0];\n")
        file.write("main_Color_10_in_3 = 1.0;\n")
        file.write("main_Color_10_in_4 = NULL;\n")
        file.write("main_Color_10_in_5 = NULL;\n")
        file.write("main_Color_10_out_1 = NULL;\n")
        file.write("main_Transmitter_1_out_1 = NULL;\n")
        file.write("main_Receiver_2_out_1 = NULL;\n")

    # Common code.
    file.write("\n")
    file.write("main_Import_3_in_1 = \"%s.general\";\n" % file_prefix)
    file.write("main_Import_3_in_2 = NULL;\n")
    file.write("main_Import_3_in_3 = NULL;\n")
    file.write("main_Import_3_in_4 = NULL;\n")
    file.write("main_Import_3_in_5 = NULL;\n")
    file.write("main_Import_3_in_6 = NULL;\n")
    file.write("main_Import_3_out_1 = NULL;\n")
    file.write("main_Isosurface_5_in_2 = 500.0;\n")
    file.write("main_Isosurface_5_in_3 = NULL;\n")
    file.write("main_Isosurface_5_in_4 = NULL;\n")
    file.write("main_Isosurface_5_in_5 = NULL;\n")
    file.write("main_Isosurface_5_in_6 = NULL;\n")
    file.write("main_Isosurface_5_out_1 = NULL;\n")
    file.write("main_Transmitter_7_out_1 = NULL;\n")
    file.write("main_Receiver_14_out_1 = NULL;\n")
    file.write("main_Color_6_in_2 = [0 0 0];\n")
    file.write("main_Color_6_in_3 = 0.2;\n")
    file.write("main_Color_6_in_4 = NULL;\n")
    file.write("main_Color_6_in_5 = NULL;\n")
    file.write("main_Color_6_out_1 = NULL;\n")
    file.write("main_Isosurface_6_in_2 = 100.0;\n")
    file.write("main_Isosurface_6_in_3 = NULL;\n")
    file.write("main_Isosurface_6_in_4 = NULL;\n")
    file.write("main_Isosurface_6_in_5 = NULL;\n")
    file.write("main_Isosurface_6_in_6 = NULL;\n")
    file.write("main_Isosurface_6_out_1 = NULL;\n")
    file.write("main_Transmitter_8_out_1 = NULL;\n")
    file.write("main_Receiver_13_out_1 = NULL;\n")
    file.write("main_Color_7_in_2 = [0.2 0.2 0.2];\n")
    file.write("main_Color_7_in_3 = 0.4;\n")
    file.write("main_Color_7_in_4 = NULL;\n")
    file.write("main_Color_7_in_5 = NULL;\n")
    file.write("main_Color_7_out_1 = NULL;\n")
    file.write("main_Collect_5_out_1 = NULL;\n")
    file.write("main_Isosurface_7_in_2 = 20.0;\n")
    file.write("main_Isosurface_7_in_3 = NULL;\n")
    file.write("main_Isosurface_7_in_4 = NULL;\n")
    file.write("main_Isosurface_7_in_5 = NULL;\n")
    file.write("main_Isosurface_7_in_6 = NULL;\n")
    file.write("main_Isosurface_7_out_1 = NULL;\n")
    file.write("main_Transmitter_9_out_1 = NULL;\n")
    file.write("main_Receiver_12_out_1 = NULL;\n")
    file.write("main_Color_8_in_2 = [0.5 0.5 0.5];\n")
    file.write("main_Color_8_in_3 = 0.5;\n")
    file.write("main_Color_8_in_4 = NULL;\n")
    file.write("main_Color_8_in_5 = NULL;\n")
    file.write("main_Color_8_out_1 = NULL;\n")
    file.write("main_Isosurface_8_in_2 = 7.0;\n")
    file.write("main_Isosurface_8_in_3 = NULL;\n")
    file.write("main_Isosurface_8_in_4 = NULL;\n")
    file.write("main_Isosurface_8_in_5 = NULL;\n")
    file.write("main_Isosurface_8_in_6 = NULL;\n")
    file.write("main_Isosurface_8_out_1 = NULL;\n")
    file.write("main_Transmitter_10_out_1 = NULL;\n")
    file.write("main_Receiver_11_out_1 = NULL;\n")
    file.write("main_Color_9_in_2 = \"white\";\n")
    file.write("main_Color_9_in_3 = 0.7;\n")
    file.write("main_Color_9_in_4 = NULL;\n")
    file.write("main_Color_9_in_5 = NULL;\n")
    file.write("main_Color_9_out_1 = NULL;\n")
    file.write("main_Collect_6_out_1 = NULL;\n")
    file.write("main_Collect_7_out_1 = NULL;\n")

    # Include the sphere.
    if num_points > 0:
        file.write("\n")
        file.write("main_Collect_8_out_1 = NULL;\n")

    # Common code.
    file.write("\n")
    file.write("main_Transmitter_4_out_1 = NULL;\n")
    file.write("main_Receiver_3_out_1 = NULL;\n")
    file.write("main_Scale_3_in_2 = [1 1 1];\n")
    file.write("main_Scale_3_out_1 = NULL;\n")
    file.write("main_AutoCamera_2_in_2 = [1 -1 1];\n")
    file.write("main_AutoCamera_2_in_3 = NULL;\n")
    file.write("main_AutoCamera_2_in_4 = 640;\n")
    file.write("main_AutoCamera_2_in_5 = .75;\n")
    file.write("main_AutoCamera_2_in_6 = [-1 1 0 ];\n")
    file.write("main_AutoCamera_2_in_7 = 0;\n")
    file.write("main_AutoCamera_2_in_8 = 30.0;\n")
    file.write("main_AutoCamera_2_in_9 = \"white\";\n")
    file.write("main_AutoCamera_2_out_1 = NULL;\n")
    file.write("main_AutoAxes_2_in_3 = %s;\n" % labels)
    file.write("main_AutoAxes_2_in_4 = 30;\n")
    file.write("main_AutoAxes_2_in_5 = %s;\n" % corners)
    file.write("main_AutoAxes_2_in_6 = 1;\n")
    file.write("main_AutoAxes_2_in_7 = NULL;\n")
    file.write("main_AutoAxes_2_in_8 = NULL;\n")
    file.write("main_AutoAxes_2_in_9 = 1;\n")
    file.write("main_AutoAxes_2_in_10 = {[1 1 1] [0.1 0.1 0.1] [0 0 0] [0 0 0]};\n")
    file.write("main_AutoAxes_2_in_11 = {\"background\" \"grid\" \"labels\" \"ticks\"};\n")
    file.write("main_AutoAxes_2_in_12 = NULL;\n")
    file.write("main_AutoAxes_2_in_13 = \"area\";\n")
    file.write("main_AutoAxes_2_in_14 = %s;\n" % tick_locations[0])
    file.write("main_AutoAxes_2_in_15 = %s;\n" % tick_locations[1])
    file.write("main_AutoAxes_2_in_16 = %s;\n" % tick_locations[2])
    file.write("main_AutoAxes_2_in_17 = %s;\n" % tick_values[0])
    file.write("main_AutoAxes_2_in_18 = %s;\n" % tick_values[1])
    file.write("main_AutoAxes_2_in_19 = %s;\n" % tick_values[2])
    file.write("main_AutoAxes_2_out_1 = NULL;\n")

    # Include the sphere.
    if num_points > 0:
        file.write("\n")
        file.write("main_Color_11_in_2 = [1 0 0];\n")
        file.write("main_Color_11_in_3 = 1.0;\n")
        file.write("main_Color_11_in_4 = NULL;\n")
        file.write("main_Color_11_in_5 = NULL;\n")
        file.write("main_Color_11_out_1 = NULL;\n")
        file.write("main_Transmitter_2_out_1 = NULL;\n")
        file.write("main_Receiver_1_out_1 = NULL;\n")

    # Common code.
    file.write("\n")
    file.write("main_Receiver_10_out_1 = NULL;\n")
    file.write("main_Color_12_in_2 = [0 0 0.2];\n")
    file.write("main_Color_12_in_3 = 0.3;\n")
    file.write("main_Color_12_in_4 = NULL;\n")
    file.write("main_Color_12_in_5 = NULL;\n")
    file.write("main_Color_12_out_1 = NULL;\n")
    file.write("main_Receiver_9_out_1 = NULL;\n")
    file.write("main_Color_13_in_2 = \"blue\";\n")
    file.write("main_Color_13_in_3 = 0.3;\n")
    file.write("main_Color_13_in_4 = NULL;\n")
    file.write("main_Color_13_in_5 = NULL;\n")
    file.write("main_Color_13_out_1 = NULL;\n")
    file.write("main_Collect_9_out_1 = NULL;\n")
    file.write("main_Receiver_8_out_1 = NULL;\n")
    file.write("main_Color_14_in_2 = [0.5 0.5 1];\n")
    file.write("main_Color_14_in_3 = 0.2;\n")
    file.write("main_Color_14_in_4 = NULL;\n")
    file.write("main_Color_14_in_5 = NULL;\n")
    file.write("main_Color_14_out_1 = NULL;\n")
    file.write("main_Receiver_7_out_1 = NULL;\n")
    file.write("main_Color_15_in_2 = \"white\";\n")
    file.write("main_Color_15_in_3 = 0.55;\n")
    file.write("main_Color_15_in_4 = NULL;\n")
    file.write("main_Color_15_in_5 = NULL;\n")
    file.write("main_Color_15_out_1 = NULL;\n")
    file.write("main_Collect_10_out_1 = NULL;\n")
    file.write("main_Collect_11_out_1 = NULL;\n")

    # Include the sphere.
    if num_points > 0:
        file.write("\n")
        file.write("main_Collect_12_out_1 = NULL;\n")

    # Common code.
    file.write("\n")
    file.write("main_Transmitter_3_out_1 = NULL;\n")
    file.write("main_Receiver_4_out_1 = NULL;\n")
    file.write("main_Scale_5_in_2 = [1 1 1];\n")
    file.write("main_Scale_5_out_1 = NULL;\n")
    file.write("main_AutoCamera_4_in_2 = [1 -1 1];\n")
    file.write("main_AutoCamera_4_in_3 = NULL;\n")
    file.write("main_AutoCamera_4_in_4 = NULL;\n")
    file.write("main_AutoCamera_4_in_5 = .75;\n")
    file.write("main_AutoCamera_4_in_6 = [-1 1 0 ];\n")
    file.write("main_AutoCamera_4_in_7 = 0;\n")
    file.write("main_AutoCamera_4_in_8 = 30.0;\n")
    file.write("main_AutoCamera_4_in_9 = \"black\";\n")
    file.write("main_AutoCamera_4_out_1 = NULL;\n")
    file.write("main_AutoAxes_4_in_3 = %s;\n" % labels)
    file.write("main_AutoAxes_4_in_4 = 30;\n")
    file.write("main_AutoAxes_4_in_5 = %s;\n" % corners)
    file.write("main_AutoAxes_4_in_6 = 1;\n")
    file.write("main_AutoAxes_4_in_7 = NULL;\n")
    file.write("main_AutoAxes_4_in_8 = NULL;\n")
    file.write("main_AutoAxes_4_in_9 = 1;\n")
    file.write("main_AutoAxes_4_in_10 = {[0.05 0.05 0.05] [0.3 0.3 0.3] [1 1 1] [1 1 0]};\n")
    file.write("main_AutoAxes_4_in_11 = {\"background\" \"grid\" \"labels\" \"ticks\"};\n")
    file.write("main_AutoAxes_4_in_12 = NULL;\n")
    file.write("main_AutoAxes_4_in_13 = \"area\";\n")
    file.write("main_AutoAxes_4_in_14 = %s;\n" % tick_locations[0])
    file.write("main_AutoAxes_4_in_15 = %s;\n" % tick_locations[1])
    file.write("main_AutoAxes_4_in_16 = %s;\n" % tick_locations[2])
    file.write("main_AutoAxes_4_in_17 = %s;\n" % tick_values[0])
    file.write("main_AutoAxes_4_in_18 = %s;\n" % tick_values[1])
    file.write("main_AutoAxes_4_in_19 = %s;\n" % tick_values[2])
    file.write("main_AutoAxes_4_out_1 = NULL;\n")
    file.write("main_Selector_1_in_1 = \"Selector_1\";\n")
    file.write("main_Selector_1_in_2 = \"Colour\" ;\n")
    file.write("main_Selector_1_in_3 = 1 ;\n")
    file.write("main_Selector_1_in_4 = NULL;\n")
    file.write("main_Selector_1_in_5 = NULL;\n")
    file.write("main_Selector_1_in_6 = NULL;\n")
    file.write("main_Selector_1_in_7 = NULL;\n")
    file.write("main_Selector_1_out_1 = 1 ;\n")
    file.write("main_Transmitter_6_out_1 = NULL;\n")
    file.write("main_Receiver_5_out_1 = NULL;\n")
    file.write("main_Transmitter_5_out_1 = NULL;\n")
    file.write("main_Receiver_6_out_1 = NULL;\n")
    file.write("main_Switch_1_out_1 = NULL;\n")
    file.write("main_Switch_14_in_2 = \"black\";\n")
    file.write("main_Switch_14_in_3 = \"white\";\n")
    file.write("main_Switch_14_out_1 = NULL;\n")
    file.write("macro Image(\n")
    file.write("        id,\n")
    file.write("        object,\n")
    file.write("        where,\n")
    file.write("        useVector,\n")
    file.write("        to,\n")
    file.write("        from,\n")
    file.write("        width,\n")
    file.write("        resolution,\n")
    file.write("        aspect,\n")
    file.write("        up,\n")
    file.write("        viewAngle,\n")
    file.write("        perspective,\n")
    file.write("        options,\n")
    file.write("        buttonState = 1,\n")
    file.write("        buttonUpApprox = \"none\",\n")
    file.write("        buttonDownApprox = \"none\",\n")
    file.write("        buttonUpDensity = 1,\n")
    file.write("        buttonDownDensity = 1,\n")
    file.write("        renderMode = 0,\n")
    file.write("        defaultCamera,\n")
    file.write("        reset,\n")
    file.write("        backgroundColor,\n")
    file.write("        throttle,\n")
    file.write("        RECenable = 0,\n")
    file.write("        RECfile,\n")
    file.write("        RECformat,\n")
    file.write("        RECresolution,\n")
    file.write("        RECaspect,\n")
    file.write("        AAenable = 0,\n")
    file.write("        AAlabels,\n")
    file.write("        AAticks,\n")
    file.write("        AAcorners,\n")
    file.write("        AAframe,\n")
    file.write("        AAadjust,\n")
    file.write("        AAcursor,\n")
    file.write("        AAgrid,\n")
    file.write("        AAcolors,\n")
    file.write("        AAannotation,\n")
    file.write("        AAlabelscale,\n")
    file.write("        AAfont,\n")
    file.write("        interactionMode,\n")
    file.write("        title,\n")
    file.write("        AAxTickLocs,\n")
    file.write("        AAyTickLocs,\n")
    file.write("        AAzTickLocs,\n")
    file.write("        AAxTickLabels,\n")
    file.write("        AAyTickLabels,\n")
    file.write("        AAzTickLabels,\n")
    file.write("        webOptions) -> (\n")
    file.write("        object,\n")
    file.write("        camera,\n")
    file.write("        where)\n")
    file.write("{\n")
    file.write("    ImageMessage(\n")
    file.write("        id,\n")
    file.write("        backgroundColor,\n")
    file.write("        throttle,\n")
    file.write("        RECenable,\n")
    file.write("        RECfile,\n")
    file.write("        RECformat,\n")
    file.write("        RECresolution,\n")
    file.write("        RECaspect,\n")
    file.write("        AAenable,\n")
    file.write("        AAlabels,\n")
    file.write("        AAticks,\n")
    file.write("        AAcorners,\n")
    file.write("        AAframe,\n")
    file.write("        AAadjust,\n")
    file.write("        AAcursor,\n")
    file.write("        AAgrid,\n")
    file.write("        AAcolors,\n")
    file.write("        AAannotation,\n")
    file.write("        AAlabelscale,\n")
    file.write("        AAfont,\n")
    file.write("        AAxTickLocs,\n")
    file.write("        AAyTickLocs,\n")
    file.write("        AAzTickLocs,\n")
    file.write("        AAxTickLabels,\n")
    file.write("        AAyTickLabels,\n")
    file.write("        AAzTickLabels,\n")
    file.write("        interactionMode,\n")
    file.write("        title,\n")
    file.write("        renderMode,\n")
    file.write("        buttonUpApprox,\n")
    file.write("        buttonDownApprox,\n")
    file.write("        buttonUpDensity,\n")
    file.write("        buttonDownDensity) [instance: 1, cache: 1];\n")
    file.write("    autoCamera =\n")
    file.write("        AutoCamera(\n")
    file.write("            object,\n")
    file.write("            \"front\",\n")
    file.write("            object,\n")
    file.write("            resolution,\n")
    file.write("            aspect,\n")
    file.write("            [0,1,0],\n")
    file.write("            perspective,\n")
    file.write("            viewAngle,\n")
    file.write("            backgroundColor) [instance: 1, cache: 1];\n")
    file.write("    realCamera =\n")
    file.write("        Camera(\n")
    file.write("            to,\n")
    file.write("            from,\n")
    file.write("            width,\n")
    file.write("            resolution,\n")
    file.write("            aspect,\n")
    file.write("            up,\n")
    file.write("            perspective,\n")
    file.write("            viewAngle,\n")
    file.write("            backgroundColor) [instance: 1, cache: 1];\n")
    file.write("    coloredDefaultCamera = \n")
    file.write("        UpdateCamera(defaultCamera,\n")
    file.write("            background=backgroundColor) [instance: 1, cache: 1];\n")
    file.write("    nullDefaultCamera =\n")
    file.write("        Inquire(defaultCamera,\n")
    file.write("            \"is null + 1\") [instance: 1, cache: 1];\n")
    file.write("    resetCamera =\n")
    file.write("        Switch(\n")
    file.write("            nullDefaultCamera,\n")
    file.write("            coloredDefaultCamera,\n")
    file.write("            autoCamera) [instance: 1, cache: 1];\n")
    file.write("    resetNull = \n")
    file.write("        Inquire(\n")
    file.write("            reset,\n")
    file.write("            \"is null + 1\") [instance: 2, cache: 1];\n")
    file.write("    reset =\n")
    file.write("        Switch(\n")
    file.write("            resetNull,\n")
    file.write("            reset,\n")
    file.write("            0) [instance: 2, cache: 1];\n")
    file.write("    whichCamera =\n")
    file.write("        Compute(\n")
    file.write("            \"($0 != 0 || $1 == 0) ? 1 : 2\",\n")
    file.write("            reset,\n")
    file.write("            useVector) [instance: 1, cache: 1];\n")
    file.write("    camera = Switch(\n")
    file.write("            whichCamera,\n")
    file.write("            resetCamera,\n")
    file.write("            realCamera) [instance: 3, cache: 1];\n")
    file.write("    AAobject =\n")
    file.write("        AutoAxes(\n")
    file.write("            object,\n")
    file.write("            camera,\n")
    file.write("            AAlabels,\n")
    file.write("            AAticks,\n")
    file.write("            AAcorners,\n")
    file.write("            AAframe,\n")
    file.write("            AAadjust,\n")
    file.write("            AAcursor,\n")
    file.write("            AAgrid,\n")
    file.write("            AAcolors,\n")
    file.write("            AAannotation,\n")
    file.write("            AAlabelscale,\n")
    file.write("            AAfont,\n")
    file.write("            AAxTickLocs,\n")
    file.write("            AAyTickLocs,\n")
    file.write("            AAzTickLocs,\n")
    file.write("            AAxTickLabels,\n")
    file.write("            AAyTickLabels,\n")
    file.write("            AAzTickLabels) [instance: 1, cache: 1];\n")
    file.write("    switchAAenable = Compute(\"$0+1\",\n")
    file.write("            AAenable) [instance: 2, cache: 1];\n")
    file.write("    object = Switch(\n")
    file.write("            switchAAenable,\n")
    file.write("            object,\n")
    file.write("            AAobject) [instance:4, cache: 1];\n")
    file.write("    SWapproximation_options =\n")
    file.write("        Switch(\n")
    file.write("            buttonState,\n")
    file.write("            buttonUpApprox,\n")
    file.write("            buttonDownApprox) [instance: 5, cache: 1];\n")
    file.write("    SWdensity_options =\n")
    file.write("        Switch(\n")
    file.write("            buttonState,\n")
    file.write("            buttonUpDensity,\n")
    file.write("            buttonDownDensity) [instance: 6, cache: 1];\n")
    file.write("    HWapproximation_options =\n")
    file.write("        Format(\n")
    file.write("            \"%s,%s\",\n")
    file.write("            buttonDownApprox,\n")
    file.write("            buttonUpApprox) [instance: 1, cache: 1];\n")
    file.write("    HWdensity_options =\n")
    file.write("        Format(\n")
    file.write("            \"%d,%d\",\n")
    file.write("            buttonDownDensity,\n")
    file.write("            buttonUpDensity) [instance: 2, cache: 1];\n")
    file.write("    switchRenderMode = Compute(\n")
    file.write("            \"$0+1\",\n")
    file.write("            renderMode) [instance: 3, cache: 1];\n")
    file.write("    approximation_options = Switch(\n")
    file.write("            switchRenderMode,\n")
    file.write("            SWapproximation_options,\n")
    file.write("            HWapproximation_options) [instance: 7, cache: 1];\n")
    file.write("    density_options = Switch(\n")
    file.write("            switchRenderMode,\n")
    file.write("            SWdensity_options,\n")
    file.write("            HWdensity_options) [instance: 8, cache: 1];\n")
    file.write("    renderModeString = Switch(\n")
    file.write("            switchRenderMode,\n")
    file.write("            \"software\",\n")
    file.write("            \"hardware\")[instance: 9, cache: 1];\n")
    file.write("    object_tag = Inquire(\n")
    file.write("            object,\n")
    file.write("            \"object tag\")[instance: 3, cache: 1];\n")
    file.write("    annoted_object =\n")
    file.write("        Options(\n")
    file.write("            object,\n")
    file.write("            \"send boxes\",\n")
    file.write("            0,\n")
    file.write("            \"cache\",\n")
    file.write("            1,\n")
    file.write("            \"object tag\",\n")
    file.write("            object_tag,\n")
    file.write("            \"ddcamera\",\n")
    file.write("            whichCamera,\n")
    file.write("            \"rendering approximation\",\n")
    file.write("            approximation_options,\n")
    file.write("            \"render every\",\n")
    file.write("            density_options,\n")
    file.write("            \"button state\",\n")
    file.write("            buttonState,\n")
    file.write("            \"rendering mode\",\n")
    file.write("            renderModeString) [instance: 1, cache: 1];\n")
    file.write("    RECresNull =\n")
    file.write("        Inquire(\n")
    file.write("            RECresolution,\n")
    file.write("            \"is null + 1\") [instance: 4, cache: 1];\n")
    file.write("    ImageResolution =\n")
    file.write("        Inquire(\n")
    file.write("            camera,\n")
    file.write("            \"camera resolution\") [instance: 5, cache: 1];\n")
    file.write("    RECresolution =\n")
    file.write("        Switch(\n")
    file.write("            RECresNull,\n")
    file.write("            RECresolution,\n")
    file.write("            ImageResolution) [instance: 10, cache: 1];\n")
    file.write("    RECaspectNull =\n")
    file.write("        Inquire(\n")
    file.write("            RECaspect,\n")
    file.write("            \"is null + 1\") [instance: 6, cache: 1];\n")
    file.write("    ImageAspect =\n")
    file.write("        Inquire(\n")
    file.write("            camera,\n")
    file.write("            \"camera aspect\") [instance: 7, cache: 1];\n")
    file.write("    RECaspect =\n")
    file.write("        Switch(\n")
    file.write("            RECaspectNull,\n")
    file.write("            RECaspect,\n")
    file.write("            ImageAspect) [instance: 11, cache: 1];\n")
    file.write("    switchRECenable = Compute(\n")
    file.write("          \"$0 == 0 ? 1 : (($2 == $3) && ($4 == $5)) ? ($1 == 1 ? 2 : 3) : 4\",\n")
    file.write("            RECenable,\n")
    file.write("            switchRenderMode,\n")
    file.write("            RECresolution,\n")
    file.write("            ImageResolution,\n")
    file.write("            RECaspect,\n")
    file.write("            ImageAspect) [instance: 4, cache: 1];\n")
    file.write("    NoRECobject, RECNoRerenderObject, RECNoRerHW, RECRerenderObject = Route(switchRECenable, annoted_object);\n")
    file.write("    Display(\n")
    file.write("        NoRECobject,\n")
    file.write("        camera,\n")
    file.write("        where,\n")
    file.write("        throttle) [instance: 1, cache: 1];\n")
    file.write("    image =\n")
    file.write("        Render(\n")
    file.write("            RECNoRerenderObject,\n")
    file.write("            camera) [instance: 1, cache: 1];\n")
    file.write("    Display(\n")
    file.write("        image,\n")
    file.write("        NULL,\n")
    file.write("        where,\n")
    file.write("        throttle) [instance: 2, cache: 1];\n")
    file.write("    WriteImage(\n")
    file.write("        image,\n")
    file.write("        RECfile,\n")
    file.write("        RECformat) [instance: 1, cache: 1];\n")
    file.write("    rec_where = Display(\n")
    file.write("        RECNoRerHW,\n")
    file.write("        camera,\n")
    file.write("        where,\n")
    file.write("        throttle) [instance: 1, cache: 0];\n")
    file.write("    rec_image = ReadImageWindow(\n")
    file.write("        rec_where) [instance: 1, cache: 1];\n")
    file.write("    WriteImage(\n")
    file.write("        rec_image,\n")
    file.write("        RECfile,\n")
    file.write("        RECformat) [instance: 1, cache: 1];\n")
    file.write("    RECupdateCamera =\n")
    file.write("       UpdateCamera(\n")
    file.write("           camera,\n")
    file.write("           resolution=RECresolution,\n")
    file.write("           aspect=RECaspect) [instance: 2, cache: 1];\n")
    file.write("    Display(\n")
    file.write("        RECRerenderObject,\n")
    file.write("        camera,\n")
    file.write("        where,\n")
    file.write("        throttle) [instance: 1, cache: 1];\n")
    file.write("    RECRerenderObject =\n")
    file.write("       ScaleScreen(\n")
    file.write("           RECRerenderObject,\n")
    file.write("           NULL,\n")
    file.write("           RECresolution,\n")
    file.write("           camera) [instance: 1, cache: 1];\n")
    file.write("    image =\n")
    file.write("        Render(\n")
    file.write("            RECRerenderObject,\n")
    file.write("            RECupdateCamera) [instance: 2, cache: 1];\n")
    file.write("    WriteImage(\n")
    file.write("        image,\n")
    file.write("        RECfile,\n")
    file.write("        RECformat) [instance: 2, cache: 1];\n")
    file.write("}\n")
    file.write("main_Image_3_in_1 = \"Image_3\";\n")
    file.write("main_Image_3_in_3 = \"X24,,Surface\";\n")
    file.write("main_Image_3_in_4 = 1;\n")
    file.write("main_Image_3_in_5 = %s;\n" % image_array1)
    file.write("main_Image_3_in_6 = %s;\n" % image_array2)
    file.write("main_Image_3_in_7 = %s;\n" % image_val)
    file.write("main_Image_3_in_8 = 1376;\n")
    file.write("main_Image_3_in_9 = 0.678;\n")
    file.write("main_Image_3_in_10 = [-0.109685 0.243133 0.963772];\n")
    file.write("main_Image_3_in_11 = NULL;\n")
    file.write("main_Image_3_in_12 = 0;\n")
    file.write("main_Image_3_in_13 = NULL;\n")
    file.write("main_Image_3_in_14 = 1;\n")
    file.write("main_Image_3_in_15 = \"none\";\n")
    file.write("main_Image_3_in_16 = \"none\";\n")
    file.write("main_Image_3_in_17 = NULL;\n")
    file.write("main_Image_3_in_18 = NULL;\n")
    file.write("main_Image_3_in_19 = 0;\n")
    file.write("main_Image_3_in_20 = NULL;\n")
    file.write("main_Image_3_in_21 = NULL;\n")
    file.write("main_Image_3_in_23 = NULL;\n")
    file.write("main_Image_3_in_25 = \"iso\";\n")
    file.write("main_Image_3_in_26 = \"tiff\";\n")
    file.write("main_Image_3_in_27 = NULL;\n")
    file.write("main_Image_3_in_28 = NULL;\n")
    file.write("main_Image_3_in_29 = 0;\n")
    file.write("main_Image_3_in_30 = NULL;\n")
    file.write("main_Image_3_in_31 = NULL;\n")
    file.write("main_Image_3_in_32 = NULL;\n")
    file.write("main_Image_3_in_33 = 1;\n")
    file.write("main_Image_3_in_34 = 0;\n")
    file.write("main_Image_3_in_35 = NULL;\n")
    file.write("main_Image_3_in_36 = 1;\n")
    file.write("main_Image_3_in_37 = NULL;\n")
    file.write("main_Image_3_in_38 = NULL;\n")
    file.write("main_Image_3_in_39 = NULL;\n")
    file.write("main_Image_3_in_40 = NULL;\n")
    file.write("main_Image_3_in_41 = \"rotate\";\n")
    file.write("main_Image_3_in_42 = \"Surface\";\n")
    file.write("main_Image_3_in_43 = NULL;\n")
    file.write("main_Image_3_in_44 = NULL;\n")
    file.write("main_Image_3_in_45 = NULL;\n")
    file.write("main_Image_3_in_46 = NULL;\n")
    file.write("main_Image_3_in_47 = NULL;\n")
    file.write("main_Image_3_in_48 = NULL;\n")
    file.write("main_Image_3_in_49 = NULL;\n")
    file.write("Executive(\"product version 4 3 2\");\n")
    file.write("$sync\n")
    file.write("main();\n")

    # Close the file.
    file.close()
