#Copyright 2009 Diego Duclos
#
#This program is free software: you can redistribute it and/or modify
#it under the terms of the GNU General Public License as published by
#the Free Software Foundation, either version 3 of the License, or
#(at your option) any later version.
#
#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License
#along with this program.  If not, see <http://www.gnu.org/licenses/>.

import urllib
import tarfile
import zipfile
import os.path
import launch
import sys
import shutil
import tempfile
import ConfigParser

from __init__ import Updater
import model.dataFolder

class DataFolderUpdater(Updater):
    def __init__(self):
        self.datafolder = model.dataFolder.dataFolder
        Updater.__init__(self, "http://pyfa.sourceforge.net/files.ini", os.path.join(self.datafolder, "files.ini"))
        
    def updateHandler(self, section, config):
        self.setStatus("checking")
        url = config.get(section, "url")
        destination = config.get(section, "destination")
        self.setStatus("downloading")
        archivePath = urllib.urlretrieve(url, None, self.setProgress)[0]
        self.setStatus("extracting")
        archive = tarfile.open(archivePath, 'r:bz2')
        archive.extractall(os.path.join(self.datafolder, destination))
        archive.close()
        self.setStatus("cleaning")
        os.unlink(archivePath)

def remove_recurse(path):
    if os.path.isdir(path):
        for i in os.listdir(path):
            full_path = path + "/" + i
            if os.path.isdir(full_path):
                remove_recurse(full_path)
            else:
                os.remove(full_path)
        os.rmdir(path)
    elif os.path.isfile(path):
        os.remove(path)
    
class PyfaUpdater(Updater):
    def __init__(self):
        self.datafolder = model.dataFolder.dataFolder
        Updater.__init__(self, "http://pyfa.sourceforge.net/pyfa.ini", os.path.join(self.datafolder, "pyfa.ini"))
        
    def updateHandler(self, section, config):
        #If there's a .git folder, we don't do anything
        self.setStatus("checking")
        if(os.path.exists(os.path.join(launch.basePath, ".git"))):
            return
        
        #Either get the bin or source build
        self.setStatus("downloading")
        if hasattr(sys, "frozen"):
            archivePath = urllib.urlretrieve(config.get(section, "bin"), None, self.setProgress)[0]
            archive = zipfile.ZipFile(archivePath, "r")
        else:
            archivePath = urllib.urlretrieve(config.get(section, "src"), None, self.setProgress)[0]
            archive = tarfile.open(archivePath, "r:gz")
        
        self.setStatus("extracting")
        #Extract stuff
        archive.extractall(tempfile.gettempdir())
        archive.close()
        
        self.setStatus("cleaning")
        pyfaDir = os.path.join(tempfile.gettempdir(), "pyfa")
        for file in os.listdir(pyfaDir):
            remove_recurse(os.path.join(launch.basePath, file))
            shutil.move(os.path.join(pyfaDir, file), os.path.join(launch.basePath, file))
        
        os.rmdir(pyfaDir)
        os.unlink(archivePath)
        
    def isUpToDate(self, sections = "*"):
        if(os.path.exists(os.path.join(launch.basePath, ".git"))):
            return True
        
        try:
            iniNew = urllib.urlopen(self._iniUrl)
            configNew = ConfigParser.ConfigParser()
            configNew.readfp(iniNew)
            if model.dataFolder.pyfaVersion != "git" and configNew.get("pyfa", "version") > model.dataFolder.pyfaVersion:
                return False
            
            return Updater.isUpToDate(self, sections)
        finally:
            iniNew.close()