#Copyright 2009 Diego Duclos
#
#This program is free software: you can redistribute it and/or modify
#it under the terms of the GNU General Public License as published by
#the Free Software Foundation, either version 3 of the License, or
#(at your option) any later version.
#
#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License
#along with this program.  If not, see <http://www.gnu.org/licenses/>.

import category
import re
import marketGroup
import attribute
import effect
import group
import graphic
import overrides
from model.cache import cached_property, cached_query

baseQuery = '''
SELECT
typeID,
groupID,
typeName,
description,
graphicID,
radius,
mass,
volume,
capacity,
portionSize,
raceID,
basePrice,
published,
marketGroupID,
chanceOfDuplicating,
soundID,
categoryID,
iconID,
dataID
FROM invTypes
'''
getItemQueryByName = baseQuery + " WHERE typeName = ?"
getItemQueryByID = baseQuery + " WHERE typeID = ?"
getItemQueryByMarketGroup = baseQuery + " WHERE marketGroupID = ?"
getItemQueryByGroup = baseQuery + " WHERE groupID = ?"
getVariationsQuery = "SELECT typeID, parentTypeID, metaGroupID FROM invmetatypes WHERE parentTypeID = ?"
getBaseQuery = "SELECT parentTypeID FROM invmetatypes WHERE typeID = ? LIMIT 1"
getMetaGroupIDQuery = "SELECT metaGroupID FROM invmetatypes WHERE typeID = ?"

class item(object):
    """
    This class is used to describe items
    """
    @cached_property
    def marketGroup(self): return marketGroup.getMarketGroup(ID=self.marketGroupID)
    
    @cached_property
    def attributes(self):
        attributes = attribute.buildAttributes(self)
        attributes.update(self.baseAttributes)
        return attributes
    
    @cached_property
    def effects(self): return effect.buildEffects(self)
    
    @cached_property
    def group(self): return group.getGroup(ID = self.groupID)
    
    @cached_property
    def icon(self): return graphic.getIconName(self.iconID)
    
    @cached_property
    def category(self): return category.getCategory(ID = self.categoryID)
    
    @cached_property
    def requiredSkills(self):
        requiredSkills = {}
        for i in range(5):
            i = i + 1
            skillID, skillLevel = None, None
            try:
                skillID = self.attributes['requiredSkill' + str(i)]
                skillLevel = self.attributes['requiredSkill' + str(i) + 'Level']
            except KeyError: continue
            if skillID == None or skillLevel == None: continue
            item = getItem(ID=skillID.getModifiedValue())
            requiredSkills[item] = skillLevel
        return requiredSkills
    
    @cached_property
    def race(self):
        if self.marketGroup and self.marketGroup.name == "ORE":
            race = "ore"
        else: 
            #Getting the actual race for ships can be tricky so we'll rely on the races of the needed skills
            skillRaces = {1: False,
                          2: False,
                          4: False,
                          8: False}
            
            for skill, level in self.requiredSkills.iteritems():
                race = skill.raceID
                if race != None:
                    skillRaces[race] = True
            race = getRace(self.raceID, skillRaces)
        return race
    
    @cached_property
    def metaGroupID(self): return getMetaGroupID(self.ID)
    
    def __init__(self, ID, name, description, marketGroupID = None, groupID = None, iconID = None, raceID = None, published = 0, ignoreOverrides = False):
        '''
        Constructor
        @param ID: the ID of the item
        @param name: the name of the item
        @param description: the description of the item
        @param marketGroupID: the ID of the marketgroup of the item [optional]
        @param groupID: the id of the group of the item [optional]
        @param graphicID: the graphicID of the item [optional]
        '''
        self.iconID = iconID
        self.ID = ID
        self.name = name
        self.description = description
        self.marketGroupID = marketGroupID
        self.groupID = groupID
        self.raceID = raceID
        self.published = published
        if not ignoreOverrides:
            overrideMod = overrides.getOverrideModule(name, ("basevalues",))
            overrides.override(self, overrideMod)
                
    def __hash__(self):
        return self.ID
    
    def __eq__(self, other):
        return self.__hash__() == other.__hash__()
    
    def __ne__(self, other):
        return not self.__eq__(other)
    
    def __str__(self, level = 0, showDescription = True):
        '''
        Returns a textual representation of this item
        @param level: the indentation level
        @param showDescription: wether to show the description of this item [can be rather long]
        '''
        formatting = ((level * 2) * " ") + "{0:15} = {1}\n"
        newline = ((level * 2) * " ") + "\n"
        ret = ((level * 2) * " ") + "[item]\n" + formatting.format("itemID", self.ID) + formatting.format("name", self.name)
        if self.attributes != None:
            for name, value in self.attributes.iteritems():
                ret = ret + formatting.format(name, value)

        if self.effects != None:
            ret = ret + newline + "effects:\n"
            for effect in self.effects:
                ret = ret + effect.__str__(level + 1) + "\n"

        if self.requiredSkills != None:
            ret = ret + newline + "required skills:\n"
            for skill, lvl in self.requiredSkills.iteritems():
                ret = ret +  formatting.format(skill.name, str(lvl))
            
        if showDescription:
            ret = ret + newline + "description:\n" + self.description.encode('ascii', 'replace')

        if self.marketGroup != None:
            ret = ret + newline * 2 + "market group info:\n" + self.marketGroup.__str__(level + 1)

        return ret
    
    def clearModifiedAttributes(self):
        '''
        Clear all modified values of all the attributes of this item
        '''
        for key in self.attributes.keys():
            attribute = self.attributes[key]
            if attribute.item != self: del self.attributes[key]
            else: attribute.clear()
            
    def getModifiedAttribute(self, attribute):
        '''
        Gets the modified value of the given attribute
        @param attribute: the attrbute to get the value for
        '''
        try:
            return self.attributes[attribute].getModifiedValue()
        except KeyError:
            return None
        
    def getType(self):
        types = set()
        if self.effects:
            for effect in self.effects:
                for type in effect.type:
                    types.add(type)
                    
        return types

def getMetaGroupID(itemID):
    if itemID == None: return
    c = cached_query(getMetaGroupIDQuery, (itemID,))
    if c: return c[0][0]
    
def getVariations(item):
    '''
    @param item: the item to get all the variations for
    @param ...: Args to pass to createItemFromRow
    '''
    c = cached_query(getVariationsQuery, (item.ID,))
    variations = []
    for row in c:
        i = getItem(ID = row[0])
        variations.append(i)
        
    return variations

def getBase(variationID):
    '''
    @param variation: the variation to get the base item for
    @param ...: Args to pass to createItemFromRow
    '''
    c = cached_query(getBaseQuery, (variationID,))
    if not c: return None
    row = c[0]
    return getItem(ID = row[0])
    
def getItemsByGroup(marketGroup = None, group = None):     
    '''
    returns a list with all items belonging to a certain market group or group.
    Either marketGroup or group must be specified NOT BOTH
    @param marketGroup: the market group to get items for
    @param group: the group to get items for
    '''
    if not isinstance(marketGroup, int) and marketGroup != None: marketGroup = marketGroup.ID
    if not isinstance(group, int) and group != None: group = group.ID
    if marketGroup != None and group == None:
        c = cached_query(getItemQueryByMarketGroup, (marketGroup,))
    elif group != None: c = cached_query(getItemQueryByGroup, (group,))
    else: return None
    itemList = []
    for row in c:
        itemList.append(createItemFromRow(row))
        
    return itemList

def getItem(ID = None, name = None):
    '''
    Get an item from database. Either name or ID must be specified NOT BOTH
    @param ID: the ID of the item to fetch
    @param name: the name of the item to fetch
    '''
    if ID != None and name == None:
        c = cached_query(getItemQueryByID, (ID,))
    elif name != None:
        c = cached_query(getItemQueryByName, (name,))
    else: return None
    if not c: return None
    row = c[0]
    if row == None: return None
    return createItemFromRow(row)
    
def searchItem(nameLike, marketGroup = None, group = None):
    '''
    Search an item in the database
    @param nameLike: what to look for. This can be in a few different formats.
    - A simple string without any *, % or _ signs. In this case wildcards will be placed before and after it.
    - A string with * signs in it. The *s will be replaced with %s to be valid SQL syntax
    - A string with % and/or _ signs in it. The strign will be passed directly to SQL
    @param marketGroup: the marketGroup the item belongs to, if any
    @param group: the group the item belongs to, if any
    @param ...: Args to pass to createItemFromRow
    '''
    #Build the query for searching
    query = baseQuery + " WHERE typeName LIKE ?"
    if marketGroup != None:
        query = query + " AND marketGroupID = ?"
    if group != None:
        query = query + " AND groupID = ?"
    
    #Check if the string contains * signs we need to convert to %
    if "*" in nameLike:
        nameLike = nameLike.replace("*", "%")
    #Check for % or _ signs, if there aren't any we'll add a % at start and another one at end
    elif not "%" in nameLike and not "_" in nameLike:
        nameLike = "%" + nameLike + "%"
        
    argList = filter(None, [nameLike, marketGroup, group])
    c = cached_query(query, tuple(argList))
    resultList = []
    addedItems = set()
    for row in c:
        i = createItemFromRow(row)
        addedItems.add(i.name)
        resultList.append(i)
        
    #Make up a preg
    nameLike = re.escape(nameLike).replace("\\%", ".*").replace("\\_", ".")
    for itemName in overrides.getItemList():
        if not itemName in addedItems:
            pattern = re.compile(nameLike)
            if re.match(pattern, itemName):
                addedItems.add(itemName)
                resultList.append(createItemFromRow((None, None, itemName, None, None, None, None, None, None, None, None, None, None, None)))

    return resultList

def createItemFromRow((itemID, groupID, itemName, description, graphicID, radius, mass, volume, capacity, portionSize, raceID, basePrice, published, marketGroupID, chanceOfDuplicating, soundID, categoryID, iconID, dataID), itemToComplete = None):
    '''
    Create an item from it's database row information
    @param itemID: the ID of the item
    @param groupID: the ID of the group the item is in
    @param itemName: the name of the item
    @param description: the description of the item
    @param graphicID: the graphicID of the item
    @param radius: the radius of the item
    @param mass: the mass of the item
    @param volume: the volume of the item
    @param capacity: the capacity of the item
    @param portionSize: the number of units required for reprocessing of the item
    @param raceID: the ID of the race of this item
    @param basePrice: the basePrice of the item
    @param published: whether this item has been published
    @param marketGroupID: the ID of the marketgroup of the item
    @param chanceOfDuplicating: the chance to duplicate the item
    @param soundID: the soundID of the item
    @param categoryID: the categoryID of the item
    @param iconID: the iconID of the item
    @param dataID: the dataID of the item
    '''
    if itemToComplete:
        itemToComplete.__init__(itemID, itemName, description, marketGroupID, groupID, iconID, raceID, published)
        currItem = itemToComplete
    else:
        currItem = item(itemID, itemName, description, marketGroupID, groupID, iconID, raceID, published)
    
    currItem.baseAttributes = {'radius' : attribute.basicAttribute(currItem, "radius", radius), 
                               'mass' : attribute.basicAttribute(currItem, "mass", mass), 
                               'volume' : attribute.basicAttribute(currItem, "volume", volume),
                               'capacity' : attribute.basicAttribute(currItem, "capacity", capacity),
                               'portionSize' : attribute.basicAttribute(currItem, "portionSize", portionSize), 
                               'basePrice' : attribute.basicAttribute(currItem, "basePrice", basePrice)}
    
    return currItem
    
if __name__ == "__main__":
    print("This module defines the following:\n")
    for name in vars().keys():
        print("{0:20} = {1}".format(name, vars()[name]))

def getRace(raceID, raceList = None):
    '''
    Get the race of an item
    @param raceID: the RaceID of the item in question.
    @param raceList: the list of races this item belongs to [pirate faction stuff]
    '''
    if raceList == None:
        raceList = {1: False, 2 : False, 4: False, 8: False}
    
    if raceID: raceList[raceID] = True
    if raceList[1] and raceList[8]: #Guristas
        return "guristas"
    elif raceList[1] and raceList[4]: #Sansha
        return "sansha"
    elif raceList[8] and raceList[2]: #Angel / Serp
        if raceID == 2: #Angel
            return "angel"
        else: #Serpentis
            return "serpentis"
    elif raceList[2] and raceList[4]: #Blood
        return "blood"
    elif raceList[1]: #Caldari
        return "caldari"
    elif raceList[2]: #Minmatar
       return "minmatar"
    elif raceList[4]: #Amarr
        return "amarr"
    elif raceList[8]: #Gallente
        return "gallente"
    else:
        return "unknown"