#Copyright 2009 Diego Duclos
#
#This program is free software: you can redistribute it and/or modify
#it under the terms of the GNU General Public License as published by
#the Free Software Foundation, either version 3 of the License, or
#(at your option) any later version.
#
#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License
#along with this program.  If not, see <http://www.gnu.org/licenses/>.
import eveapi
import item
import dataFolder
import collections
import copy
import group
import category
dummyAll0 = None
dummyAll5 = None

class character(object):
    '''
    This class is used to represent characters
    '''
    #Skills should be a list of tuples with the skillbook item as first element of the tuple and the level of the skill as second element
    def __init__(self, name, skills=None, userID = None, apiKey = None, charID = None):
        '''
        Constructor
        @param name: the name of the character
        @param skills: a dictionary containing the skills of the character. (Defaults to None)
        @param implants: a dictionary containing the implants plugged into the character. The key should be the slot, the value the implant (defaults to None)
        @param boosters: a dictionary containing the boosters plugged into the character. The key should be the slot, the value the implant (defaults to None)
        @param userID: the userID of the account the character is on for fetching from eve api (defaults to None)
        @param apiKey: the apiKey of the account the character is on for fetching from eve api (defaults to None)
        @param charID: the ID of the character for fetching from eve api (defaults to None)
        '''
        self.name = name
        self.skills = skills or {}
        self.userID = userID
        self.apiKey = apiKey
        self.charID = charID

    def __str__(self, level=0):
        '''
        Returns a textual representation of this class
        @param level: indentation level
        '''
        formatting = ((level * 2) * " ") + "{0:35} = {1}\n"
        newline = ((level * 2) * " ") + "\n"
        str = ((level * 2) * " ") + "[character]\n"
        if self.skills != None:
            for skill, level in self.skills.iteritems():
                str = str + formatting.format(skill.name, level)

        if self.implants != None:
            i = ""
            for n in self.implants:
                i = i + n.name

            str = str + formatting.format("implants", i)

        return str
    
    def getSkills(self):
        '''
        Fetch the skills of this character from eve api
        '''
        api = eveapi.EVEAPIConnection()
        auth = api.auth(userID=self.userID, apiKey=self.apiKey)
        char = auth.character(self.charID)
        sheet = char.CharacterSheet()
        
        skillDict = {}
        for typeID, level in sheet.skills.Select("typeID", "level"):
            i = item.getItem(ID=typeID)
            if i: skillDict[i] = level
        
        self.skills = skillDict
        
    def getSkill(self, name):
        for skill, level in self.skills.iteritems():
            if skill.name == name:
                return skill, level
            
        return None, 0
    
    def __getstate__(self):
        skills = {}
        for skill, level in self.skills.iteritems():
            skills[skill.ID] = level
            
        
        return {'name' : self.name,
                'userID' : self.userID,
                'apiKey' : self.apiKey,
                'charID' : self.charID,
                'skills' : skills}

    def __setstate__(self, state):
        skills = {}
        for skillID, level in state["skills"].iteritems():
            skills[item.getItem(skillID)] = level
        
        del state["skills"]
        
        if "implants" in state: del state["implants"]
        if "boosters" in state: del state["boosters"]
        self.__init__(skills = skills, **state)
        
def getCharacterList(userID, apiKey):
    '''
    Returns a list containing all the characters on the account passed
    @param userID: the userID of the account to get the characters for
    @param apiKey: the apiKey of the account
    '''
    api = eveapi.EVEAPIConnection() 
    auth = api.auth(userID=userID, apiKey=apiKey)
    result = auth.account.Characters()

    list = []
    for char in result.characters:
        list.append((char.characterID, char.name))

    return list

def allFive():
    return 5

def allZero():
    return 0

#Make sure to return the same dict across calls
savedData = {}
def loadCharacters():
    '''
    Load character data from hard disk (location: ~/.pyfa/characters)
    If no file is found then a profile with one character with all skills at five and another with all at zero is returned
    '''
    #Loading all the skills for the All 5 character takes a while
    #Let's ensure we only ever do it once
    global savedData, dummyAll0, dummyAll5
    savedData = savedData or dataFolder.pickleFileToObj("characters") or savedData
    if not "All 0" in savedData:
        if not dummyAll0: dummyAll0 = character("All 0")
        savedData["All 0"] = dummyAll0
    if not "All 5" in savedData:
        if not dummyAll5:
            dummyAll5 = character("All 5")
            skillsCat = category.getCategory(name="Skill")
            skillsGroups = group.getGroupsByCategory(skillsCat.ID)
            skills = []
            for skillGrp in skillsGroups:
                skills += item.getItemsByGroup(group = skillGrp)
            for skill in skills:
                if skill.published == 1:
                    dummyAll5.skills[skill] = 5
        savedData["All 5"] = dummyAll5
    return savedData
        
def saveCharacters(chars):
    '''
    Save character data to hard disk (location: ~/.pyfa/characters
    '''
    del chars["All 5"]
    del chars["All 0"]
    return dataFolder.pickleObjToFile(chars, "characters")

if __name__ == "__main__":
    print("This module defines the following:\n")
    for name in vars().keys():
        print("{0:20} = {1}".format(name, vars()[name]))