#Copyright 2009 Diego Duclos
#
#This program is free software: you can redistribute it and/or modify
#it under the terms of the GNU General Public License as published by
#the Free Software Foundation, either version 3 of the License, or
#(at your option) any later version.
#
#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License
#along with this program.  If not, see <http://www.gnu.org/licenses/>.
import sql
import math
import graphic
import overrides
from model.cache import cached_property
import traceback

#The cache we use to not have to relookup things all the time
cache = {}

buildAttributeQueryByTypeID = '''
SELECT
dta.attributeID,
dta.value,
dat.attributeName,
dat.attributeCategory,
dat.description,
dat.maxAttributeID,
dat.attributeIdx,
dat.graphicID,
dat.chargeRechargeTimeID,
dat.defaultValue,
dat.published,
dat.unitID,
dat.displayName,
dat.stackable,
dat.highIsGood,
dat.iconID,
dat.dataID,
dac.categoryName
FROM dgmtypeattribs dta
LEFT JOIN dgmattribs dat ON dta.attributeID = dat.attributeID
LEFT JOIN dgmAttributeCategories dac ON dac.categoryID = dat.attributeCategory
WHERE dta.typeID = ?
'''

from model.cache import cached_query
#Note: Custom attributes should NOT have a value but only a modifiedValue 
#so they can be cleared and if included in the attributes dict of an item 
#they should have a _ in front their name
class basicAttribute(object):
    """
    This is the class all attributes inherit from. It should be used for custom attributes not from the database
    """
    def __init__(self, item, name, value, modifiedValue = None, highIsGood = 0, displayName = None):
        '''
        Constructor
        @param item: the item this attribute is for
        @param name: the name of this attribute
        @param value: the "base" value of this attribute.
        @param modifiedValue: the "modified" value of this attribute.
        '''
        self.displayName = displayName
        self.highIsGood = highIsGood
        self.item = item
        self.unmodifiedValue = value
        self.modifiedValue = modifiedValue
        self.name = name
        self.overrideValue = None
        self.preIncreases = 0
        self.postIncreases = 0
        self.muls = 1
        self.penalizedMuls = []
        self.sources = set()
        
    def __str__(self, level = 0):
        '''
        Returns the string form of this class
        @param level: The level of indentation that should be used when returning
        '''
        #When getting printed we only return the value, most other data is superfluous for printing purpose
        return ((level * 2) * " ") + "[attribute] " + str(self.getModifiedValue())
    
    def addSource(self, source): self.sources.add(source)
    
    def getModifiedValue(self):
        '''
        Returns the modifiedValue if it is not none, if it is returns the unmodified value
        '''
        if self.overrideValue != None: return self.overrideValue
        if self.modifiedValue != None: val = self.modifiedValue
        else: val = self.unmodifiedValue
        if val == None and self.preIncreases == 0 and self.postIncreases == 0\
         and self.muls == 1 and len(self.penalizedMuls) == 0: return None
        if val == None or isinstance(val, int) or isinstance(val, float):
            val = (val or 0)
            #Apply preincreases
            val += self.preIncreases
            #Apply muls
            val *= self.muls
            #Check if there's any stacking penalty mechanics pending and apply em
            if self.penalizedMuls:
                l = self.penalizedMuls                
                #Sort bonusses biggest to smallest
                #Positive and negative effects (eg sensor boosters and remote sensor dampeners) 
                #Have different stacks even though they effect the same attribute
                #We have to seperate them.
                
                pos = lambda val: val > 1
                neg = lambda val: val < 1
                lGood = filter(self.highIsGood and pos or neg, l)
                lBad = filter(self.highIsGood and neg or pos, l)
                lGood.sort(reverse=self.highIsGood)
                lBad.sort(reverse=not self.highIsGood)
                    
                for i in range(len(lGood)):
                    bonus = lGood[i]
                    val *= 1 + (bonus - 1) * math.exp(- math.pow(i, 2) / 7.1289)

                for i in range(len(lBad)):
                    bonus = lBad[i]
                    val *= 1 + (bonus - 1) * math.exp(- math.pow(i, 2) / 7.1289)
                val += self.postIncreases
        return val
        
    def clear(self):
        self.overrideValue = None
        self.modifiedValue = None
        del self.penalizedMuls[:]
        self.preIncreases = 0
        self.postIncreases = 0
        self.muls = 1
        self.sources.clear()
            
    def increase(self, bonus, position = "pre"):
        if position == "post": self.postIncreases += bonus
        elif position == "pre": self.preIncreases += bonus
        
    def multiply(self, bonus, penalized = False):
        if penalized: self.penalizedMuls.append(bonus)
        else: self.muls *= bonus
        
class attribute(basicAttribute):
    '''
    Class used to build attributes that come from the database
    '''
    @cached_property
    def icon(self): return graphic.getIconName(self.iconID)
    def __init__(self, item, (ID, value, name, attributeCategory, description, maxAttributeID, attributeIdx, graphicID, chargeRechargeTimeID, defaultValue, published, unitID, displayName, stackable, highIsGood, iconID, dataID, categoryName), ignoreOverrides = False):
        '''
        Constructor
        @param item: the item this attribute is for
        @param ID: the ID of this attribute [from database]
        @param value: the value of this attribute [from database]
        @param name: the name [from database]
        @param attributeCategory: the attributeCategory of this attribute [from database]
        @param description: the description [from database]
        @param maxAttributeID: the maxAttributeID [from database]
        @param attributeIdx: the attributeIdx [from database]
        @param graphicID: the graphicID [from database]
        @param chargeRechargeTimeID: the chargeRechargeTimeID [from database]
        @param defaultValue: the default value [from database]
        @param published: whether this attribute has been published [from database]
        @param unitID: the unitID of this attribute [from database]
        @param displayName: the display name of this attribute [from database]
        @param stackable: whether this attribute is stackable [from database]
        @param highIsGood: whether higher is better [from database]
        @param iconID: the iconID [from database]
        @param dataID: the dataID [from database]
        @param categoryName: the categoryName [from database]
        '''
        basicAttribute.__init__(self, item, name, value, None, highIsGood, displayName)
        self.ID = ID
        self.description = description
        self.iconID = iconID
        self.defaultValue = defaultValue
        self.published = published
        self.unitID = unitID
        self.stackable = stackable
        self.categoryID = attributeCategory
        self.categoryName = categoryName
        if not ignoreOverrides:
            overrideMod = overrides.getOverrideModule(item.name, ("attributes", name))
            if overrideMod:
                for key in dir(overrideMod): setattr(self, key, getattr(overrideMod, key))
            
def buildAttributes(item, attributeDict = None):
    '''
    Builds the attributes of the passed item
    @param item: the item to build for
    @param attributeDict: the dictionary to put the attributes in. If this is not passed a new dictionary is created
    '''
    if attributeDict == None: attributeDict = {}
    c = cached_query(buildAttributeQueryByTypeID, (item.ID,))

    for row in c:
        attributeName = row[2]
        attr = attribute(item, row)
        attributeDict[attributeName] = attr
    for attrName in overrides.getModList(item.name, ("attributes",)):
        if not attrName in attributeDict:
            attributeDict[attrName] = attribute(item, (None, None, attrName, None, None, None, None, None, None,
                                                       None, None, None, None, None, None, None, None, None))
    return attributeDict

if __name__ == "__main__":
    print("This module defines the following:\n")
    for name in vars().keys():
        print("{0:20} = {1}".format(name, vars()[name]))
