#!/usr/bin/env python
"""Unit test for shellwords.py

(C) Copyright 2002 by Hartmut Goebel <h.goebel@crazy-compilers.com>
"""
__author__ = "Hartmut Goebel <h.goebel@crazy-compilers.com>"
__version__ = "0.1"
__date__ = "2002-11-17"
__copyright__ = "(C) Copyright 2002 by Hartmut Goebel"
__license__ = "GPL"

import os, sys
import unittest
import re
import shellwords

testdir = os.path.dirname(sys.argv[0])
testscript   = os.path.join(testdir, 'test_shellwords.sh')
testdatafile = os.path.join(testdir, 'test_data.txt')

unmatched_quote_pattern = re.compile(
    """unexpected EOF while looking for matching `(.)'"""
    )
unexpected_eof_pattern = re.compile(
    """syntax error: unexpected end of file"""
    )

test_strings = [
    'x y z',   # 0
    # leading whitespace
    ' x y z',  # 1
    '  x y z', # 2
    # trailing whitespace
    'x y z ',  #1 
    'x y z  ', #2
    # leading and trailing whitespace
    ' x y z ',    # 1,1
    '  x y z ',   # 2,1
    ' x y z  ',   # 1,2
    '  x y z  ',  # 2,2
    # empty:
    '',
    ' ',
    '  ',
    ]

test_strings.extend([
    line.replace(' ', '\t') for line in test_strings
    ])


def getstatusoutput(cmd):
    """Return (status, output) of executing cmd in a shell.

    This is a variation of commands.getstatusoutput which uses
    popen3 to catch stderr 
    """
    import popen2
    child = popen2.Popen3('{ ' + cmd + '; } 2>&1', 1) # capture stderr
    sts = child.wait()
    if sts is None: sts = 0
    if sts == 0:
        text = child.fromchild.read()
    else:
        text = child.childerr.read()
    if text.endswith('\n'): text = text[:-1]
    return sts, text


class ShellwordsTest(unittest.TestCase):
    def __parseArgs(self, text):
        if text.startswith("#"):
            args = [ x[1:]  # strip of hashmark
                     for x in text.splitlines()]
        else:
            assert text == ''
            args = []
        return args

    def __parseError(self, line, text):
        reason = None
        if reason is None:
            match = unmatched_quote_pattern.search(text)
            if match:
                if match.group(1) == "'":
                    reason = shellwords.UnmatchedSingleQuoteError
                else:
                    reason = shellwords.UnmatchedDoubleQuoteError
        if reason is None:
            match = unexpected_eof_pattern.search(text)
            if match:
                reason = shellwords.EOFError
        if reason is None:
            print 'failed for unknown reason:'
            print '\t', line
            print '\t', text
        return reason

    def setUp(self):
        self.okay_data = okay_data = [] # establish shortcut
        self.fail_data = fail_data = [] # establish shortcut

        lines = test_strings[:]
        lines.extend(
            open(testdatafile).read().splitlines() )
        
        for line in lines:
            if line.startswith('#'): continue
            status, output = getstatusoutput(
                '%s --line-by-line %s' % (testscript, line)
                )
            if status == 0: # okay
                args = self.__parseArgs(output)
                okay_data.append((line, args))
            else:
                reason = self.__parseError(line, output)
                fail_data.append((line, reason))

    def testShellwords(self):
        for line, args in self.okay_data:
            #print 'testing:', line
            words = shellwords.shellwords(line)
            self.assertEqual(words, args,
                             "\n%s\nis:\t%r\nshould:\t%r" % (line, words,args))

    def testUnmatchedQuotes(self):
        for line, reason in self.fail_data:
            if issubclass(reason, shellwords.EOFError):
                #print 'testing:', reason, line
                self.assertRaises(reason, shellwords.shellwords, line)
            else:
                print 'skipping:', reason, line

if __name__ == "__main__":
    unittest.main()
