"""In memory store for fact base.
"""
from anysets import Set, ImmutableSet, protect
from identity import rdf, Namespace
empty_set = ImmutableSet()

class Population:
	"""Provides a reference implementation of the Population protocol.

	This protocol may be implemented over your favourite RDF API to
	enable use of GraphPath expressions.

	Only the rdf_type attribute and the match() and values() methods
	are required (these methods have direct equivalents in typical API's).

	The type of the subject, predicate and object (or value)
	arguments should be whatever type is used
	for these in underlying RDF API.

	It should also be possible to use elementary python types for values,
	including strings at least.  The adapter should convert these to
	and from the underlying RDF API value node type.

	Additionally, the add() method and mapping protocol
	may be implemented to support some experimental GraphPath features,
	but this is not required.

	For the optional mapping protocol, the key is a subject resource and
	the value is a set of (property, object) tuples for that object.

	The protocol is implemented here using a Resource type from
	identity.py over an in-memory model made up of
	two structure.

	* The _tree is a mapping of subjects to mappings of predicates to lists of objects.

	* The _matches dictionary maps (predicate object) tuples to a set of subjects.

	"""
	def __init__(self):
                self._tree = {}
		self._matches = {}

	rdf_type = rdf.type

        def match( self, prop, value ):
		"""Return a (possibly immutable) set of subjects for
		statements (*, prop, value)."""
		try:
			return protect(self._matches[prop, value])
		except KeyError:
			return empty_set

        def values( self, subj, prop):
		"""Return a (possibly immutable) set of objects for
		statements (subj, prop, *)."""
		try:
			return ImmutableSet( self._tree[subj][prop] )
		except KeyError:
			return empty_set

	def add( self, subj, prop, value):
		"""Assert the statement (subj, prop, value)."""
		pred = prop, value
		if pred in self._matches:
			match = self._matches[pred]
			if subj in match:
				return # shortcut if statement already asserted
		else:
			match = self._matches[pred] = Set()

		match.add(subj)

		if subj in self._tree:
			desc = self._tree[subj]
			if prop in desc:
				desc[prop].append( value )
			else:
				desc[prop] = [value]
		else:
			self._tree[subj] = { prop : [value] }

	def update( self, other ):
		"""Copy all statements from another population."""
		for subj in other:
			for prop, value in other[subj]:
				self.add(subj, prop, value)

	def __setstate__(self, state):
		"""Unpickle the object, covert from older pickles as necessary."""
		try:
			self._tree = state['_tree']
			self._matches = state['_matches']

		except KeyError:
			# (slowly) reconstruct missing members
			print "NOTICE: coverting older Population pickle"
	                self.__init__()
			matches = state['_matches']
			for prop, value in matches:
				for subj in matches[prop, value]:
					self.add( subj, prop, value)
			print "Done."

	def __contains__(self, subj):
		"""Test whether the resource object serves as a subject
		in this population."""
		return subj in self._tree

	def __iter__(self):
		"""Iterate the (unique) subjects of all statements."""
		return iter(self._tree)

	def __getitem__(self, subj):
		"""Return a sequence of (predicate, object)
		tuples for a given subject."""
		desc = self._tree[subj]
		return [ (prop, value) for prop in desc for value in desc[prop]]

	def __repr__(self):
		return "%s(%s)" % (self.__class__.__name__, id(self))

	__str__ = __repr__

def test01():
	assert pop.match(pop.rdf_type, ex.Female) == Set([ex.julie])

def test02():
	assert pop.values( ex.julie, pop.rdf_type ) == Set([ex.Female])

def test03():
	assert ex.julie in pop

def test04():
	assert Set(pop[ex.julie]) == Set([(pop.rdf_type, ex.Female)])

if __name__ == "__main__":
	import sys
	from testrunner import run

	# setup tet data
	pop = Population()
	ex = Namespace("#")
	pop.add( ex.julie, pop.rdf_type, ex.Female)

	run(globals())
