<?php

/**
* @package   s9e\TextFormatter
* @copyright Copyright (c) 2010-2019 The s9e Authors
* @license   http://www.opensource.org/licenses/mit-license.php The MIT License
*/
namespace s9e\TextFormatter\Plugins\MediaEmbed\Configurator\Collections;

class CachedDefinitionCollection extends SiteDefinitionCollection
{
	/** {@inheritdoc} */
	protected $items = [
		'abcnews'=>['attributes'=>[],'example'=>'https://abcnews.go.com/WNN/video/dog-goes-wild-when-owner-leaves-22936610','extract'=>['!abcnews\\.go\\.com/(?:video/embed\\?id=|[^/]+/video/[^/]+-)(?\'id\'\\d+)!'],'homepage'=>'https://abcnews.go.com/','host'=>['abcnews.go.com'],'iframe'=>['src'=>'//abcnews.go.com/video/embed?id={@id}'],'name'=>'ABC News','scrape'=>[],'tags'=>['news']],
		'amazon'=>['attributes'=>[],'example'=>['https://www.amazon.ca/gp/product/B00GQT1LNO/','https://www.amazon.co.jp/gp/product/B003AKZ6I8/','https://www.amazon.co.uk/dp/B00EO4NN5C/','https://www.amazon.com/dp/B002MUC0ZY','https://www.amazon.com/The-BeerBelly-200-001-80-Ounce-Belly/dp/B001RB2CXY/','https://www.amazon.com/gp/product/B00ST0KGCU/','https://www.amazon.de/Netgear-WN3100RP-100PES-Repeater-integrierte-Steckdose/dp/B00ET2LTE6/','https://www.amazon.es/Microsoft-Sculpt-Ergonomic-Desktop-L5V-00011/dp/B00FO10ZK0/','https://www.amazon.fr/Vans-Authentic-Baskets-mixte-adulte/dp/B005NIKPAY/','https://www.amazon.in/Vans-Unisex-Classic-Loafers-Moccasins/dp/B01I3LWMVC/','https://www.amazon.it/Super-Maxi-Pot-de-Nutella/dp/B0090GJ8VM/','https://www.amazon.com/dp/B0018CDWLS/'],'extract'=>['#/(?:dp|gp/product)/(?\'id\'[A-Z0-9]+)#','#amazon\\.(?:co\\.)?(?\'tld\'ca|de|es|fr|in|it|jp|uk)#'],'homepage'=>'https://affiliate-program.amazon.com/','host'=>['amazon.ca','amazon.co.uk','amazon.co.jp','amazon.com','amazon.de','amazon.es','amazon.fr','amazon.in','amazon.it'],'iframe'=>['height'=>240,'src'=>'//ws-<xsl:choose><xsl:when test="@tld=\'jp\'">fe</xsl:when><xsl:when test="@tld and contains(\'desfrinituk\',@tld)">eu</xsl:when><xsl:otherwise>na</xsl:otherwise></xsl:choose>.assoc-amazon.com/widgets/cm?l=as1&amp;f=ifr&amp;o=<xsl:choose><xsl:when test="@tld=\'ca\'">15&amp;t=<xsl:value-of select="$AMAZON_ASSOCIATE_TAG_CA"/></xsl:when><xsl:when test="@tld=\'de\'">3&amp;t=<xsl:value-of select="$AMAZON_ASSOCIATE_TAG_DE"/></xsl:when><xsl:when test="@tld=\'es\'">30&amp;t=<xsl:value-of select="$AMAZON_ASSOCIATE_TAG_ES"/></xsl:when><xsl:when test="@tld=\'fr\'">8&amp;t=<xsl:value-of select="$AMAZON_ASSOCIATE_TAG_FR"/></xsl:when><xsl:when test="@tld=\'in\'">31&amp;t=<xsl:value-of select="$AMAZON_ASSOCIATE_TAG_IN"/></xsl:when><xsl:when test="@tld=\'it\'">29&amp;t=<xsl:value-of select="$AMAZON_ASSOCIATE_TAG_IT"/></xsl:when><xsl:when test="@tld=\'jp\'">9&amp;t=<xsl:value-of select="$AMAZON_ASSOCIATE_TAG_JP"/></xsl:when><xsl:when test="@tld=\'uk\'">2&amp;t=<xsl:value-of select="$AMAZON_ASSOCIATE_TAG_UK"/></xsl:when><xsl:otherwise>1&amp;t=<xsl:value-of select="$AMAZON_ASSOCIATE_TAG"/></xsl:otherwise></xsl:choose>&amp;asins=<xsl:value-of select="@id"/>','style'=>['background'=>'url(https://www.amazon.com/favicon.ico) center no-repeat'],'width'=>120],'name'=>'Amazon Product','parameters'=>['AMAZON_ASSOCIATE_TAG'=>['title'=>'Amazon Associate tag (.com)'],'AMAZON_ASSOCIATE_TAG_CA'=>['title'=>'Amazon Associate tag (.ca)'],'AMAZON_ASSOCIATE_TAG_DE'=>['title'=>'Amazon Associate tag (.de)'],'AMAZON_ASSOCIATE_TAG_ES'=>['title'=>'Amazon Associate tag (.es)'],'AMAZON_ASSOCIATE_TAG_FR'=>['title'=>'Amazon Associate tag (.fr)'],'AMAZON_ASSOCIATE_TAG_IN'=>['title'=>'Amazon Associate tag (.in)'],'AMAZON_ASSOCIATE_TAG_IT'=>['title'=>'Amazon Associate tag (.it)'],'AMAZON_ASSOCIATE_TAG_JP'=>['title'=>'Amazon Associate tag (.co.jp)'],'AMAZON_ASSOCIATE_TAG_UK'=>['title'=>'Amazon Associate tag (.co.uk)']],'scrape'=>[]],
		'anchor'=>['attributes'=>[],'example'=>'https://anchor.fm/aconversationwith/episodes/Answering-Voicemails-A-Conversation-With-You--2-e6e7lg','extract'=>['@anchor.fm/[-\\w]+/episodes/(?:[-\\w]+-)(?\'id\'\\w+)(?![-\\w])@'],'homepage'=>'https://anchor.fm/','host'=>['anchor.fm'],'iframe'=>['height'=>102,'max-width'=>700,'src'=>'//anchor.fm/embed/episodes/x-{@id}','width'=>'100%'],'name'=>'Anchor','scrape'=>[],'tags'=>['podcasts']],
		'audioboom'=>['attributes'=>[],'example'=>'https://audioboom.com/posts/2493448-robert-patrick','extract'=>['!audioboo(?:\\.f|m\\.co)m/(?:boo|post)s/(?\'id\'\\d+)!'],'homepage'=>'https://audioboom.com/','host'=>['audioboo.fm','audioboom.com'],'iframe'=>['height'=>150,'max-width'=>700,'src'=>'//audioboom.com/posts/{@id}/embed/v3','width'=>'100%'],'name'=>'Audioboom','scrape'=>[],'tags'=>['podcasts']],
		'audiomack'=>['attributes'=>[],'choose'=>['otherwise'=>['iframe'=>['height'=>252,'max-width'=>900,'src'=>'https://www.audiomack.com/embed/song/{@id}','width'=>'100%']],'when'=>['iframe'=>['height'=>400,'max-width'=>900,'src'=>'https://www.audiomack.com/embed/album/{@id}','width'=>'100%'],'test'=>'@mode=\'album\'']],'example'=>['https://www.audiomack.com/song/your-music-fix/jammin-kungs-remix-1','https://www.audiomack.com/album/chance-the-rapper/acid-rap'],'extract'=>['!audiomack\\.com/(?\'mode\'album|song)/(?\'id\'[-\\w]+/[-\\w]+)!'],'homepage'=>'https://www.audiomack.com/','host'=>['audiomack.com'],'name'=>'Audiomack','scrape'=>[],'tags'=>['music']],
		'bandcamp'=>['attributes'=>[],'example'=>['https://proleter.bandcamp.com/album/curses-from-past-times-ep','https://proleter.bandcamp.com/track/downtown-irony','https://therunons.bandcamp.com/track/still-feel'],'extract'=>[],'homepage'=>'https://bandcamp.com/','host'=>['bandcamp.com'],'iframe'=>['height'=>400,'src'=>'//bandcamp.com/EmbeddedPlayer/size=large/minimal=true/<xsl:choose><xsl:when test="@album_id">album=<xsl:value-of select="@album_id"/><xsl:if test="@track_num">/t=<xsl:value-of select="@track_num"/></xsl:if></xsl:when><xsl:otherwise>track=<xsl:value-of select="@track_id"/></xsl:otherwise></xsl:choose>','width'=>400],'name'=>'Bandcamp','scrape'=>[['extract'=>['!/album=(?\'album_id\'\\d+)!'],'match'=>['!bandcamp\\.com/album/.!']],['extract'=>['!"album_id":(?\'album_id\'\\d+)!','!"track_num":(?\'track_num\'\\d+)!','!/track=(?\'track_id\'\\d+)!'],'match'=>['!bandcamp\\.com/track/.!']]],'tags'=>['music']],
		'bbcnews'=>['attributes'=>['id'=>['filterChain'=>['stripslashes']]],'example'=>['https://www.bbc.com/news/video_and_audio/must_see/42847060/calls-to-clean-off-banksy-mural-in-hull','https://www.bbc.co.uk/news/av/world-africa-48141248/fifteen-year-old-nigerian-builds-small-scale-construction-machines'],'extract'=>['@bbc\\.co(?:m|\\.uk)/news/av/embed/(?\'id\'[-\\\\\\w/]+)@'],'homepage'=>'https://www.bbc.com/news/video_and_audio/headlines/','host'=>['bbc.co.uk','bbc.com'],'iframe'=>['src'=>'//www.bbc.com<xsl:choose><xsl:when test="starts-with(@id,\'av/\')">/news/<xsl:value-of select="@id"/>/embed</xsl:when><xsl:when test="starts-with(@playlist,\'/news/\')and contains(@playlist,\'A\')"><xsl:value-of select="substring-before(@playlist,\'A\')"/>/embed</xsl:when><xsl:otherwise>/news/av/embed/<xsl:value-of select="@id"/></xsl:otherwise></xsl:choose>'],'name'=>'BBC News','scrape'=>[['extract'=>['@bbc\\.co(?:m|\\.uk)\\\\?/news\\\\?/av\\\\?/embed\\\\?/(?\'id\'[-\\\\\\w/]+)@'],'match'=>['@bbc\\.co(?:m|\\.uk)/news/(?:av(?!/embed)|video_and_audio)/\\w@']]],'tags'=>['news']],
		'bitchute'=>['attributes'=>[],'example'=>'https://www.bitchute.com/video/y9AejeZ2vD0/','extract'=>['@bitchute\\.com/video/(?\'id\'\\w+)@'],'homepage'=>'https://www.bitchute.com/','host'=>['bitchute.com'],'iframe'=>['src'=>'https://www.bitchute.com/embed/{@id}/'],'name'=>'BitChute','scrape'=>[],'tags'=>['videos']],
		'bleacherreport'=>['attributes'=>[],'example'=>'https://bleacherreport.com/articles/2687569-how-bell-brown-have-shattered-records-on-steelers-historic-playoff-run','extract'=>[],'homepage'=>'https://bleacherreport.com/videos','host'=>['bleacherreport.com'],'iframe'=>['height'=>320,'src'=>'//bleacherreport.com/video_embed?id={@id}&library=video-cms','width'=>320],'name'=>'Bleacher Report videos','scrape'=>[['extract'=>['!id="video-(?\'id\'[-\\w]+)!','!video_embed\\?id=(?\'id\'[-\\w]+)!'],'match'=>['!/articles/.!']]],'tags'=>['sports']],
		'break'=>['attributes'=>[],'example'=>'https://www.break.com/video/video-game-playing-frog-wants-more-2278131','extract'=>['!break\\.com/video/.*-(?\'id\'\\d+)$!'],'homepage'=>'https://www.break.com/','host'=>['break.com'],'iframe'=>['height'=>280,'src'=>'//break.com/embed/{@id}','width'=>464],'name'=>'Break','scrape'=>[],'tags'=>['entertainment']],
		'brightcove'=>['attributes'=>['bckey'=>['required'=>false]],'example'=>['https://players.brightcove.net/219646971/default_default/index.html?videoId=4815779906001','https://link.brightcove.com/services/player/bcpid4773906090001?bckey=AQ~~,AAAAAA0Xi_s~,r1xMuE8k5Nyz5IyYc0Hzhl5eZ5cEEvlm&bctid=4815779906001'],'extract'=>['@link\\.brightcove\\.com/services/player/bcpid(?\'bcpid\'\\d+).*?bckey=(?\'bckey\'[-,~\\w]+).*?bctid=(?\'bctid\'\\d+)@','@players\\.brightcove\\.net/(?\'bcpid\'\\d+)/.*?videoId=(?\'bctid\'\\d+)@'],'homepage'=>'https://www.brightcove.com/','host'=>['link.brightcove.com','players.brightcove.net'],'iframe'=>['src'=>'https://<xsl:choose><xsl:when test="@bckey">link.brightcove.com/services/player/bcpid<xsl:value-of select="@bcpid"/>?bckey=<xsl:value-of select="@bckey"/>&amp;bctid=<xsl:value-of select="@bctid"/>&amp;secureConnections=true&amp;secureHTMLConnections=true&amp;autoStart=false&amp;height=360&amp;width=640</xsl:when><xsl:otherwise>players.brightcove.net/<xsl:value-of select="@bcpid"/>/default_default/index.html?videoId=<xsl:value-of select="@bctid"/></xsl:otherwise></xsl:choose>'],'name'=>'Brightcove','scrape'=>[],'tags'=>['videos']],
		'cbsnews'=>['attributes'=>[],'choose'=>['otherwise'=>['flash'=>['flashvars'=>'si=254&contentValue={@id}','padding-height'=>40,'src'=>'//i.i.cbsi.com/cnwk.1d/av/video/cbsnews/atlantis2/cbsnews_player_embed.swf']],'when'=>[['iframe'=>['src'=>'https://www.cbsnews.com/embed/videos/{@id}/'],'test'=>'contains(@id, \'-\')'],['flash'=>['flashvars'=>'pType=embed&si=254&pid={@pid}','padding-height'=>38,'src'=>'//www.cbsnews.com/common/video/cbsnews_player.swf'],'test'=>'@pid']]],'example'=>'https://www.cbsnews.com/video/is-carbonated-water-a-healthy-option/','extract'=>['#cbsnews\\.com/videos?/(?!watch/)(?\'id\'[-\\w]+)#','#cbsnews\\.com/video/watch/\\?id=(?\'id\'\\d+)#'],'homepage'=>'https://www.cbsnews.com/video/','host'=>['cbsnews.com'],'name'=>'CBS News Video','scrape'=>[],'tags'=>['news']],
		'cnbc'=>['attributes'=>[],'example'=>['https://video.cnbc.com/gallery/?video=3000269279','https://www.cnbc.com/video/2019/06/05/carplay-android-auto-alexa-big-tech-gets-into-our-cars.html'],'extract'=>['!cnbc\\.com/gallery/\\?video=(?\'id\'\\d+)!'],'homepage'=>'https://www.cnbc.com/video/','host'=>['cnbc.com'],'iframe'=>['src'=>'https://player.cnbc.com/p/gZWlPC/cnbc_global?playertype=synd&byGuid={@id}&size=640_360'],'name'=>'CNBC','scrape'=>[['extract'=>['!byGuid=(?\'id\'\\d+)!'],'match'=>['!cnbc\\.com/video/20\\d\\d/\\d\\d/\\d\\d/\\w!']]],'tags'=>['news']],
		'cnn'=>['attributes'=>[],'example'=>['https://edition.cnn.com/videos/tv/2015/06/09/airplane-yoga-rachel-crane-ts-orig.cnn','https://us.cnn.com/video/data/2.0/video/bestoftv/2013/10/23/vo-nr-prince-george-christening-arrival.cnn.html'],'extract'=>['!cnn.com/videos/(?\'id\'.*\\.cnn)!','!cnn\\.com/video/data/2\\.0/video/(?\'id\'.*\\.cnn)!'],'homepage'=>'https://edition.cnn.com/video/','host'=>['cnn.com'],'iframe'=>['src'=>'//fave.api.cnn.io/v1/fav/?video={@id}&customer=cnn&edition=international&env=prod'],'name'=>'CNN','scrape'=>[],'tags'=>['news']],
		'cnnmoney'=>['attributes'=>[],'example'=>'https://money.cnn.com/video/technology/2014/05/20/t-twitch-vp-on-future.cnnmoney/','extract'=>['!money\\.cnn\\.com/video/(?\'id\'.*\\.cnnmoney)!'],'homepage'=>'https://money.cnn.com/video/','host'=>['money.cnn.com'],'iframe'=>['height'=>360,'src'=>'//money.cnn.com/.element/ssi/video/7.0/players/embed.player.html?videoid=video/{@id}&width=560&height=360','width'=>560],'name'=>'CNNMoney','scrape'=>[],'tags'=>['news']],
		'collegehumor'=>['attributes'=>[],'example'=>'https://www.collegehumor.com/video/1181601/more-than-friends','extract'=>['!collegehumor\\.com/(?:video|embed)/(?\'id\'\\d+)!'],'homepage'=>'https://www.collegehumor.com/','host'=>['collegehumor.com'],'iframe'=>['height'=>369,'src'=>'//www.collegehumor.com/e/{@id}','width'=>600],'name'=>'CollegeHumor','scrape'=>[],'tags'=>['entertainment']],
		'comedycentral'=>['attributes'=>[],'example'=>['https://www.cc.com/video-clips/uu5qz4/key-and-peele-dueling-hats','https://www.comedycentral.com/video-clips/uu5qz4/key-and-peele-dueling-hats','https://tosh.cc.com/video-clips/aet4lh/rc-car-crash'],'extract'=>[],'homepage'=>'https://www.cc.com/','host'=>['cc.com','comedycentral.com'],'iframe'=>['src'=>'//media.mtvnservices.com/embed/{@id}'],'name'=>'Comedy Central','scrape'=>[['extract'=>['!(?\'id\'mgid:arc:(?:episode|video):[.\\w]+:[-\\w]+)!'],'match'=>['!c(?:c|omedycentral)\\.com/(?:full-episode|video-clip)s/!']]],'tags'=>['entertainment']],
		'coub'=>['attributes'=>[],'example'=>'https://coub.com/view/6veusoty','extract'=>['!coub\\.com/view/(?\'id\'\\w+)!'],'homepage'=>'https://coub.com/','host'=>['coub.com'],'iframe'=>['src'=>'//coub.com/embed/{@id}'],'name'=>'Coub','scrape'=>[],'tags'=>['videos']],
		'dailymotion'=>['attributes'=>[],'example'=>['https://dai.ly/x5e9eog','https://www.dailymotion.com/video/x5e9eog'],'extract'=>['!dai\\.ly/(?\'id\'[a-z0-9]+)!i','!dailymotion\\.com/(?:live/|swf/|user/[^#]+#video=|(?:related/\\d+/)?video/)(?\'id\'[a-z0-9]+)!i','!start=(?\'t\'\\d+)!'],'homepage'=>'https://www.dailymotion.com/','host'=>['dai.ly','dailymotion.com'],'iframe'=>['src'=>'//www.dailymotion.com/embed/video/<xsl:value-of select="@id"/><xsl:if test="@t">?start=<xsl:value-of select="@t"/></xsl:if>'],'name'=>'Dailymotion','oembed'=>['url'=>'http://www.dailymotion.com/services/oembed'],'scrape'=>[],'source'=>'http://www.dailymotion.com/doc/api/player.html','tags'=>['videos']],
		'democracynow'=>['attributes'=>[],'example'=>['https://www.democracynow.org/2014/7/2/dn_at_almedalen_week_at_swedens','https://www.democracynow.org/blog/2015/3/13/part_2_bruce_schneier_on_the','https://www.democracynow.org/shows/2006/2/20','https://www.democracynow.org/2015/5/21/headlines','https://m.democracynow.org/stories/15236'],'extract'=>['!democracynow.org/(?:embed/)?(?\'id\'(?:\\w+/)?\\d+/\\d+/\\d+(?:/\\w+)?)!'],'homepage'=>'https://www.democracynow.org/','host'=>['democracynow.org'],'iframe'=>['src'=>'//www.democracynow.org/embed/<xsl:choose><xsl:when test="contains(@id,\'/headlines\')">headlines/<xsl:value-of select="substring-before(@id,\'/headlines\')"/></xsl:when><xsl:when test="starts-with(@id,\'2\')">story/<xsl:value-of select="@id"/></xsl:when><xsl:when test="starts-with(@id,\'shows/\')">show/<xsl:value-of select="substring-after(@id,\'/\')"/></xsl:when><xsl:otherwise><xsl:value-of select="@id"/></xsl:otherwise></xsl:choose>'],'name'=>'Democracy Now!','scrape'=>[['extract'=>['!democracynow\\.org/(?\'id\'(?:\\w+/)?\\d+/\\d+/\\d+(?:/\\w+)?)\' rel=\'canonical!'],'match'=>['!m\\.democracynow\\.org/stories/\\d!']]]],
		'dumpert'=>['attributes'=>[],'example'=>'https://www.dumpert.nl/mediabase/6622577/4652b140/r_mi_gaillard_doet_halloween_prank.html','extract'=>['!dumpert\\.nl/mediabase/(?\'id\'\\d+[/_]\\w+)!'],'homepage'=>'https://www.dumpert.nl/','host'=>['dumpert.nl'],'iframe'=>['src'=>'//www.dumpert.nl/embed/{translate(@id,\'_\',\'/\')}/'],'name'=>'dumpert','scrape'=>[],'tags'=>['.nl','entertainment']],
		'eighttracks'=>['attributes'=>[],'example'=>['https://8tracks.com/lovinq/headphones-in-world-out','https://8tracks.com/lovinq/4982023'],'extract'=>['!8tracks\\.com/[-\\w]+/(?\'id\'\\d+)(?=#|$)!'],'homepage'=>'https://8tracks.com/','host'=>['8tracks.com'],'iframe'=>['height'=>400,'src'=>'//8tracks.com/mixes/{@id}/player_v3_universal','width'=>400],'name'=>'8tracks','scrape'=>[['extract'=>['!eighttracks://mix/(?\'id\'\\d+)!'],'match'=>['!8tracks\\.com/[-\\w]+/\\D!']]],'tags'=>['music']],
		'espn'=>['attributes'=>[],'example'=>['https://www.espn.com/video/clip?id=17474659','https://www.espn.com/espnw/video/13887284/kyrgios-angry-code-violation-almost-hitting-ref','https://broadband.espn.go.com/video/clip?id=17481969'],'extract'=>['#video/(?:clip(?:\\?id=|/_/id/))?(?\'id\'\\d+)#'],'homepage'=>'https://www.espn.com/','host'=>['espn.com','espn.go.com'],'iframe'=>['src'=>'//www.espn.com/core/video/iframe?id={@id}'],'name'=>'ESPN','scrape'=>[],'tags'=>['sports']],
		'facebook'=>['attributes'=>[],'example'=>['https://www.facebook.com/FacebookDevelopers/posts/10151471074398553','https://www.facebook.com/video/video.php?v=10150451523596807','https://www.facebook.com/photo.php?fbid=10152476416772631','https://www.facebook.com/ign/videos/10153762113196633/','https://www.facebook.com/southamptonfc/videos/vb.220396037973624/1357764664236750/'],'extract'=>['@/(?!(?:apps|developers|graph)\\.)[-\\w.]*facebook\\.com/(?:[/\\w]+/permalink|(?!pages/|groups/).*?)(?:/|fbid=|\\?v=)(?\'id\'\\d+)(?=$|[/?&#])@','@facebook\\.com/(?\'user\'[.\\w]+)/(?\'type\'post|video)s?/@','@facebook\\.com/video/(?\'type\'post|video)\\.php@'],'homepage'=>'https://www.facebook.com/','host'=>['facebook.com'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/facebook.min.html#{@type}{@id}'],'name'=>'Facebook','scrape'=>[],'tags'=>['social']],
		'flickr'=>['attributes'=>[],'example'=>['https://www.flickr.com/photos/8757881@N04/2971804544/lightbox/','https://flic.kr/8757881@N04/2971804544','https://flic.kr/p/5wBgXo'],'extract'=>['@flickr\\.com/photos/[^/]+/(?\'id\'\\d+)@','@flic\\.kr/(?!p/)[^/]+/(?\'id\'\\d+)@'],'homepage'=>'https://www.flickr.com/','host'=>['flic.kr','flickr.com'],'iframe'=>['height'=>500,'src'=>'https://www.flickr.com/photos/_/{@id}/player/','width'=>500],'name'=>'Flickr','scrape'=>[['extract'=>['@flickr\\.com/photos/[^/]+/(?\'id\'\\d+)@'],'match'=>['@flic\\.kr/p/(?\'short\'\\w+)@'],'url'=>'https://www.flickr.com/photo.gne?rb=1&short={@short}']],'tags'=>['images']],
		'foxnews'=>['attributes'=>[],'example'=>'https://video.foxnews.com/v/3592758613001/reddit-helps-fund-homemade-hot-sauce-venture/','extract'=>['!video\\.foxnews\\.com/v/(?\'id\'\\d+)!'],'homepage'=>'https://www.foxnews.com/','host'=>['video.foxnews.com'],'iframe'=>['src'=>'//video.foxnews.com/v/video-embed.html?video_id={@id}'],'name'=>'Fox News','scrape'=>[],'tags'=>['news']],
		'foxsports'=>['attributes'=>[],'example'=>'https://www.foxsports.com/home/video/1132379203958','extract'=>[],'homepage'=>'https://www.foxsports.com/video','host'=>['foxsports.com'],'iframe'=>['src'=>'//player.foxfdm.com/sports/embed-iframe.html?videourl=http://link.theplatform.com/s/BKQ29B/{@id}?mbr=true&assetTypes=Video&feed=Fox%20Sports%20All&autoplay=false&policy=68249219&sdk=PDK%205.7.15'],'name'=>'FOX Sports','scrape'=>[['extract'=>['@BKQ29B/(?\'id\'\\w+)@'],'match'=>['@/video/\\d@']]],'tags'=>['sports']],
		'funnyordie'=>['attributes'=>[],'example'=>'https://www.funnyordie.com/videos/bf313bd8b4/murdock-with-keith-david','extract'=>['!funnyordie\\.com/videos/(?\'id\'[0-9a-f]+)!'],'homepage'=>'https://www.funnyordie.com/','host'=>['funnyordie.com'],'iframe'=>['src'=>'//www.funnyordie.com/embed/{@id}'],'name'=>'Funny or Die','scrape'=>[],'source'=>'http://support.funnyordie.com/discussions/problems/918-embed-user-videos-widget','tags'=>['entertainment']],
		'gamespot'=>['attributes'=>[],'example'=>['https://www.gamespot.com/destiny/videos/destiny-the-moon-trailer-6415176/','https://www.gamespot.com/events/game-crib-tsm-snapdragon/gamecrib-extras-cooking-with-dan-dinh-6412922/','https://www.gamespot.com/videos/beat-the-pros-pax-prime-2013/2300-6414307/'],'extract'=>['!gamespot\\.com.*?/(?:events|videos)/.*?-(?\'id\'\\d+)/(?:[#?].*)?$!'],'homepage'=>'https://www.gamespot.com/','host'=>['gamespot.com'],'iframe'=>['src'=>'//www.gamespot.com/videos/embed/{@id}/'],'name'=>'Gamespot','scrape'=>[],'tags'=>['gaming']],
		'gametrailers'=>['attributes'=>[],'example'=>'https://www.gametrailers.com/videos/view/pop-fiction/102300-Metal-Gear-Solid-3-Still-in-a-Dream','extract'=>[],'homepage'=>'https://www.gametrailers.com/','host'=>['gametrailers.com'],'iframe'=>['src'=>'//<xsl:choose><xsl:when test="starts-with(@id,\'mgid:\')">media.mtvnservices.com/embed/<xsl:value-of select="@id"/></xsl:when><xsl:otherwise>embed.gametrailers.com/embed/<xsl:value-of select="@id"/>?embed=1&amp;suppressBumper=1</xsl:otherwise></xsl:choose>'],'name'=>'GameTrailers','scrape'=>[['extract'=>['!embed/(?\'id\'\\d+)!'],'match'=>['!gametrailers\\.com/(?:full-episode|review|video)s/!']]],'tags'=>['gaming']],
		'getty'=>['attributes'=>['height'=>['defaultValue'=>360,'filterChain'=>['#uint']],'width'=>['defaultValue'=>640,'filterChain'=>['#uint']]],'example'=>['https://gty.im/3232182','https://www.gettyimages.com/detail/3232182','https://www.gettyimages.com/detail/news-photo/the-beatles-travel-by-coach-to-the-west-country-for-some-news-photo/3232182','https://www.gettyimages.co.uk/detail/3232182'],'extract'=>['!gty\\.im/(?\'id\'\\d+)!','!gettyimages\\.[.\\w]+/detail(?=/).*?/(?\'id\'\\d+)!','!#[-\\w]*picture-id(?\'id\'\\d+)$!'],'homepage'=>'https://www.gettyimages.com/','host'=>['gettyimages.be','gettyimages.cn','gettyimages.co.jp','gettyimages.co.uk','gettyimages.com','gettyimages.com.au','gettyimages.de','gettyimages.dk','gettyimages.es','gettyimages.fr','gettyimages.ie','gettyimages.it','gettyimages.nl','gettyimages.pt','gettyimages.se','gty.im'],'iframe'=>['height'=>'{@height}','src'=>'//embed.gettyimages.com/embed/{@id}?et={@et}&tld=com&sig={@sig}&caption=false&ver=1','width'=>'{@width}'],'name'=>'Getty Images','scrape'=>[['extract'=>['!"height":[ "]*(?\'height\'\\d+)!','!"width":[ "]*(?\'width\'\\d+)!','!\\bid[=:][\'"]?(?\'et\'[-=\\w]+)!','!\\bsig[=:][\'"]?(?\'sig\'[-=\\w]+)!'],'match'=>['//'],'url'=>'http://embed.gettyimages.com/preview/{@id}']],'tags'=>['images']],
		'gfycat'=>['attributes'=>['height'=>['defaultValue'=>360,'filterChain'=>['#uint']],'width'=>['defaultValue'=>640,'filterChain'=>['#uint']]],'example'=>['https://gfycat.com/SereneIllfatedCapybara','https://giant.gfycat.com/SereneIllfatedCapybara.gif'],'extract'=>['#gfycat\\.com/(?!gaming|reactions|stickers|gifs/tag)(?:gifs/detail/|ifr(?:ame)?/)?(?\'id\'\\w+)#'],'homepage'=>'https://gfycat.com/','host'=>['gfycat.com'],'iframe'=>['height'=>'{@height}','padding-height'=>44,'src'=>'//gfycat.com/ifr/{@id}','width'=>'{@width}'],'name'=>'Gfycat','scrape'=>[['extract'=>['!/ifr/(?\'id\'\\w+)!'],'match'=>['#gfycat\\.com/(?!gaming|reactions|stickers|gifs/tag)(?:gifs/detail/|ifr(?:ame)?/)?[a-z]#'],'url'=>'https://gfycat.com/ifr/{@id}'],['extract'=>['!"height":(?\'height\'\\d+)!','!"width":(?\'width\'\\d+)!'],'match'=>['//'],'url'=>'https://api.gfycat.com/v1/oembed?url=https://gfycat.com/{@id}']],'tags'=>['images']],
		'gifs'=>['attributes'=>['height'=>['defaultValue'=>360,'filterChain'=>['#uint']],'width'=>['defaultValue'=>640,'filterChain'=>['#uint']]],'example'=>['https://gifs.com/gif/zm4DLy','https://j.gifs.com/Y6YZoO.gif'],'extract'=>['!gifs\\.com/(?:gif/)?(?\'id\'\\w+)!'],'homepage'=>'https://gifs.com/','host'=>['gifs.com'],'iframe'=>['height'=>'{@height}','src'=>'//gifs.com/embed/{@id}','width'=>'{@width}'],'name'=>'Gifs.com','scrape'=>[['extract'=>['!meta property="og:image:width" content="(?\'width\'\\d+)!','!meta property="og:image:height" content="(?\'height\'\\d+)!'],'match'=>['//'],'url'=>'https://gifs.com/gif/{@id}']],'tags'=>['images']],
		'gist'=>['attributes'=>[],'example'=>['https://gist.github.com/s9e/0ee8433f5a9a779d08ef','https://gist.github.com/6806305','https://gist.github.com/s9e/6806305/ad88d904b082c8211afa040162402015aacb8599'],'extract'=>['!gist\\.github\\.com/(?\'id\'(?:\\w+/)?[\\da-f]+(?:/[\\da-f]+)?)!'],'homepage'=>'https://gist.github.com/','host'=>['github.com'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>180,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','scrolling'=>'','src'=>'https://s9e.github.io/iframe/2/gist.min.html#{@id}','width'=>'100%'],'name'=>'GitHub Gist','scrape'=>[]],
		'globalnews'=>['attributes'=>[],'example'=>['https://globalnews.ca/video/1647385/mark-channels-his-70s-look/','https://globalnews.ca/video/rd/350539843530/?jwsource=cl'],'extract'=>['!globalnews\\.ca/video/(?\'id\'\\d+)!'],'homepage'=>'https://globalnews.ca/','host'=>['globalnews.ca'],'iframe'=>['src'=>'//globalnews.ca/video/embed/{@id}/'],'name'=>'Global News','scrape'=>[['extract'=>['!globalnews\\.ca/video/(?\'id\'\\d+)!'],'match'=>['!globalnews\\.ca/video/rd/!']]],'tags'=>['.ca','news']],
		'gofundme'=>['attributes'=>[],'example'=>'https://www.gofundme.com/2p37ao','extract'=>['@gofundme\\.com/(?\'id\'\\w+)(?![^#?])@'],'homepage'=>'https://www.gofundme.com/','host'=>['gofundme.com'],'iframe'=>['height'=>207,'padding-height'=>208,'src'=>'//www.gofundme.com/mvc.php?route=widgets/mediawidget&fund={@id}&image=1&coinfo=1','width'=>349],'name'=>'GoFundMe','scrape'=>[],'tags'=>['fundraising']],
		'googledrive'=>['attributes'=>[],'example'=>'https://drive.google.com/file/d/0B_4NRUjxLBejNjVmeG5MUzA3Q3M/view?usp=sharing','extract'=>['!drive\\.google\\.com/.*?(?:file/d/|id=)(?\'id\'[-\\w]+)!'],'homepage'=>'https://drive.google.com','host'=>['drive.google.com'],'iframe'=>['height'=>480,'src'=>'//drive.google.com/file/d/{@id}/preview','width'=>640],'name'=>'Google Drive','scrape'=>[],'tags'=>['documents','images','videos']],
		'googleplus'=>['attributes'=>['name'=>['filterChain'=>['urldecode']]],'example'=>['https://plus.google.com/+TonyHawk/posts/C5TMsDZJWBd','https://plus.google.com/106189723444098348646/posts/V8AojCoTzxV'],'extract'=>['!//plus\\.google\\.com/(?:u/\\d+/)?(?:\\+(?\'name\'[^/]+)|(?\'oid\'\\d+))/posts/(?\'pid\'\\w+)!'],'homepage'=>'https://plus.google.com/','host'=>['plus.google.com'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>240,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/googleplus.min.html#<xsl:choose><xsl:when test="@oid"><xsl:value-of select="@oid"/></xsl:when><xsl:otherwise>+<xsl:value-of select="@name"/></xsl:otherwise></xsl:choose>/posts/<xsl:value-of select="@pid"/>','width'=>450],'name'=>'Google+','scrape'=>[],'source'=>'https://developers.google.com/+/web/embedded-post/','tags'=>['social']],
		'googlesheets'=>['attributes'=>[],'choose'=>['otherwise'=>['iframe'=>['height'=>500,'src'=>'
					<xsl:text>https://docs.google.com/spreadsheets/d/</xsl:text>
					<xsl:value-of select="@id"/>
					<xsl:choose>
						<xsl:when test="starts-with(@id, \'e/\')">/pubhtml?widget=true&amp;headers=false</xsl:when>
						<xsl:otherwise>/preview</xsl:otherwise>
					</xsl:choose>
					<xsl:text>#gid=</xsl:text>
					<xsl:value-of select="@gid"/>
				','style'=>['resize'=>'vertical'],'width'=>'100%']],'when'=>['iframe'=>['height'=>372,'src'=>'https://docs.google.com/spreadsheets/d/{@id}/pubchart?oid={@oid}&format=interactive','width'=>600],'test'=>'@type=\'chart\'']],'example'=>['https://docs.google.com/spreadsheets/d/1f988o68HDvk335xXllJD16vxLBuRcmm3vg6U9lVaYpA','https://docs.google.com/spreadsheets/u/1/d/e/2PACX-1vQ7iYECa_ziPWygKBf9Og-PkQl5-gdnFRU-jP-z1FyjLfLXMAdRJN1J5Lx31NjnCq5lkRcL-T0Idh4_/pubhtml','https://docs.google.com/spreadsheets/d/e/2PACX-1vSbbciMU7t5dCB8auk-nyK-p-25TJjgUrzBuMCAiq_2ZDqoMSFAlZCctIC7y3TXS71ys_MRdoAU4Jqr/pubchart?oid=1729348495'],'extract'=>['@docs\\.google\\.com/spreadsheet(?:/ccc\\?key=|(?:[^e]/)+)(?\'id\'(?:e/)?[-\\w]+)@','@oid=(?\'oid\'\\d+)@','@#gid=(?\'gid\'\\d+)@','@/pub(?\'type\'chart)@'],'homepage'=>'https://www.google.com/sheets/about/','host'=>['docs.google.com'],'name'=>'Google Sheets','scrape'=>[],'tags'=>['documents']],
		'hudl'=>['attributes'=>['id'=>['required'=>false]],'example'=>['https://www.hudl.com/video/59dfca6090eeac23f0eef838','https://www.hudl.com/athlete/2067184/highlights/163744377','https://www.hudl.com/video/3/323679/57719969842eb243e47883f8'],'extract'=>['!hudl\\.com/athlete/(?\'athlete\'\\d+)/highlights/(?\'highlight\'[\\da-f]+)!','!hudl\\.com/video/\\d+/(?\'athlete\'\\d+)/(?\'highlight\'[\\da-f]+)!','@hudl\\.com/video/(?\'id\'\\w+)(?![\\w/])@'],'homepage'=>'https://www.hudl.com/','host'=>['hudl.com'],'iframe'=>['src'=>'//www.hudl.com/embed/<xsl:choose><xsl:when test="@id">video/<xsl:value-of select="@id"/></xsl:when><xsl:otherwise>athlete/<xsl:value-of select="@athlete"/>/highlights/<xsl:value-of select="@highlight"/></xsl:otherwise></xsl:choose>'],'name'=>'Hudl','scrape'=>[['extract'=>['!hudl\\.com/video/\\d+/(?\'athlete\'\\d+)/(?\'highlight\'[\\da-f]+)!','@hudl\\.com/video/(?\'id\'\\w+)(?![\\w/])@'],'match'=>['!hudl\\.com/v/!']]],'tags'=>['sports']],
		'hulu'=>['attributes'=>[],'example'=>'https://www.hulu.com/watch/484180','extract'=>[],'homepage'=>'https://www.hulu.com/','host'=>['hulu.com'],'iframe'=>['src'=>'https://secure.hulu.com/embed/{@id}'],'name'=>'Hulu','scrape'=>[['extract'=>['!eid=(?\'id\'[-\\w]+)!'],'match'=>['!hulu\\.com/watch/!']]]],
		'ign'=>['attributes'=>[],'example'=>'https://www.ign.com/videos/2013/07/12/pokemon-x-version-pokemon-y-version-battle-trailer','extract'=>['!(?\'id\'https?://.*?ign\\.com/videos/.+)!i'],'homepage'=>'https://www.ign.com/videos/','host'=>['ign.com'],'iframe'=>['height'=>263,'src'=>'//widgets.ign.com/video/embed/content.html?url={@id}','width'=>468],'name'=>'IGN','scrape'=>[],'tags'=>['gaming']],
		'imdb'=>['attributes'=>[],'example'=>['https://www.imdb.com/video/imdb/vi2482677785/','https://www.imdb.com/title/tt2294629/videoplayer/vi2482677785'],'extract'=>['!imdb\\.com/[/\\w]+/vi(?\'id\'\\d+)!'],'homepage'=>'https://www.imdb.com/','host'=>['imdb.com'],'iframe'=>['src'=>'//www.imdb.com/video/imdb/vi{@id}/imdb/embed?autoplay=false&width=640'],'name'=>'IMDb','scrape'=>[],'tags'=>['movies']],
		'imgur'=>['attributes'=>[],'example'=>['https://imgur.com/AsQ0K3P','https://imgur.com/a/9UGCL','https://imgur.com/gallery/9UGCL','https://i.imgur.com/u7Yo0Vy.gifv','https://i.imgur.com/UO1UrIx.mp4'],'extract'=>['@imgur\\.com/(?\'id\'a/\\w+)@','@i\\.imgur\\.com/(?\'id\'\\w{5,7})[lms]?\\.@','@imgur\\.com/(?\'id\'\\w+)(?![\\w./])@'],'homepage'=>'https://imgur.com/','host'=>['imgur.com'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>400,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{var d=e.data.split(\' \');style.height=d[0]+\'px\';style.width=d[1]+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/imgur.min.html#<xsl:if test="@type=\'album\'">a/</xsl:if><xsl:value-of select="@id"/>','width'=>542],'name'=>'Imgur','scrape'=>[['extract'=>['@data-id="(?\'id\'[\\w/]+)"@'],'match'=>['@imgur\\.com/(?![art]/|user/)(?\'path\'(?:gallery/)?\\w+)(?![\\w.])@'],'url'=>'https://api.imgur.com/oembed.xml?url=/{@path}']],'tags'=>['images']],
		'indiegogo'=>['attributes'=>[],'example'=>'https://www.indiegogo.com/projects/gameheart-redesigned','extract'=>['!indiegogo\\.com/projects/(?\'id\'[-\\w]+)!'],'homepage'=>'https://www.indiegogo.com/','host'=>['indiegogo.com'],'iframe'=>['height'=>445,'src'=>'//www.indiegogo.com/project/{@id}/embedded','width'=>222],'name'=>'Indiegogo','scrape'=>[],'tags'=>['fundraising']],
		'instagram'=>['attributes'=>[],'example'=>['https://instagram.com/p/gbGaIXBQbn/','https://www.instagram.com/tv/BkQjCfsBIzi/'],'extract'=>['!instagram\\.com/(?:p|tv)/(?\'id\'[-\\w]+)!'],'homepage'=>'https://instagram.com/','host'=>['instagram.com'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>540,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/instagram.min.html#{@id}','style'=>['background'=>'url(https://www.instagram.com/static/images/ico/favicon.svg/fc72dd4bfde8.svg) no-repeat 50% 50% / 50%'],'width'=>540],'name'=>'Instagram','scrape'=>[],'source'=>'http://help.instagram.com/513918941996087','tags'=>['social']],
		'internetarchive'=>['attributes'=>['height'=>['defaultValue'=>360,'filterChain'=>['#uint']],'id'=>['filterChain'=>['htmlspecialchars_decode']],'width'=>['defaultValue'=>640,'filterChain'=>['#uint']]],'choose'=>['otherwise'=>['iframe'=>['height'=>'{@height}','src'=>'https://archive.org/embed/{@id}','width'=>'{@width}']],'when'=>['iframe'=>['height'=>170,'max-width'=>640,'src'=>'https://archive.org/embed/{@id}&list_height=150','width'=>'100%'],'test'=>'contains(@id,\'playlist=1\')']],'example'=>['https://archive.org/details/BillGate99','https://archive.org/details/DFTS2014-05-30'],'extract'=>[],'homepage'=>'https://archive.org/','host'=>['archive.org'],'name'=>'Internet Archive','scrape'=>[['extract'=>['!meta property="twitter:player" content="https://archive.org/embed/(?\'id\'[^/"]+)!','!meta property="og:video:width" content="(?\'width\'\\d+)!','!meta property="og:video:height" content="(?\'height\'\\d+)!'],'match'=>['!archive\\.org/(?:details|embed)/!']]]],
		'izlesene'=>['attributes'=>[],'example'=>'https://www.izlesene.com/video/tam-1-kilo-yap-gram-gram-kapisma/10412573','extract'=>['!izlesene\\.com/video/[-\\w]+/(?\'id\'\\d+)!'],'homepage'=>'https://www.izlesene.com/','host'=>['izlesene.com'],'iframe'=>['src'=>'//www.izlesene.com/embedplayer/{@id}?autoplay=0'],'name'=>'İzlesene','scrape'=>[],'tags'=>['.tr']],
		'jwplatform'=>['attributes'=>[],'example'=>['https://content.jwplatform.com/players/X6tRZpKj-7Y21S9TB.html','https://content.jwplatform.com/previews/YfTSAegE-L0l35Tsd'],'extract'=>['!jwplatform\\.com/\\w+/(?\'id\'[-\\w]+)!'],'homepage'=>'https://www.jwplayer.com/products/jwplatform/','host'=>['jwplatform.com'],'iframe'=>['src'=>'//content.jwplatform.com/players/{@id}.html'],'name'=>'JW Platform','scrape'=>[],'tags'=>['videos']],
		'khl'=>['attributes'=>[],'example'=>['https://video.khl.ru/events/233677','https://video.khl.ru/quotes/251237'],'extract'=>[],'homepage'=>'https://www.khl.ru/','host'=>['video.khl.ru'],'iframe'=>['src'=>'//video.khl.ru/iframe/feed/start/{@id}?type_id=18&width=560&height=315'],'name'=>'Kontinental Hockey League (КХЛ)','scrape'=>[['extract'=>['!/feed/start/(?\'id\'[/\\w]+)!'],'match'=>['!video\\.khl\\.ru/(?:event|quote)s/\\d!']]],'tags'=>['.ru','sports']],
		'kickstarter'=>['attributes'=>[],'choose'=>['otherwise'=>['iframe'=>['height'=>420,'src'=>'//www.kickstarter.com/projects/{@id}/widget/card.html','width'=>220]],'when'=>['iframe'=>['height'=>360,'src'=>'//www.kickstarter.com/projects/{@id}/widget/video.html','width'=>640],'test'=>'@video']],'example'=>['https://www.kickstarter.com/projects/1869987317/wish-i-was-here-1','https://www.kickstarter.com/projects/1869987317/wish-i-was-here-1/widget/card.html','https://www.kickstarter.com/projects/1869987317/wish-i-was-here-1/widget/video.html'],'extract'=>['!kickstarter\\.com/projects/(?\'id\'[^/]+/[^/?]+)(?:/widget/(?:(?\'card\'card)|(?\'video\'video)))?!'],'homepage'=>'https://www.kickstarter.com/','host'=>['kickstarter.com'],'name'=>'Kickstarter','scrape'=>[],'tags'=>['fundraising']],
		'libsyn'=>['attributes'=>[],'example'=>'http://pexlives.libsyn.com/the-ark-in-space-ep4-commentary-with-el-and-jack','extract'=>[],'homepage'=>'https://www.libsyn.com/','host'=>['libsyn.com'],'iframe'=>['height'=>45,'max-width'=>900,'src'=>'//html5-player.libsyn.com/embed/episode/id/{@id}/height/45/width/900/theme/standard/direction/no/autoplay/no/autonext/no/thumbnail/no/preload/no/no_addthis/no/','width'=>'100%'],'name'=>'Libsyn','scrape'=>[['extract'=>['!embed/episode/id/(?\'id\'\\d+)!'],'match'=>['@(?!\\.mp3)....$@']]],'tags'=>['podcasts']],
		'liveleak'=>['attributes'=>[],'example'=>'https://www.liveleak.com/view?t=yIcw_1520190567','extract'=>['!liveleak\\.com/(?:e/|view\\?i=)(?\'id\'\\w+)!'],'homepage'=>'https://www.liveleak.com/','host'=>['liveleak.com'],'iframe'=>['src'=>'//www.liveleak.com/e/{@id}'],'name'=>'Liveleak','scrape'=>[['extract'=>['!liveleak\\.com/e/(?\'id\'\\w+)!'],'match'=>['!liveleak\\.com/view\\?t=!']]],'tags'=>['videos']],
		'livestream'=>['attributes'=>[],'example'=>['https://new.livestream.com/jbtvlive/musicmarathon','https://livestream.com/ccscsl/USChessChampionships/videos/83267610','https://livestre.am/58XNV'],'extract'=>['!livestream\\.com/accounts/(?\'account_id\'\\d+)/events/(?\'event_id\'\\d+)!','!/videos/(?\'video_id\'\\d+)!','!original\\.livestream\\.com/(?\'channel\'\\w+)/video\\?clipId=(?\'clip_id\'[-\\w]+)!'],'homepage'=>'https://new.livestream.com/','host'=>['livestre.am','livestream.com'],'iframe'=>['src'=>'//<xsl:choose><xsl:when test="@clip_id">cdn.livestream.com/embed/<xsl:value-of select="@channel"/>?layout=4&amp;autoplay=false&amp;clip=<xsl:value-of select="@clip_id"/></xsl:when><xsl:otherwise>livestream.com/accounts/<xsl:value-of select="@account_id"/>/events/<xsl:value-of select="@event_id"/><xsl:if test="@video_id">/videos/<xsl:value-of select="@video_id"/></xsl:if>/player?autoPlay=false</xsl:otherwise></xsl:choose>'],'name'=>'Livestream','scrape'=>[['extract'=>['!accounts/(?\'account_id\'\\d+)/events/(?\'event_id\'\\d+)!'],'match'=>['@livestream\\.com/(?!accounts/\\d+/events/\\d)@']],['extract'=>['!//original\\.livestream\\.com/(?\'channel\'\\w+)/video/(?\'clip_id\'[-\\w]+)!'],'match'=>['!livestre.am!']]],'tags'=>['livestreaming','videos']],
		'mailru'=>['attributes'=>[],'example'=>['https://my.mail.ru/corp/auto/video/testdrive/34.html','https://my.mail.ru/mail/alenka1957/video/1/7.html'],'extract'=>[],'homepage'=>'https://my.mail.ru/','host'=>['my.mail.ru'],'iframe'=>['src'=>'https://my.mail.ru/video/embed/{@id}'],'name'=>'Mail.Ru','scrape'=>[['extract'=>['!"itemId": ?"?(?\'id\'\\d+)!'],'match'=>['!my\\.mail\\.ru/\\w+/\\w+/video/\\w+/\\d!']]],'tags'=>['.ru']],
		'medium'=>['attributes'=>[],'example'=>'https://medium.com/@donnydonny/team-internet-is-about-to-win-net-neutrality-and-they-didnt-need-googles-help-e7e2cf9b8a95','extract'=>['!medium\\.com/(?:s/)?[^/]*/(?:[-\\w]+-)?(?\'id\'[\\da-f]+)!'],'homepage'=>'https://medium.com/','host'=>['medium.com'],'iframe'=>['height'=>313,'max-width'=>900,'src'=>'https://api.medium.com/embed?type=story&path=%2F%2F{@id}','style'=>['border'=>'1px solid','border-color'=>'#eee #ddd #bbb','border-radius'=>'5px','box-shadow'=>'rgba(0,0,0,.15) 0 1px 3px'],'width'=>'100%'],'name'=>'Medium','scrape'=>[],'tags'=>['blogging']],
		'megaphone'=>['attributes'=>[],'example'=>['https://cms.megaphone.fm/channel/lockedonheat?selected=LKN8165322853','https://player.megaphone.fm/LKN8165322853'],'extract'=>['@megaphone\\.fm/.*?\\?(?:e|selected)=(?\'id\'\\w+)@','@(?:dcs|player|traffic)\\.megaphone\\.fm/(?\'id\'\\w+)@','@megaphone\\.link/(?\'id\'\\w+)@'],'homepage'=>'https://megaphone.fm/','host'=>['megaphone.fm','megaphone.link'],'iframe'=>['height'=>200,'max-width'=>900,'src'=>'https://player.megaphone.fm/{@id}','width'=>'100%'],'name'=>'Megaphone','scrape'=>[],'tags'=>['podcasts']],
		'metacafe'=>['attributes'=>[],'example'=>'https://www.metacafe.com/watch/10785282/chocolate_treasure_chest_epic_meal_time/','extract'=>['!metacafe\\.com/watch/(?\'id\'\\d+)!'],'homepage'=>'https://www.metacafe.com/','host'=>['metacafe.com'],'iframe'=>['src'=>'//www.metacafe.com/embed/{@id}/'],'name'=>'Metacafe','scrape'=>[],'tags'=>['videos']],
		'mixcloud'=>['attributes'=>[],'example'=>'https://www.mixcloud.com/OneTakeTapes/timsch-one-take-tapes-2/','extract'=>['@mixcloud\\.com/(?!categories|tag)(?\'id\'[-\\w]+/[^/&]+)/@'],'homepage'=>'https://www.mixcloud.com/','host'=>['mixcloud.com'],'iframe'=>['height'=>400,'src'=>'//www.mixcloud.com/widget/iframe/?feed=http%3A%2F%2Fwww.mixcloud.com%2F{@id}%2F&embed_type=widget_standard','width'=>400],'name'=>'Mixcloud','scrape'=>[],'tags'=>['music']],
		'mixer'=>['attributes'=>[],'example'=>['https://mixer.com/Ninja','https://mixer.com/Ninja?vod=p4sVuGe4Fk-4EPoN5enIzw'],'extract'=>['#mixer.com/(?!browse/)(?\'channel\'\\w+)(?!\\?clip|\\w)(?:\\?vod=(?\'vod\'[-\\w]+))?#'],'homepage'=>'https://mixer.com/','host'=>['mixer.com'],'iframe'=>['src'=>'https://mixer.com/embed/player/<xsl:value-of select="@channel"/><xsl:if test="@vod">?vod=<xsl:value-of select="@vod"/></xsl:if>'],'name'=>'Mixer','scrape'=>[],'tags'=>['gaming','livestreaming']],
		'mlb'=>['attributes'=>[],'example'=>['https://www.mlb.com/video/statcast-stantons-two-homers/c-1898424783','https://m.mlb.com/video/v1205791883'],'extract'=>['#mlb\\.com/video/(?:[-\\w/]+/)?(?:c-|v|[-\\w]+-c)(?\'id\'\\d+)#'],'homepage'=>'https://mlb.com/video/','host'=>['mlb.com'],'iframe'=>['src'=>'https://www.mlb.com/video/share/c-{@id}'],'name'=>'MLB','scrape'=>[],'tags'=>['sports']],
		'mrctv'=>['attributes'=>[],'example'=>'https://mrctv.org/videos/cnn-frets-about-tobacco-companies-color-coding-tricks','extract'=>[],'homepage'=>'https://www.mrctv.org/','host'=>['mrctv.org'],'iframe'=>['src'=>'https://www.mrctv.org/embed/{@id}'],'name'=>'MRCTV','scrape'=>[['extract'=>['!mrctv\\.org/embed/(?\'id\'\\d+)!'],'match'=>['!mrctv\\.org/videos/.!']]]],
		'msnbc'=>['attributes'=>[],'choose'=>['otherwise'=>['iframe'=>['src'=>'https://www.msnbc.com/msnbc/embedded-video/{@id}']],'when'=>['iframe'=>['padding-height'=>80,'src'=>'//player.theplatform.com/p/7wvmTC/MSNBCEmbeddedOffSite?guid={@id}'],'test'=>'contains(@id, \'_\')']],'example'=>['https://www.msnbc.com/ronan-farrow-daily/watch/thats-no-moon--300512323725','https://on.msnbc.com/1qkH62o'],'extract'=>[],'homepage'=>'https://www.msnbc.com/watch','host'=>['msnbc.com'],'name'=>'MSNBC','scrape'=>[['extract'=>['@"video":\\{"current".*?id":"(?\'id\'\\w+)@'],'match'=>['@msnbc\\.com/[-\\w]+/watch/@','@on\\.msnbc\\.com/.@']]],'tags'=>['news']],
		'natgeochannel'=>['attributes'=>[],'example'=>['https://channel.nationalgeographic.com/channel/brain-games/videos/jason-silva-on-intuition/','https://channel.nationalgeographic.com/wild/urban-jungle/videos/leopard-in-the-city/'],'extract'=>['@channel\\.nationalgeographic\\.com/(?\'id\'[-/\\w]+/videos/[-\\w]+)@'],'homepage'=>'https://channel.nationalgeographic.com/','host'=>['channel.nationalgeographic.com'],'iframe'=>['src'=>'//channel.nationalgeographic.com/{@id}/embed/'],'name'=>'National Geographic Channel','scrape'=>[]],
		'natgeovideo'=>['attributes'=>[],'example'=>['https://video.nationalgeographic.com/tv/changing-earth','https://video.nationalgeographic.com/video/weirdest-superb-lyrebird'],'extract'=>[],'homepage'=>'https://video.nationalgeographic.com/','host'=>['video.nationalgeographic.com'],'iframe'=>['src'=>'//player.d.nationalgeographic.com/players/ngsvideo/share/?guid={@id}'],'name'=>'National Geographic Video','scrape'=>[['extract'=>['@guid="(?\'id\'[-\\w]+)"@'],'match'=>['@video\\.nationalgeographic\\.com/(?:tv|video)/\\w@']]],'tags'=>['documentaries']],
		'nbcnews'=>['attributes'=>[],'example'=>'https://www.nbcnews.com/video/bob-dylan-awarded-nobel-prize-for-literature-785193027834','extract'=>['!nbcnews\\.com/(?:widget/video-embed/|video/[-\\w]+?-)(?\'id\'\\d+)!'],'homepage'=>'https://www.nbcnews.com/video/','host'=>['nbcnews.com'],'iframe'=>['src'=>'//www.nbcnews.com/widget/video-embed/{@id}'],'name'=>'NBC News','scrape'=>[],'tags'=>['news']],
		'nbcsports'=>['attributes'=>[],'example'=>'https://www.nbcsports.com/video/countdown-rio-olympics-what-makes-perfect-performance','extract'=>[],'homepage'=>'https://www.nbcsports.com/video','host'=>['nbcsports.com'],'iframe'=>['src'=>'//vplayer.nbcsports.com/p/BxmELC/nbcsports_embed/select/media/{@id}?parentUrl='],'name'=>'NBC Sports','scrape'=>[['extract'=>['!select/media/(?\'id\'\\w+)!'],'match'=>['!nbcsports\\.com/video/.!']]],'tags'=>['sports']],
		'nhl'=>['attributes'=>[],'example'=>'https://www.nhl.com/video/recap-min-2-ott-1-fot/t-277753022/c-46330703','extract'=>['#nhl\\.com/(?:\\w+/)?video(?:/(?![ct]-)[-\\w]+)?(?:/t-(?\'t\'\\d+))?(?:/c-(?\'c\'\\d+))?#'],'homepage'=>'https://www.nhl.com/video','host'=>['nhl.com'],'iframe'=>['src'=>'https://www.nhl.com/video/embed<xsl:if test="@t">/t-<xsl:value-of select="@t"/></xsl:if><xsl:if test="@c">/c-<xsl:value-of select="@c"/></xsl:if>?autostart=false'],'name'=>'NHL Videos and Highlights','scrape'=>[],'tags'=>['sports']],
		'npr'=>['attributes'=>[],'example'=>['https://www.npr.org/blogs/goatsandsoda/2015/02/11/385396431/the-50-most-effective-ways-to-transform-the-developing-world','https://n.pr/1Qky1m5'],'extract'=>[],'homepage'=>'https://www.npr.org/','host'=>['npr.org','n.pr'],'iframe'=>['height'=>228,'max-width'=>800,'src'=>'//www.npr.org/player/embed/{@i}/{@m}','width'=>'100%'],'name'=>'NPR','scrape'=>[['extract'=>['!player/embed/(?\'i\'\\d+)/(?\'m\'\\d+)!'],'header'=>'Cookie: trackingChoice=false; choiceVersion=1','match'=>['!npr\\.org/[/\\w]+/\\d+!','!n\\.pr/\\w!']]],'tags'=>['podcasts']],
		'nytimes'=>['attributes'=>[],'example'=>['https://www.nytimes.com/video/magazine/100000003166834/small-plates.html','https://www.nytimes.com/video/technology/personaltech/100000002907606/soylent-taste-test.html','https://www.nytimes.com/video/2012/12/17/business/100000001950744/how-wal-mart-conquered-teotihuacan.html','https://movies.nytimes.com/movie/131154/Crooklyn/trailers'],'extract'=>['!nytimes\\.com/video/[a-z]+/(?:[a-z]+/)?(?\'id\'\\d+)!','!nytimes\\.com/video/\\d+/\\d+/\\d+/[a-z]+/(?\'id\'\\d+)!'],'homepage'=>'https://www.nytimes.com/video/','host'=>['nytimes.com'],'iframe'=>['height'=>400,'src'=>'//graphics8.nytimes.com/video/players/offsite/index.html?videoId={@id}','width'=>585],'name'=>'The New York Times Video','scrape'=>[['extract'=>['!/video/movies/(?\'id\'\\d+)!'],'match'=>['!nytimes\\.com/movie(?:s/movie)?/(?\'playlist\'\\d+)/[-\\w]+/trailers!'],'url'=>'http://www.nytimes.com/svc/video/api/playlist/{@playlist}?externalId=true']],'tags'=>['movies','news']],
		'orfium'=>['attributes'=>[],'example'=>['https://www.orfium.com/album/24371/everybody-loves-kanye-totom/','https://www.orfium.com/live-set/614763/foof-no-lights-5-foof/','https://www.orfium.com/playlist/511651/electronic-live-sessions-creamtronic/','https://www.orfium.com/track/625367/the-ambience-of-the-goss-vistas/'],'extract'=>['@album/(?\'album_id\'\\d+)@','@playlist/(?\'playlist_id\'\\d+)@','@live-set/(?\'set_id\'\\d+)@','@track/(?\'track_id\'\\d+)@'],'homepage'=>'https://www.orfium.com/','host'=>['orfium.com'],'iframe'=>['height'=>'<xsl:choose><xsl:when test="@album_id">550</xsl:when><xsl:otherwise>275</xsl:otherwise></xsl:choose>','max-width'=>900,'src'=>'https://www.orfium.com/embedded/<xsl:choose><xsl:when test="@album_id">album/<xsl:value-of select="@album_id"/></xsl:when><xsl:when test="@playlist_id">playlist/<xsl:value-of select="@playlist_id"/></xsl:when><xsl:when test="@set_id">live-set/<xsl:value-of select="@set_id"/></xsl:when><xsl:otherwise>track/<xsl:value-of select="@track_id"/></xsl:otherwise></xsl:choose>','width'=>'100%'],'name'=>'Orfium','scrape'=>[],'tags'=>['music']],
		'pastebin'=>['attributes'=>[],'example'=>'https://pastebin.com/9jEf44nc','extract'=>['@pastebin\\.com/(?!u/)(?:\\w+(?:\\.php\\?i=|/))?(?\'id\'\\w+)@'],'homepage'=>'https://pastebin.com/','host'=>['pastebin.com'],'iframe'=>['height'=>300,'scrolling'=>'','src'=>'//pastebin.com/embed_iframe.php?i={@id}','style'=>['resize'=>'vertical'],'width'=>'100%'],'name'=>'Pastebin','scrape'=>[]],
		'pinterest'=>['attributes'=>[],'example'=>['https://www.pinterest.com/pin/99360735500167749/','https://www.pinterest.com/pinterest/official-news/'],'extract'=>['@pinterest.com/pin/(?\'id\'\\d+)@','@pinterest.com/(?!_/|discover/|explore/|news_hub/|pin/|search/)(?\'id\'[-\\w]+/[-\\w]+)@'],'homepage'=>'https://www.pinterest.com/','host'=>['pinterest.com'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>360,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/pinterest.min.html#{@id}','width'=>'<xsl:choose><xsl:when test="contains(@id,\'/\')">730</xsl:when><xsl:otherwise>345</xsl:otherwise></xsl:choose>'],'name'=>'Pinterest','scrape'=>[],'source'=>'https://developers.pinterest.com/tools/widget-builder/','tags'=>['social']],
		'podbean'=>['attributes'=>[],'example'=>['https://dialhforheroclix.podbean.com/e/dial-h-for-heroclix-episode-46-all-ya-need-is-love/','https://www.podbean.com/media/share/pb-qtwub-4ee10c'],'extract'=>['!podbean\\.com/media/(?:player/|share/pb-)(?\'id\'[-\\w]+)!'],'homepage'=>'https://www.podbean.com/','host'=>['podbean.com'],'iframe'=>['height'=>100,'max-width'=>900,'src'=>'//www.podbean.com/media/player/{@id}','width'=>'100%'],'name'=>'Podbean','scrape'=>[['extract'=>['!podbean\\.com/media/player/(?\'id\'[-\\w]+)!'],'match'=>['@podbean\\.com/(?:media/shar)?e/(?!pb-)@']]],'tags'=>['podcasts']],
		'prezi'=>['attributes'=>[],'example'=>'https://prezi.com/5ye8po_hmikp/10-most-common-rookie-presentation-mistakes/','extract'=>['#//prezi\\.com/(?!(?:a(?:bout|mbassadors)|c(?:o(?:llaborate|mmunity|ntact)|reate)|exp(?:erts|lore)|ip(?:ad|hone)|jobs|l(?:ear|ogi)n|m(?:ac|obility)|pr(?:es(?:s|ent)|icing)|recommend|support|user|windows|your)/)(?\'id\'\\w+)/#'],'homepage'=>'https://prezi.com/','host'=>['prezi.com'],'iframe'=>['height'=>400,'src'=>'//prezi.com/embed/{@id}/','width'=>550],'name'=>'Prezi','scrape'=>[],'tags'=>['presentations']],
		'reddit'=>['attributes'=>[],'example'=>['https://www.reddit.com/r/pics/comments/9wp1qg/my_mom_recently_started_watercolor_painting_and/','https://www.reddit.com/r/pics/comments/9wp1qg/my_mom_recently_started_watercolor_painting_and/e9m6pn9/'],'extract'=>['!(?\'id\'\\w+/comments/\\w+(?:/\\w+/\\w+)?)!'],'homepage'=>'https://www.reddit.com/','host'=>['reddit.com'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>165,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/reddit.min.html#{@id}{@path}','width'=>800],'name'=>'Reddit threads and comments','scrape'=>[],'source'=>'https://www.reddit.com/wiki/embeds','tags'=>['social']],
		'rutube'=>['attributes'=>[],'example'=>['https://rutube.ru/video/0771e73d1d10eee9c43ce31decc1dfce/','https://rutube.ru/tracks/4118278.html?v=8b490a46447720d4ad74616f5de2affd'],'extract'=>['!rutube\\.ru/tracks/(?\'id\'\\d+)!'],'homepage'=>'https://rutube.ru/','host'=>['rutube.ru'],'iframe'=>['height'=>405,'src'=>'//rutube.ru/play/embed/{@id}','width'=>720],'name'=>'Rutube','scrape'=>[['extract'=>['!rutube\\.ru/play/embed/(?\'id\'\\d+)!'],'header'=>'User-agent: Mozilla/5.0 (X11; Linux x86_64; rv:62.0) Gecko/20100101 Firefox/62.0','match'=>['!rutube\\.ru/video/(?\'vid\'[0-9a-f]{32})!'],'url'=>'http://rutube.ru/api/oembed/?url=https://rutube.ru/video/{@vid}/']],'tags'=>['.ru']],
		'scribd'=>['attributes'=>[],'example'=>['https://www.scribd.com/doc/237147661/Calculus-2-Test-1-Review','https://www.scribd.com/presentation/324333881/600-1450-World-History-Jeopardy'],'extract'=>['!scribd\\.com/(?:mobile/)?(?:doc(?:ument)?|presentation)/(?\'id\'\\d+)!'],'homepage'=>'https://www.scribd.com/','host'=>['scribd.com'],'iframe'=>['height'=>500,'src'=>'https://www.scribd.com/embeds/{@id}/content?view_mode=scroll&show_recommendations=false','style'=>['resize'=>'vertical'],'width'=>'100%'],'name'=>'Scribd','scrape'=>[],'tags'=>['documents','presentations']],
		'sendvid'=>['attributes'=>[],'example'=>'https://sendvid.com/w99u3jjp','extract'=>['!sendvid\\.com/(?\'id\'\\w+)!'],'homepage'=>'https://www.sendvid.com/','host'=>['sendvid.com'],'iframe'=>['src'=>'//sendvid.com/embed/{@id}'],'name'=>'Sendvid','scrape'=>[],'tags'=>['videos']],
		'slideshare'=>['attributes'=>[],'example'=>'https://www.slideshare.net/Slideshare/how-23431564','extract'=>['!slideshare\\.net/[^/]+/[-\\w]+-(?\'id\'\\d{6,})$!'],'homepage'=>'https://www.slideshare.net/','host'=>['slideshare.net'],'iframe'=>['height'=>356,'src'=>'//www.slideshare.net/slideshow/embed_code/{@id}','width'=>427],'name'=>'SlideShare','scrape'=>[['extract'=>['!"presentationId":(?\'id\'\\d+)!'],'match'=>['@slideshare\\.net/[^/]+/\\w(?![-\\w]+-\\d{6,}$)@']]],'source'=>'http://help.slideshare.com/forums/67665-Embedding-Sharing','tags'=>['presentations']],
		'soundcloud'=>['attributes'=>[],'example'=>['https://api.soundcloud.com/tracks/98282116','https://soundcloud.com/andrewbird/three-white-horses','https://soundcloud.com/tenaciousd/sets/rize-of-the-fenix/'],'extract'=>['@https?://(?:api\\.)?soundcloud\\.com/(?!pages/)(?\'id\'[-/\\w]+/[-/\\w]+|^[^/]+/[^/]+$)@i','@api\\.soundcloud\\.com/playlists/(?\'playlist_id\'\\d+)@','@api\\.soundcloud\\.com/tracks/(?\'track_id\'\\d+)(?:\\?secret_token=(?\'secret_token\'[-\\w]+))?@','@soundcloud\\.com/(?!playlists|tracks)[-\\w]+/[-\\w]+/(?=s-)(?\'secret_token\'[-\\w]+)@'],'homepage'=>'https://soundcloud.com/','host'=>['soundcloud.com'],'iframe'=>['height'=>'<xsl:choose><xsl:when test="@playlist_id or contains(@id,\'/sets/\')">450</xsl:when><xsl:otherwise>166</xsl:otherwise></xsl:choose>','max-width'=>900,'src'=>'https://w.soundcloud.com/player/?url=<xsl:choose><xsl:when test="@playlist_id">https%3A//api.soundcloud.com/playlists/<xsl:value-of select="@playlist_id"/></xsl:when><xsl:when test="@track_id">https%3A//api.soundcloud.com/tracks/<xsl:value-of select="@track_id"/>&amp;secret_token=<xsl:value-of select="@secret_token"/></xsl:when><xsl:otherwise><xsl:if test="not(contains(@id,\'://\'))">https%3A//soundcloud.com/</xsl:if><xsl:value-of select="@id"/></xsl:otherwise></xsl:choose>','width'=>'100%'],'name'=>'SoundCloud','scrape'=>[['extract'=>['@soundcloud:tracks:(?\'track_id\'\\d+)@'],'header'=>'User-agent: PHP (not Mozilla)','match'=>['@soundcloud\\.com/(?!playlists/\\d|tracks/\\d)[-\\w]+/[-\\w]@']],['extract'=>['@soundcloud://playlists:(?\'playlist_id\'\\d+)@'],'header'=>'User-agent: PHP (not Mozilla)','match'=>['@soundcloud\\.com/\\w+/sets/@']]],'source'=>'https://soundcloud.com/pages/widgets','tags'=>['music']],
		'sporcle'=>['attributes'=>[],'example'=>'https://www.sporcle.com/games/bowsntoys/mothers-day-jokes','extract'=>['#sporcle.com/framed/.*?gid=(?\'id\'\\w+)#'],'homepage'=>'https://www.sporcle.com/','host'=>['sporcle.com'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>535,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'*\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/sporcle.min.html#{@id}','width'=>820],'name'=>'Sporcle','scrape'=>[['extract'=>['#encodedGameID\\W+(?\'id\'\\w+)#'],'match'=>['#sporcle\\.com/games/(?!\\w*category/)[-\\w]+/[-\\w]#']]],'tags'=>['entertainment']],
		'sportsnet'=>['attributes'=>[],'example'=>'https://www.sportsnet.ca/soccer/west-ham-2-hull-2/','extract'=>[],'homepage'=>'https://www.sportsnet.ca/','host'=>['sportsnet.ca'],'iframe'=>['src'=>'https://images.rogersdigitalmedia.com/video_service.php?videoId={@id}&playerKey=AQ~~,AAAAAGWRwLc~,cRCmKE8Utf7OFWP38XQcokFZ80fR-u_y&autoStart=false&width=100%25&height=100%25'],'name'=>'Sportsnet','scrape'=>[['extract'=>['/vid(?:eoId)?=(?\'id\'\\d+)/','/param name="@videoPlayer" value="(?\'id\'\\d+)"/'],'match'=>['//']]],'tags'=>['.ca','sports']],
		'spotify'=>['attributes'=>[],'example'=>['https://open.spotify.com/user/ryan.lloyd.french/playlist/4jHNeLTj7CnL5JwZzmnqOL','https://open.spotify.com/track/3CPjqkaMwrbht1lJpELoY0?si=K7cm7-dWTR6U7BjDn1oP7g','https://play.spotify.com/album/5OSzFvFAYuRh93WDNCTLEz'],'extract'=>['!(?:open|play)\\.spotify\\.com/(?\'id\'(?:user/[-.\\w]+/)?(?:album|artist|playlist|track)(?:[:/][-.\\w]+)+)!'],'homepage'=>'https://www.spotify.com/','host'=>['open.spotify.com','play.spotify.com'],'iframe'=>['allow'=>'encrypted-media','height'=>400,'src'=>'https://open.spotify.com/embed/{translate(@id,\':\',\'/\')}{@path}','width'=>400],'name'=>'Spotify','scrape'=>[],'source'=>['https://developer.spotify.com/technologies/widgets/spotify-play-button/','http://news.spotify.com/2008/01/14/linking-to-spotify/'],'tags'=>['music']],
		'spreaker'=>['attributes'=>[],'example'=>['https://www.spreaker.com/show/2-up-front_1','https://www.spreaker.com/user/publichousemedia/2-up-front-119-final-show-peter-wilt-rac'],'extract'=>['!spreaker\\.com/episode/(?\'episode_id\'\\d+)!'],'homepage'=>'https://www.spreaker.com','host'=>['spreaker.com'],'iframe'=>['height'=>'{400 - 200 * boolean(@episode_id)}','max-width'=>900,'src'=>'https://widget.spreaker.com/player?episode_id={@episode_id}&show_id={@show_id}','width'=>'100%'],'name'=>'Spreaker','scrape'=>[['extract'=>['!episode_id=(?\'episode_id\'\\d+)!','!show_id=(?\'show_id\'\\d+)!'],'match'=>['!(?\'url\'.+/(?:show/|user/.+/).+)!'],'url'=>'https://api.spreaker.com/oembed?format=json&url={@url}']],'tags'=>['podcasts']],
		'steamstore'=>['attributes'=>[],'example'=>'https://store.steampowered.com/app/250520/','extract'=>['!store.steampowered.com/app/(?\'id\'\\d+)!'],'homepage'=>'https://store.steampowered.com/','host'=>['store.steampowered.com'],'iframe'=>['height'=>190,'max-width'=>900,'src'=>'//store.steampowered.com/widget/{@id}','width'=>'100%'],'name'=>'Steam store','scrape'=>[],'tags'=>['gaming']],
		'stitcher'=>['attributes'=>[],'example'=>'https://www.stitcher.com/podcast/twit/tech-news-today/e/twitter-shares-fall-18-percent-after-earnings-leak-on-twitter-37808629','extract'=>[],'homepage'=>'https://www.stitcher.com/','host'=>['stitcher.com'],'iframe'=>['height'=>150,'max-width'=>900,'src'=>'//app.stitcher.com/splayer/f/{@fid}/{@eid}','width'=>'100%'],'name'=>'Stitcher','scrape'=>[['extract'=>['!data-eid="(?\'eid\'\\d+)!','!data-fid="(?\'fid\'\\d+)!'],'match'=>['!/podcast/!']]],'tags'=>['podcasts']],
		'strawpoll'=>['attributes'=>[],'example'=>'https://strawpoll.me/738091','extract'=>['!strawpoll\\.me/(?\'id\'\\d+)!'],'homepage'=>'https://strawpoll.me/','host'=>['strawpoll.me'],'iframe'=>['scrolling'=>'','src'=>'//www.strawpoll.me/embed_1/{@id}'],'name'=>'Straw Poll','scrape'=>[]],
		'streamable'=>['attributes'=>[],'example'=>'https://streamable.com/e4d','extract'=>['!streamable\\.com/(?\'id\'\\w+)!'],'homepage'=>'https://streamable.com/','host'=>['streamable.com'],'iframe'=>['src'=>'//streamable.com/e/{@id}'],'name'=>'Streamable','scrape'=>[],'tags'=>['videos']],
		'streamja'=>['attributes'=>[],'example'=>'https://streamja.com/2nr','extract'=>['@streamja\\.com/(?!login|signup|terms|videos)(?\'id\'\\w+)@'],'homepage'=>'https://streamja.com','host'=>['streamja.com'],'iframe'=>['src'=>'https://streamja.com/embed/{@id}'],'name'=>'Streamja','scrape'=>[],'tags'=>['sports']],
		'teamcoco'=>['attributes'=>[],'example'=>['https://teamcoco.com/video/serious-jibber-jabber-a-scott-berg-full-episode','https://teamcoco.com/video/73784/historian-a-scott-berg-serious-jibber-jabber-with-conan-obrien'],'extract'=>['!teamcoco\\.com/video/(?\'id\'\\d+)!'],'homepage'=>'https://teamcoco.com/','host'=>['teamcoco.com'],'iframe'=>['src'=>'//teamcoco.com/embed/v/{@id}'],'name'=>'Team Coco','scrape'=>[['extract'=>['!embed/v/(?\'id\'\\d+)!'],'match'=>['!teamcoco\\.com/video/\\D!']]],'tags'=>['entertainment']],
		'ted'=>['attributes'=>[],'example'=>['https://www.ted.com/talks/eli_pariser_beware_online_filter_bubbles.html','https://embed.ted.com/playlists/26/our_digital_lives.html'],'extract'=>['#ted\\.com/(?\'id\'(?:talk|playlist)s/[-\\w]+(?:\\.html)?)(?![-\\w]|/transcript)#i'],'homepage'=>'https://www.ted.com/','host'=>['ted.com'],'iframe'=>['src'=>'//embed.ted.com/<xsl:value-of select="@id"/><xsl:if test="not(contains(@id,\'.html\'))">.html</xsl:if>'],'name'=>'TED Talks','scrape'=>[],'source'=>'http://blog.ted.com/2011/04/01/now-you-can-embed-tedtalks-with-subtitles-enabled/','tags'=>['presentations']],
		'telegram'=>['attributes'=>[],'example'=>'https://t.me/telegram/83','extract'=>['@//t.me/(?!addstickers/|joinchat/)(?\'id\'\\w+/\\d+)@'],'homepage'=>'https://telegram.org/','host'=>['t.me'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>96,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/telegram.min.html#{@id}','width'=>500],'name'=>'Telegram','scrape'=>[],'tags'=>['social']],
		'theatlantic'=>['attributes'=>[],'example'=>'https://www.theatlantic.com/video/index/358928/computer-vision-syndrome-and-you/','extract'=>['!theatlantic\\.com/video/index/(?\'id\'\\d+)!'],'homepage'=>'https://www.theatlantic.com/video/','host'=>['theatlantic.com'],'iframe'=>['src'=>'//www.theatlantic.com/video/iframe/{@id}/'],'name'=>'The Atlantic Video','scrape'=>[],'tags'=>['news']],
		'theguardian'=>['attributes'=>[],'example'=>'https://www.theguardian.com/world/video/2016/apr/07/tokyos-hedgehog-cafe-encourages-you-to-embrace-prickly-pets-video','extract'=>['!theguardian\\.com/(?\'id\'\\w+/video/20(?:0[0-9]|1[0-7])[-/\\w]+)!'],'homepage'=>'https://www.theguardian.com/video','host'=>['theguardian.com'],'iframe'=>['src'=>'//embed.theguardian.com/embed/video/{@id}'],'name'=>'The Guardian (obsolete)','scrape'=>[],'tags'=>['news']],
		'theonion'=>['attributes'=>[],'example'=>['https://www.theonion.com/video/nation-successfully-completes-mothers-day-by-918-a,35998/','https://www.theonion.com/video/the-onion-reviews-avengers-age-of-ultron-38524'],'extract'=>['!theonion\\.com/video/[-\\w]+[-,](?\'id\'\\d+)!'],'homepage'=>'https://www.theonion.com/video/','host'=>['theonion.com'],'iframe'=>['src'=>'//www.theonion.com/video_embed/?id={@id}'],'name'=>'The Onion','scrape'=>[],'tags'=>['entertainment']],
		'tiktok'=>['attributes'=>[],'example'=>'https://m.tiktok.com/v/6583727202272021766.html','extract'=>['@tiktok\\.com/(?:v|i18n/share/video)/(?\'id\'\\d+)@'],'homepage'=>'https://www.tiktok.com','host'=>['tiktok.com'],'iframe'=>['height'=>700,'src'=>'//www.tiktok.com/embed/{@id}','width'=>340],'name'=>'TikTok','scrape'=>[['extract'=>['@tiktok\\.com/v/(?\'id\'\\d+)@','@musically://musical\\?id=(?\'id\'\\d+)@'],'header'=>'User-agent: PHP','match'=>['@//vm\\.tiktok\\.com/.@']]],'tags'=>['videos']],
		'tmz'=>['attributes'=>[],'example'=>'https://www.tmz.com/videos/0_2pr9x3rb/','extract'=>['@tmz\\.com/videos/(?\'id\'\\w+)@'],'homepage'=>'https://www.tmz.com/videos','host'=>['tmz.com'],'iframe'=>['src'=>'//www.kaltura.com/index.php/kwidget/cache_st/133592691/wid/_591531/partner_id/591531/uiconf_id/9071262/entry_id/{@id}'],'name'=>'TMZ','scrape'=>[],'tags'=>['gossip']],
		'traileraddict'=>['attributes'=>[],'example'=>'https://www.traileraddict.com/the-amazing-spider-man-2/super-bowl-tv-spot','extract'=>[],'homepage'=>'https://www.traileraddict.com/','host'=>['traileraddict.com'],'iframe'=>['src'=>'//v.traileraddict.com/{@id}'],'name'=>'Trailer Addict','scrape'=>[['extract'=>['@v\\.traileraddict\\.com/(?\'id\'\\d+)@'],'match'=>['@traileraddict\\.com/(?!tags/)[^/]+/.@']]],'tags'=>['movies']],
		'trendingviews'=>['attributes'=>[],'example'=>'https://trendingviews.co/video/chick-fil-a-customers-sing-impromtu-lean-on-me-8213.html','extract'=>['!trendingviews.co/(?:video|embed)/(?:[^/]+-)?(?\'id\'\\d+)!'],'homepage'=>'https://trendingviews.co/','host'=>['trendingviews.co'],'iframe'=>['src'=>'https://trendingviews.co/embed/{@id}'],'name'=>'Trending Views','scrape'=>[],'tags'=>['videos']],
		'tumblr'=>['attributes'=>[],'example'=>'https://staff.tumblr.com/post/180624227620/','extract'=>['!(?\'name\'[-\\w]+)\\.tumblr\\.com/post/(?\'id\'\\d+)!'],'homepage'=>'https://www.tumblr.com/','host'=>['tumblr.com'],'iframe'=>['data-s9e-livepreview-ignore-attrs'=>'style','height'=>300,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/tumblr.min.html#{@key}/{@id}','width'=>520],'name'=>'Tumblr','scrape'=>[['extract'=>['!did=\\\\u0022(?\'did\'[-\\w]+)!','!embed\\\\/post\\\\/(?\'key\'[-\\w]+)!'],'match'=>['!\\w\\.tumblr\\.com/post/\\d!'],'url'=>'http://www.tumblr.com/oembed/1.0?url=http://{@name}.tumblr.com/post/{@id}']],'tags'=>['social']],
		'twentyfoursevensports'=>['attributes'=>[],'choose'=>['otherwise'=>['iframe'=>['height'=>200,'onload'=>'window.addEventListener(\'message\',function(e){{if(e.source===this.contentWindow&&e.data.height)this.style.height=e.data.height+\'px\'}})','src'=>'https://247sports.com/PlayerSport/{@player_id}/Embed/','width'=>600]],'when'=>['iframe'=>['src'=>'https://www.cbssports.com/video/player/embed?args=player_id%3D{@video_id}%26channel%3Dmobilevideo%26pcid%3D{@video_id}%26width%3D640%26height%3D360%26autoplay%3Dfalse%26comp_ads_enabled%3Dfalse%26uvpc%3Dhttp%3A%2F%2Fsports.cbsimg.net%2Fvideo%2Fuvp%2Fconfig%2Fv4%2Fuvp_247sports.xml%26uvpc_m%3Dhttp%3A%2F%2Fsports.cbsimg.net%2Fvideo%2Fuvp%2Fconfig%2Fv4%2Fuvp_247sports_m.xml%26partner%3D247%26partner_m%3D247_mobile%26utag%3D247sportssite%26resizable%3Dtrue'],'test'=>'@video_id']],'example'=>['https://247sports.com/Player/LeSean-McCoy-25234/','https://247sports.com/Video/FFT-Week-12-QBRB-Start-or-Sit-1122-8838631/'],'extract'=>['!247sports\\.com/PlayerSport/[-\\w]*?(?\'player_id\'\\d+)/Embed!'],'homepage'=>'https://247sports.com/','host'=>['247sports.com'],'name'=>'247Sports','scrape'=>[['extract'=>['!247sports\\.com/PlayerSport/[-\\w]*?(?\'player_id\'\\d+)/Embed!'],'header'=>'User-agent: Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/73.0.3683.86 Safari/537.36','match'=>['!247sports\\.com/Player/[-\\w]*?\\d!']],['extract'=>['!player_id%3D(?\'video_id\'\\d+)!'],'header'=>'User-agent: Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/73.0.3683.86 Safari/537.36','match'=>['!247sports\\.com/Video/.!']]],'tags'=>['sports']],
		'twitch'=>['attributes'=>[],'example'=>['https://www.twitch.tv/twitch','https://www.twitch.tv/twitch/v/29415830?t=17m17s','https://www.twitch.tv/videos/29415830?t=17m17s','https://clips.twitch.tv/twitch/HorribleWoodpeckerHassanChop','https://clips.twitch.tv/NeighborlyBetterJellyfishWTRuck'],'extract'=>['#twitch\\.tv/(?:videos|\\w+/v)/(?\'video_id\'\\d+)?#','#www\\.twitch\\.tv/(?!videos/)(?\'channel\'\\w+)#','#t=(?\'t\'(?:(?:\\d+h)?\\d+m)?\\d+s)#','#clips\\.twitch\\.tv/(?:(?\'channel\'\\w+)/)?(?\'clip_id\'\\w+)#'],'homepage'=>'https://www.twitch.tv/','host'=>['twitch.tv'],'iframe'=>['src'=>'//<xsl:choose><xsl:when test="@clip_id">clips.twitch.tv/embed?autoplay=false&amp;clip=<xsl:if test="@channel"><xsl:value-of select="@channel"/>/</xsl:if><xsl:value-of select="@clip_id"/></xsl:when><xsl:otherwise>player.twitch.tv/?autoplay=false&amp;<xsl:choose><xsl:when test="@video_id">video=v<xsl:value-of select="@video_id"/></xsl:when><xsl:otherwise>channel=<xsl:value-of select="@channel"/></xsl:otherwise></xsl:choose><xsl:if test="@t">&amp;time=<xsl:value-of select="@t"/></xsl:if></xsl:otherwise></xsl:choose>'],'name'=>'Twitch','scrape'=>[],'source'=>'https://github.com/justintv/Twitch-API/blob/master/embed-video.md','tags'=>['gaming','livestreaming']],
		'twitter'=>['attributes'=>[],'example'=>['https://twitter.com/IJasonAlexander/statuses/526635414338023424','https://mobile.twitter.com/DerekTVShow/status/463372588690202624','https://twitter.com/#!/IJasonAlexander/status/526635414338023424'],'extract'=>['@twitter\\.com/(?:#!/|i/)?\\w+/(?:status(?:es)?|tweet)/(?\'id\'\\d+)@'],'homepage'=>'https://twitter.com/','host'=>['twitter.com'],'iframe'=>['allow'=>'autoplay *','data-s9e-livepreview-ignore-attrs'=>'style','height'=>250,'onload'=>'var c=new MessageChannel;c.port1.onmessage=function(e){{style.height=e.data+\'px\'}};contentWindow.postMessage(\'s9e:init\',\'https://s9e.github.io\',[c.port2])','src'=>'https://s9e.github.io/iframe/2/twitter.min.html#{@id}','style'=>['background'=>'url(https://abs.twimg.com/favicons/favicon.ico) no-repeat 50% 50%'],'width'=>500],'name'=>'Twitter','scrape'=>[],'tags'=>['social']],
		'ustream'=>['attributes'=>[],'choose'=>['otherwise'=>['iframe'=>['src'=>'//www.ustream.tv/embed/{@cid}?html5ui']],'when'=>['iframe'=>['src'=>'//www.ustream.tv/embed/recorded/{@vid}?html5ui'],'test'=>'@vid']],'example'=>['https://www.ustream.tv/channel/ps4-ustream-gameplay','https://www.ustream.tv/baja1000tv','https://www.ustream.tv/recorded/40688256'],'extract'=>['!ustream\\.tv/recorded/(?\'vid\'\\d+)!'],'homepage'=>'https://www.ustream.tv/','host'=>['ustream.tv'],'name'=>'Ustream','scrape'=>[['extract'=>['!embed/(?\'cid\'\\d+)!'],'match'=>['#ustream\\.tv/(?!explore/|platform/|recorded/|search\\?|upcoming$|user/)(?:channel/)?[-\\w]+#']]],'tags'=>['gaming']],
		'vbox7'=>['attributes'=>[],'example'=>'https://vbox7.com/play:3975300ec6','extract'=>['!vbox7\\.com/play:(?\'id\'[\\da-f]+)!'],'homepage'=>'https://vbox7.com/','host'=>['vbox7.com'],'iframe'=>['src'=>'//vbox7.com/emb/external.php?vid={@id}'],'name'=>'VBOX7','scrape'=>[],'tags'=>['.bg']],
		'veoh'=>['attributes'=>[],'example'=>'https://www.veoh.com/watch/v6335577TeB8kyNR','extract'=>['!veoh\\.com/(?:m/watch\\.php\\?v=|watch/)v(?\'id\'\\w+)!'],'flash'=>['padding-height'=>40,'src'=>'//www.veoh.com/swf/webplayer/WebPlayer.swf?version=AFrontend.5.7.0.1509&permalinkId=v{@id}&player=videodetailsembedded&videoAutoPlay=0&id=anonymous'],'homepage'=>'https://www.veoh.com/','host'=>['veoh.com'],'name'=>'Veoh','scrape'=>[],'tags'=>['videos']],
		'vevo'=>['attributes'=>[],'example'=>['https://www.vevo.com/watch/USUV71400682','https://www.vevo.com/watch/eminem/the-monster-explicit/USUV71302925'],'extract'=>['!vevo\\.com/watch/(.*?/)?(?\'id\'[A-Z]+\\d+)!'],'homepage'=>'https://vevo.com/','host'=>['vevo.com'],'iframe'=>['src'=>'https://embed.vevo.com/?isrc={@id}'],'name'=>'Vevo','scrape'=>[],'tags'=>['music']],
		'videodetective'=>['attributes'=>[],'example'=>'https://www.videodetective.com/movies/zootopia/658596','extract'=>['!videodetective\\.com/\\w+/[-\\w]+/(?:trailer/P0*)?(?\'id\'\\d+)!'],'homepage'=>'https://www.videodetective.com/','host'=>['videodetective.com'],'iframe'=>['src'=>'//www.videodetective.com/embed/video/?options=false&autostart=false&playlist=none&publishedid={@id}'],'name'=>'Video Detective','scrape'=>[]],
		'vimeo'=>['attributes'=>['t'=>['filterChain'=>['#timestamp']]],'example'=>['https://vimeo.com/67207222','https://vimeo.com/channels/staffpicks/67207222'],'extract'=>['!vimeo\\.com/(?:channels/[^/]+/|video/)?(?\'id\'\\d+)!','!#t=(?\'t\'[\\dhms]+)!'],'homepage'=>'https://vimeo.com/','host'=>['vimeo.com'],'iframe'=>['src'=>'//player.vimeo.com/video/<xsl:value-of select="@id"/><xsl:if test="@t">#t=<xsl:value-of select="@t"/></xsl:if>'],'name'=>'Vimeo','scrape'=>[],'source'=>'http://developer.vimeo.com/player/embedding','tags'=>['videos']],
		'vine'=>['attributes'=>[],'example'=>'https://vine.co/v/bYwPIluIipH','extract'=>['!vine\\.co/v/(?\'id\'[^/]+)!'],'homepage'=>'https://vine.co/','host'=>['vine.co'],'iframe'=>['height'=>480,'src'=>'https://vine.co/v/{@id}/embed/simple?audio=1','width'=>480],'name'=>'Vine','scrape'=>[],'tags'=>['social','videos']],
		'vk'=>['attributes'=>[],'example'=>['https://vk.com/video121599878_165723901?hash=e06b0878046e1d32','https://vk.com/video_ext.php?oid=121599878&id=165723901&hash=e06b0878046e1d32'],'extract'=>['!vk(?:\\.com|ontakte\\.ru)/(?:[\\w.]+\\?z=)?video(?\'oid\'-?\\d+)_(?\'vid\'\\d+).*?hash=(?\'hash\'[0-9a-f]+)!','!vk(?:\\.com|ontakte\\.ru)/video_ext\\.php\\?oid=(?\'oid\'-?\\d+)&id=(?\'vid\'\\d+)&hash=(?\'hash\'[0-9a-f]+)!'],'homepage'=>'https://vk.com/','host'=>['vk.com','vkontakte.ru'],'iframe'=>['src'=>'//vk.com/video_ext.php?oid={@oid}&id={@vid}&hash={@hash}&hd=1'],'name'=>'VK','scrape'=>[],'tags'=>['.ru']],
		'vocaroo'=>['attributes'=>[],'example'=>'https://vocaroo.com/i/s0dRy3rZ47bf','extract'=>['!vocaroo\\.com/i/(?\'id\'\\w+)!'],'flash'=>['height'=>44,'src'=>'//vocaroo.com/player.swf?playMediaID={@id}&autoplay=0','width'=>148],'homepage'=>'https://vocaroo.com/','host'=>['vocaroo.com'],'name'=>'Vocaroo','scrape'=>[]],
		'vox'=>['attributes'=>[],'example'=>'https://www.vox.com/2015/7/21/9005857/ant-man-marvel-apology-review#ooid=ltbzJkdTpKpE-O6hOfD3YJew3t3MppXb','extract'=>['!vox.com/.*#ooid=(?\'id\'[-\\w]+)!'],'homepage'=>'https://www.vox.com/','host'=>['vox.com'],'iframe'=>['src'=>'//player.ooyala.com/iframe.html#pbid=a637d53c5c0a43c7bf4e342886b9d8b0&ec={@id}'],'name'=>'Vox','scrape'=>[]],
		'washingtonpost'=>['attributes'=>[],'example'=>['https://www.washingtonpost.com/video/c/video/df229384-9216-11e6-bc00-1a9756d4111b','https://www.washingtonpost.com/video/world/aurora-display-lights-up-the-night-sky-over-finland/2016/10/14/df229384-9216-11e6-bc00-1a9756d4111b_video.html'],'extract'=>['#washingtonpost\\.com/video/c/\\w+/(?\'id\'[-0-9a-f]+)#','#washingtonpost\\.com/video/[-/\\w]+/(?\'id\'[-0-9a-f]+)_video\\.html#'],'homepage'=>'https://www.washingtonpost.com/video/','host'=>['washingtonpost.com'],'iframe'=>['src'=>'//www.washingtonpost.com/video/c/embed/{@id}'],'name'=>'Washington Post Video','scrape'=>[],'tags'=>['news']],
		'wshh'=>['attributes'=>[],'example'=>['https://www.worldstarhiphop.com/videos/video.php?v=wshhZ8F22UtJ8sLHdja0','https://m.worldstarhiphop.com/video.php?v=wshh2SXFFe7W14DqQx61','https://www.worldstarhiphop.com/featured/71630'],'extract'=>['!worldstarhiphop\\.com/featured/(?\'id\'\\d+)!'],'homepage'=>'https://www.worldstarhiphop.com/','host'=>['worldstarhiphop.com'],'iframe'=>['src'=>'//www.worldstarhiphop.com/embed/{@id}'],'name'=>'WorldStarHipHop','scrape'=>[['extract'=>['!v: ?"?(?\'id\'\\d+)!'],'match'=>['!worldstarhiphop\\.com/(?:\\w+/)?video\\.php\\?v=\\w+!']]],'tags'=>['videos']],
		'wsj'=>['attributes'=>[],'example'=>['https://www.wsj.com/video/nba-players-primp-with-pedicures/9E476D54-6A60-4F3F-ABC1-411014552DE6.html','https://live.wsj.com/#!09FB2B3B-583E-4284-99D8-FEF6C23BE4E2','https://live.wsj.com/video/seahawks-qb-russell-wilson-on-super-bowl-win/9B3DF790-9D20-442C-B564-51524B06FD26.html'],'extract'=>['@wsj\\.com/[^#]*#!(?\'id\'[-0-9A-F]{36})@','@wsj\\.com/video/[^/]+/(?\'id\'[-0-9A-F]{36})@'],'homepage'=>'https://www.wsj.com/video/','host'=>['wsj.com'],'iframe'=>['height'=>288,'src'=>'//video-api.wsj.com/api-video/player/iframe.html?guid={@id}','width'=>512],'name'=>'The Wall Street Journal Online','scrape'=>[['extract'=>['@guid=(?\'id\'[-0-9A-F]{36})@'],'match'=>['@on\\.wsj\\.com/\\w@']]],'tags'=>['news']],
		'xboxclips'=>['attributes'=>[],'example'=>'https://xboxclips.com/dizturbd/e3a2d685-3e9f-454f-89bf-54ddea8f29b3','extract'=>['@xboxclips\\.com/(?\'user\'[^/]+)/(?!screenshots/)(?\'id\'[-0-9a-f]+)@'],'homepage'=>'https://xboxclips.com/','host'=>['xboxclips.com'],'iframe'=>['src'=>'//xboxclips.com/{@user}/{@id}/embed'],'name'=>'XboxClips','scrape'=>[],'tags'=>['gaming']],
		'xboxdvr'=>['attributes'=>[],'example'=>'https://gamerdvr.com/gamer/LOXITANE/video/12463958','extract'=>['!(?:gamer|xbox)dvr\\.com/gamer/(?\'user\'[^/]+)/video/(?\'id\'\\d+)!'],'homepage'=>'https://gamerdvr.com/','host'=>['gamerdvr.com','xboxdvr.com'],'iframe'=>['src'=>'//gamerdvr.com/gamer/{@user}/video/{@id}/embed'],'name'=>'Gamer DVR','scrape'=>[],'tags'=>['gaming']],
		'youku'=>['attributes'=>[],'example'=>'https://v.youku.com/v_show/id_XMzY0NTMyMTgxMg==.html','extract'=>['!youku\\.com/v(?:_show|ideo)/id_(?\'id\'\\w+=*)!'],'homepage'=>'https://www.youku.com/','host'=>['youku.com'],'iframe'=>['src'=>'//player.youku.com/embed/{@id}'],'name'=>'Youku','scrape'=>[],'tags'=>['.cn']],
		'youtube'=>['attributes'=>['id'=>['filterChain'=>['#identifier'],'required'=>false],'t'=>['filterChain'=>['#timestamp']]],'example'=>['https://www.youtube.com/watch?v=-cEzsCAzTak','https://youtu.be/-cEzsCAzTak','https://www.youtube.com/watch?feature=player_detailpage&v=jofNR_WkoCE#t=40','https://www.youtube.com/watch?v=pC35x6iIPmo&list=PLOU2XLYxmsIIxJrlMIY5vYXAFcO5g83gA'],'extract'=>['!youtube\\.com/(?:watch.*?v=|v/|attribution_link.*?v%3D)(?\'id\'[-\\w]+)!','!youtu\\.be/(?\'id\'[-\\w]+)!','@[#&?]t=(?\'t\'\\d[\\dhms]*)@','![&?]list=(?\'list\'[-\\w]+)!'],'homepage'=>'https://www.youtube.com/','host'=>['youtube.com','youtu.be'],'iframe'=>['src'=>'https://www.youtube.com/embed/<xsl:value-of select="@id"/><xsl:if test="@list">?list=<xsl:value-of select="@list"/></xsl:if><xsl:if test="@t"><xsl:choose><xsl:when test="@list">&amp;</xsl:when><xsl:otherwise>?</xsl:otherwise></xsl:choose>start=<xsl:value-of select="@t"/></xsl:if>','style'=>['background'=>'url(https://i.ytimg.com/vi/{@id}/hqdefault.jpg) 50% 50% / cover']],'name'=>'YouTube','scrape'=>[['extract'=>['!/vi/(?\'id\'[-\\w]+)!'],'match'=>['!/shared\\?ci=!']]],'source'=>'https://support.google.com/youtube/bin/answer.py?hl=en&answer=171780','tags'=>['livestreaming','videos']]
	];
}