(***********************************************************************)
(*                                                                     *)
(*                           Objective Caml                            *)
(*                                                                     *)
(*             Damien Doligez, projet Para, INRIA Rocquencourt         *)
(*                                                                     *)
(*  Copyright 1996 Institut National de Recherche en Informatique et   *)
(*  en Automatique.  All rights reserved.  This file is distributed    *)
(*  under the terms of the GNU Library General Public License, with    *)
(*  the special exception on linking described in file ../LICENSE.     *)
(*                                                                     *)
(***********************************************************************)

(* $Id: gc.mli,v 1.42 2005-10-25 18:34:07 doligez Exp $ *)

(** Memory management control and statistics; finalised values. *)

module Stat : sig
  type t = {
    minor_words : float;
    (** Number of words allocated in the minor heap since
       the program was started.  This number is accurate in
       byte-code programs, but only an approximation in programs
       compiled to native code. *)

    promoted_words : float;
    (** Number of words allocated in the minor heap that
       survived a minor collection and were moved to the major heap
       since the program was started. *)

    major_words : float;
    (** Number of words allocated in the major heap, including
       the promoted words, since the program was started. *)

    minor_collections : int;
    (** Number of minor collections since the program was started. *)

    major_collections : int;
    (** Number of major collection cycles completed since the program
        was started. *)

    heap_words : int;
    (** Total size of the major heap, in words. *)

    heap_chunks : int;
    (** Number of contiguous pieces of memory that make up the major heap. *)

    live_words : int;
    (** Number of words of live data in the major heap, including the header
       words. *)

    live_blocks : int;
    (** Number of live blocks in the major heap. *)

    free_words : int;
    (** Number of words in the free list. *)

    free_blocks : int;
    (** Number of blocks in the free list. *)

    largest_free : int;
    (** Size (in words) of the largest block in the free list. *)

    fragments : int;
    (** Number of wasted words due to fragmentation.  These are
       1-words free blocks placed between two live blocks.  They
       are not available for allocation. *)

    compactions : int;
    (** Number of heap compactions since the program was started. *)

    top_heap_words : int;
    (** Maximum size reached by the major heap, in words. *)

    stack_size : int
    (** Current size of the stack, in words. *)
  }
  with bin_io, sexp, fields
end

type stat = Stat.t

(** The memory management counters are returned in a [stat] record.

   The total amount of memory allocated by the program since it was started
   is (in words) [minor_words + major_words - promoted_words].  Multiply by
   the word size (4 on a 32-bit machine, 8 on a 64-bit machine) to get
   the number of bytes.
*)

module Control : sig
  type t = {
    mutable minor_heap_size : int;
    (** The size (in words) of the minor heap.  Changing
       this parameter will trigger a minor collection.  Default: 1M. *)

    mutable major_heap_increment : int;
    (** The minimum number of words to add to the
       major heap when increasing it.  Default: 1M. *)

    mutable space_overhead : int;
    (** The major GC speed is computed from this parameter.
        This is the memory that will be "wasted" because the GC does not
       immediatly collect unreachable blocks.  It is expressed as a
       percentage of the memory used for live data.
       The GC will work more (use more CPU time and collect
       blocks more eagerly) if [space_overhead] is smaller.
       Default: 100. *)

    mutable verbose : int;
    (** This value controls the GC messages on standard error output.
       It is a sum of some of the following flags, to print messages
       on the corresponding events:
       - [0x001] Start of major GC cycle.
       - [0x002] Minor collection and major GC slice.
       - [0x004] Growing and shrinking of the heap.
       - [0x008] Resizing of stacks and memory manager tables.
       - [0x010] Heap compaction.
       - [0x020] Change of GC parameters.
       - [0x040] Computation of major GC slice size.
       - [0x080] Calling of finalisation functions.
       - [0x100] Bytecode executable search at start-up.
       - [0x200] Computation of compaction triggering condition.
       Default: 0. *)

    mutable max_overhead : int;
    (** Heap compaction is triggered when the estimated amount
       of "wasted" memory is more than [max_overhead] percent of the
       amount of live data.  If [max_overhead] is set to 0, heap
       compaction is triggered at the end of each major GC cycle
       (this setting is intended for testing purposes only).
       If [max_overhead >= 1000000], compaction is never triggered.
       Default: 500. *)

    mutable stack_limit : int;
    (** The maximum size of the stack (in words).  This is only
       relevant to the byte-code runtime, as the native code runtime
       uses the operating system's stack.  Default: 256k. *)
    mutable allocation_policy : int;
    (** The policy used for allocating in the heap.  Possible
        values are 0 and 1.  0 is the next-fit policy, which is
        quite fast but can result in fragmentation.  1 is the
        first-fit policy, which can be slower in some cases but
        can be better for programs with fragmentation problems.
        Default: 0. *)
  }
  with bin_io, sexp, fields
end

type control = Control.t

(** The GC parameters are given as a [control] record.
    Note that these parameters can also be initialised
    by setting the OCAMLRUNPARAM environment variable.
    See the documentation of ocamlrun. *)

external stat : unit -> stat = "caml_gc_stat"
(** Return the current values of the memory management counters in a
   [stat] record.  This function examines every heap block to get the
   statistics. *)

external quick_stat : unit -> stat = "caml_gc_quick_stat"
(** Same as [stat] except that [live_words], [live_blocks], [free_words],
    [free_blocks], [largest_free], and [fragments] are set to 0.  This
    function is much faster than [stat] because it does not need to go
    through the heap. *)

external counters : unit -> float * float * float = "caml_gc_counters"
(** Return [(minor_words, promoted_words, major_words)].  This function
    is as fast at [quick_stat]. *)

(** The following functions return the same as [(Gc.quick_stat ()).Stat.f], avoiding any
    allocation (of the [stat] record or a float).  On 32-bit machines the [int] may
    overflow. *)
external minor_words       : unit -> int = "core_kernel_gc_minor_words"       "noalloc"
external major_words       : unit -> int = "core_kernel_gc_major_words"       "noalloc"
external promoted_words    : unit -> int = "core_kernel_gc_promoted_words"    "noalloc"
external minor_collections : unit -> int = "core_kernel_gc_minor_collections" "noalloc"
external major_collections : unit -> int = "core_kernel_gc_major_collections" "noalloc"
external heap_words        : unit -> int = "core_kernel_gc_heap_words"        "noalloc"
external heap_chunks       : unit -> int = "core_kernel_gc_heap_chunks"       "noalloc"
external compactions       : unit -> int = "core_kernel_gc_compactions"       "noalloc"
external top_heap_words    : unit -> int = "core_kernel_gc_top_heap_words"    "noalloc"

external get : unit -> control = "caml_gc_get"
(** Return the current values of the GC parameters in a [control] record. *)

external set : control -> unit = "caml_gc_set"
(** [set r] changes the GC parameters according to the [control] record [r].
    The normal usage is:
      [Gc.set { (Gc.get()) with Gc.Control.verbose = 0x00d }] *)

external minor : unit -> unit = "caml_gc_minor"
(** Trigger a minor collection. *)

external major_slice : int -> int = "caml_gc_major_slice"
(** Do a minor collection and a slice of major collection.  The argument
    is the size of the slice, 0 to use the automatically-computed
    slice size.  In all cases, the result is the computed slice size. *)

external major : unit -> unit = "caml_gc_major"
(** Do a minor collection and finish the current major collection cycle. *)

external full_major : unit -> unit = "caml_gc_full_major"
(** Do a minor collection, finish the current major collection cycle,
   and perform a complete new cycle.  This will collect all currently
   unreachable blocks. *)

external compact : unit -> unit = "caml_gc_compaction"
(** Perform a full major collection and compact the heap.  Note that heap
   compaction is a lengthy operation. *)

val print_stat : out_channel -> unit
(** Print the current values of the memory management counters (in
   human-readable form) into the channel argument. *)

val allocated_bytes : unit -> float
(** Return the total number of bytes allocated since the program was
   started.  It is returned as a [float] to avoid overflow problems
   with [int] on 32-bit machines. *)

type alarm
(** An alarm is a piece of data that calls a user function at the end of
   each major GC cycle.  The following functions are provided to create
   and delete alarms. *)

val create_alarm : (unit -> unit) -> alarm
(** [create_alarm f] will arrange for [f] to be called at the end of each
   major GC cycle, starting with the current cycle or the next one.
   A value of type [alarm] is returned that you can
   use to call [delete_alarm]. *)

val delete_alarm : alarm -> unit
(** [delete_alarm a] will stop the calls to the function associated
   to [a].  Calling [delete_alarm a] again has no effect. *)

(** Adjust the specified GC parameters. *)
val tune
  :  ?logger:(string -> unit)
  -> ?minor_heap_size:int
  -> ?major_heap_increment:int
  -> ?space_overhead:int
  -> ?verbose:int
  -> ?max_overhead:int
  -> ?stack_limit:int
  -> ?allocation_policy:int
  -> unit
  -> unit
