#include <algorithm>

#include "framecpp/Common/Description.hh"
#include "framecpp/Common/TOCInfo.hh"

#include "framecpp/Version8/FrameSpec.hh"
#include "framecpp/Version8/FrSE.hh"
#include "framecpp/Version8/FrSH.hh"
#include "framecpp/Version8/FrTOC.hh"

#include "framecpp/Version8/STRING.hh"

using FrameCPP::Common::Description;
using FrameCPP::Common::FrameSpec;
using FrameCPP::Common::TOCInfo;

template < typename T >
class sum
{
public:
  inline sum ( T i = 0 )
    : res( i )
  {
  }

  inline void operator( )( T x )
  {
    if ( x != FrameCPP::Version_8::FrTOC::NO_DATA_AVAILABLE )
    {
      res += x;
    }
  }

  inline T result( ) const
  {
    return res;
  }
private:
  T res;

};

namespace FrameCPP
{
  namespace Version_8
  {
    //===================================================================
    //===================================================================
    FrTOCEvent::
    FrTOCEvent( )
    {
    }

    FrTOCEvent::
    FrTOCEvent( Common::IStream& Stream )
    {
      nevent_type	nevent;

      Stream >> nevent;

      if ( nevent )
      {
	//---------------------------------------------------------------
	// Read in the information
	//---------------------------------------------------------------
	std::vector< name_type >	names( nevent );
	std::vector< nevent_type >	events( nevent );

	Stream >> names
	       >> events
	  ;

	sum< nevent_type >	nevent_sum;
	nevent_type		offset = 0;

	for ( std::vector< nevent_type >::const_iterator
		cur = events.begin( ),
		last = events.end( );
	      cur != last;
	      ++cur )
	{
	  nevent_sum( *cur );
	}

	nTotalEvent_type	nTotalEvent;

	Stream >> nTotalEvent;

	if ( nTotalEvent != nevent_sum.result( ) )
	{
	  throw std::runtime_error( "nTotalEvent is not the sumation of nEvent" );
	}

	std::vector< gtimesEvent_type >
	  gtimes( nevent_sum.result( ) );
	std::vector< gtimenEvent_type >
	  gtimen( nevent_sum.result( ) );
	std::vector< amplitudeEvent_type >
	  amplitude( nevent_sum.result( ) );
	std::vector< positionEvent_type >
	  position( nevent_sum.result( ) );
	
	Stream >> gtimes
	       >> gtimen
	       >> amplitude
	       >> position
	  ;
	std::vector< nevent_type >::const_iterator
	  cur_event_counter = events.begin( );

	for ( std::vector< name_type >::const_iterator
		cur = names.begin( ),
		last = names.end( );
	      cur != last;
	      ++cur, ++cur_event_counter )
	{
	  events_container_type&	cur_events = m_info[ *cur ];

	  if ( ( *cur_event_counter == 0 )
	       || ( *cur_event_counter == FrTOC::NO_DATA_AVAILABLE ) )
	  {
	    continue;
	  }
	  cur_events.resize( *cur_event_counter );
	  for ( nevent_type
		  x = 0,
		  x_last = *cur_event_counter;
		x != x_last;
		++x, ++offset )
	  {
	    cur_events[ x ].GTime =
	      GPSTime( gtimes[ offset ], gtimen[ offset ] );
	    cur_events[ x ].amplitudeEvent = amplitude[ offset ];
	    cur_events[ x ].positionEvent = position[ offset ];
	  }
	}
      }
      else
      {
	nevent_type	tevent;

	Stream >> tevent;
      }
    }

    void FrTOCEvent::
    QueryEvent( const Common::TOCInfo& Info,
		   INT_4U FrameOffset,
		   INT_8U Position )
    {
      STRING	name;
      INT_4U	sec;
      INT_4U	nsec;
      REAL_4	ampl;

      Info.TOCQuery( TOCInfo::IC_NAME, TOCInfo::DataType( name ), &name,
		     TOCInfo::IC_GTIME_S, TOCInfo::DataType( sec ), &sec,
		     TOCInfo::IC_GTIME_N, TOCInfo::DataType( nsec ), &nsec,
		     TOCInfo::IC_AMPLITUDE, TOCInfo::DataType( ampl ), &ampl,
		     TOCInfo::IC_EOQ );

      events_container_type& i( m_info[ name ] );

      event_type	e;
      e.GTime = GPSTime( sec, nsec );
      e.amplitudeEvent = ampl;
      e.positionEvent = Position;

      i.push_back( e );
    }

    FrTOCEvent::positionEvent_type FrTOCEvent::
    positionEvent( INT_4U FrameIndex, const std::string& Event ) const
    {
      const nameEvent_container_type::const_iterator
	event( m_info.find( Event ) );
      //-----------------------------------------------------------------
      // Locate the event by name
      //-----------------------------------------------------------------
      if ( event == m_info.end( ) )
      {
	//---------------------------------------------------------------
	// Event name does not exist.
	//---------------------------------------------------------------
	std::ostringstream	msg;

	msg << "No FrEvent structures with the name '" << Event << "'"
	  ;
	throw
	  std::out_of_range( msg.str( ) );
      }
      //-----------------------------------------------------------------
      // Verify that the index exists
      //-----------------------------------------------------------------
      if ( FrameIndex >= event->second.size( ) )
      {
	std::ostringstream	msg;

	msg << "Request for frame " << FrameIndex
	    << " exceeds the range of 0 through " << ( event->second.size( ) - 1 )
	  ;
	throw
	  std::out_of_range( msg.str( ) );
      }
      //-----------------------------------------------------------------
      // Return position information
      //-----------------------------------------------------------------
      return event->second[ FrameIndex ].positionEvent;
    }

    FrTOCEvent::positionEvent_type FrTOCEvent::
    positionEvent( const std::string& Event, INT_4U Index ) const
    {
      const nameEvent_container_type::const_iterator
	event( m_info.find( Event ) );
      //-----------------------------------------------------------------
      // Locate the event by name
      //-----------------------------------------------------------------
      if ( event == m_info.end( ) )
      {
	//---------------------------------------------------------------
	// Event name does not exist.
	//---------------------------------------------------------------
	std::ostringstream	msg;

	msg << "No FrEvent structures with the name '" << Event << "'"
	  ;
	throw
	  std::out_of_range( msg.str( ) );
      }
      //-----------------------------------------------------------------
      // Verify that the index exists
      //-----------------------------------------------------------------
      if ( Index >= event->second.size( ) )
      {
	std::ostringstream	msg;

	msg << "Request for index " << Index
	    << " exceeds the range of 0 through " << ( event->second.size( ) - 1 )
	  ;
	throw
	  std::out_of_range( msg.str( ) );
      }
      //-----------------------------------------------------------------
      // Return position information
      //-----------------------------------------------------------------
      return event->second[ Index ].positionEvent;
    }

    void FrTOCEvent::
    write( Common::OStream& Stream ) const
    {
      //-----------------------------------------------------------------
      // Flatten data so it is streamable
      //-----------------------------------------------------------------
      if ( m_info.size( ) > 0 ) {
	std::vector< name_type >	names( m_info.size( ) );
	std::vector< nevent_type >	nevent( m_info.size( ) );
	nevent_type			offset( 0 );
	nevent_type			eoffset( 0 );
	std::vector< gtimesEvent_type >		gtimes;
	std::vector< gtimenEvent_type >		gtimen;
	std::vector< amplitudeEvent_type >	amplitude;
	std::vector< positionEvent_type >	position;

	for ( nameEvent_container_type::const_iterator
		cur = m_info.begin( ),
		last = m_info.end( );
	      cur != last;
	      ++cur, ++offset )
	{
	  names[ offset ] = cur->first;
	  const nevent_type
	    c = ( cur->second.size( ) == 0 )
	    ? ( FrTOC::NO_DATA_AVAILABLE )
	    : ( cur->second.size( ) )
	    ;
	  nevent[ offset ] = c;
	  if ( c != FrTOC::NO_DATA_AVAILABLE )
	  {
	    const int ns( c + eoffset );

	    gtimes.resize( ns );
	    gtimen.resize( ns );
	    amplitude.resize( ns );
	    position.resize( ns );

	    for ( nevent_type x = 0;
		  x != c;
		  ++x, ++eoffset )
	    {
	      gtimes[ eoffset ] = cur->second[ x ].GTime.GetSeconds( );
	      gtimen[ eoffset ] = cur->second[ x ].GTime.GetNanoseconds( );
	      amplitude[ eoffset ] = cur->second[ x ].amplitudeEvent;
	      position[ eoffset ] = cur->second[ x ].positionEvent;
	    }
	  }
	}

	const nTotalEvent_type	nTotalEvent( gtimes.size( ) );

	Stream
	  << nevent_type( m_info.size( ) )
	  << names
	  << nevent
	  << nTotalEvent
	  << gtimes
	  << gtimen
	  << amplitude
	  << position
	  ;
      }
      else
      {
	Stream << nevent_type( 0 ); // nEventType
	Stream << nevent_type( 0 ); // nTotalEvent
      }
    }

    const FrTOCEvent& FrTOCEvent::
    operator=( const Previous::FrTOCEvent& Source )
    {
      m_info.erase( m_info.begin( ), m_info.end( ) );
      const Previous::FrTOCEvent::nameEvent_container_type&
	data( Source.GetEvent( ) );

      for ( Previous::FrTOCEvent::nameEvent_container_type::const_iterator
	      cur = data.begin( ),
	      last = data.end( );
	    cur != last;
	    ++cur )
      {
	m_info[ cur->first ].resize( cur->second.size( ) );
	std::copy( cur->second.begin( ),
		   cur->second.end( ),
		   m_info[ cur->first ].begin( ) );
      }

      return *this;
    }

  } // namespace - Version_8
} // namespace - FrameCPP
