#ifndef FRAMECPP__COMMON__FrameStreamWrapper_HH
#define FRAMECPP__COMMON__FrameStreamWrapper_HH

#include "framecpp/Common/FrameStream.hh"

namespace FrameCPP
{
  namespace Common
  {
#define DECL INT_2U SPEC, typename AdcData, typename FrameH, typename ProcData, typename SerData, typename SimData, typename RawData, typename Event, typename SimEvent
#define DECL_PARAMS SPEC, AdcData, FrameH, ProcData, SerData, SimData, RawData, Event, SimEvent

    //-------------------------------------------------------------------
    /// \brief Wrapper to properly cast return values into the namespace
    ///        associated with the frame specification namespace.
    //-------------------------------------------------------------------
    template< DECL >
    class IFrameStreamWrapper
      : public IFrameStream
    {
    public:
      typedef INT_4U frame_offset_type;

      typedef typename LDASTools::AL::SharedPtr< FrameH > frame_h_type;
      typedef typename LDASTools::AL::SharedPtr< AdcData > fr_adc_data_type;
      typedef typename LDASTools::AL::SharedPtr< Event > fr_event_type;
      typedef typename LDASTools::AL::SharedPtr< ProcData > fr_proc_data_type;
      typedef typename LDASTools::AL::SharedPtr< RawData > fr_raw_data_type;
      typedef typename LDASTools::AL::SharedPtr< SerData > fr_ser_data_type;
      typedef typename LDASTools::AL::SharedPtr< SimData > fr_sim_data_type;
      typedef typename LDASTools::AL::SharedPtr< SimEvent > fr_sim_event_type;

      //-----------------------------------------------------------------
      /// \brief Definition of buffer_type.
      //-----------------------------------------------------------------
      typedef IFrameStream::buffer_type buffer_type;

      //-----------------------------------------------------------------
      /// \brief Constructor
      //-----------------------------------------------------------------
      IFrameStreamWrapper( buffer_type* Buffer );

      //-----------------------------------------------------------------
      /// \brief Constructor
      //-----------------------------------------------------------------
      IFrameStreamWrapper( bool AutoDelete, buffer_type* Buffer );

      //-----------------------------------------------------------------
      //-----------------------------------------------------------------
      std::string FrameLibraryName( ) const;

      //-----------------------------------------------------------------
      /// \brief Extract the requested FrameH structure from the stream
      ///
      /// \param[in] Frame
      ///     Zero based index of the frame.
      /// \param[in] ContainerSet
      ///     Bit map of components to included.
      ///
      /// \return
      ///     Upon successful completion, the address of the FrameH
      ///     structure linked with all requested subcomponents
      ///     by ContainerSet.
      //-----------------------------------------------------------------
      frame_h_type ReadFrameH( frame_offset_type Frame, INT_4U ContainerSet );

      //-----------------------------------------------------------------
      /// \brief Extract the requested FrAdcData structure from the stream
      ///
      /// \param[in] Frame
      ///     Zero based index of the frame.
      /// \param[in] Channel
      ///     Name of the channel being requested.
      ///
      /// \return
      ///     Upon success, a non-NULL pointer to the FrAdcData is
      ///     returned. Upon failure, either an exception is thrown
      //      or a NULL pointer is returned.
      //-----------------------------------------------------------------
      fr_adc_data_type ReadFrAdcData( frame_offset_type Frame,
				      const std::string& Channel );
      //-----------------------------------------------------------------
      /// \brief Extract the requested FrAdcData structure from the stream
      ///
      /// \param[in] Frame
      ///     Zero based index of the frame.
      /// \param[in] Channel
      ///     Zero based index of channel being requested.
      ///
      /// \return
      ///     Upon success, a non-NULL pointer to the FrAdcData is
      ///     returned. Upon failure, either an exception is thrown
      //      or a NULL pointer is returned.
      //-----------------------------------------------------------------
      fr_adc_data_type ReadFrAdcData( frame_offset_type Frame, INT_4U Channel );
      //-----------------------------------------------------------------
      /// \brief Extract the requested FrAdcData structure from the stream
      ///
      /// \param[in] Frame
      ///     Zero based index of the frame.
      /// \param[in] Channel
      ///     Name of the channel being requested.
      ///
      /// \return
      ///     Upon success, a non-NULL pointer to the FrAdcData is
      ///     returned. Upon failure, either an exception is thrown
      //      or a NULL pointer is returned.
      //-----------------------------------------------------------------
      fr_adc_data_type ReadFrAdcStruct( frame_offset_type Frame, const std::string& Channel );
      //-----------------------------------------------------------------
      /// \brief Extract the requested FrAdcData structure from the stream
      ///
      /// \param[in] Frame
      ///     Zero based index of the frame.
      /// \param[in] Channel
      ///     Zero based index of channel being requested.
      ///
      /// \return
      ///     Upon success, a non-NULL pointer to the FrAdcData is
      ///     returned. Upon failure, either an exception is thrown
      //      or a NULL pointer is returned.
      //-----------------------------------------------------------------
      fr_adc_data_type ReadFrAdcStruct( frame_offset_type Frame, INT_4U Channel );


      //-----------------------------------------------------------------
      /// \brief Extract the requested FrEvent structure from the stream
      ///
      /// \param[in] Frame
      ///     Zero based index of the frame.
      /// \param[in] Channel
      ///     Name of the channel being requested.
      ///
      /// \return
      ///     Upon success, a non-NULL pointer to the FrEvent is
      ///     returned. Upon failure, either an exception is thrown
      //      or a NULL pointer is returned.
      //-----------------------------------------------------------------
      fr_event_type ReadFrEvent( frame_offset_type Frame, const std::string& Channel );

      //-----------------------------------------------------------------
      /// \brief Extract the requested FrEvent structure from the stream
      ///
      /// \param[in] Frame
      ///     Zero based index of the frame.
      /// \param[in] Channel
      ///     Name of the channel being requested.
      ///
      /// \return
      ///     Upon success, a non-NULL pointer to the FrEvent is
      ///     returned. Upon failure, either an exception is thrown
      //      or a NULL pointer is returned.
      //-----------------------------------------------------------------
      fr_event_type ReadFrEventStruct( frame_offset_type Frame, const std::string& Channel );

      //-----------------------------------------------------------------
      /// \brief Extract the requested FrProcData structure from the stream
      ///
      /// \param[in] Frame
      ///     Zero based index of the frame.
      /// \param[in] Channel
      ///     Name of the channel being requested.
      ///
      /// \return
      ///     Upon success, a non-NULL pointer to the FrProcData is
      ///     returned. Upon failure, either an exception is thrown
      //      or a NULL pointer is returned.
      //-----------------------------------------------------------------
      fr_proc_data_type ReadFrProcData( frame_offset_type Frame, const std::string& Channel );
      //-----------------------------------------------------------------
      /// \brief Extract the requested FrProcData structure from the stream
      ///
      /// \param[in] Frame
      ///     Zero based index of the frame.
      /// \param[in] Channel
      ///     Zero based index of channel being requested.
      ///
      /// \return
      ///     Upon success, a non-NULL pointer to the FrProcData is
      ///     returned. Upon failure, either an exception is thrown
      //      or a NULL pointer is returned.
      //-----------------------------------------------------------------
      fr_proc_data_type ReadFrProcData( frame_offset_type Frame, INT_4U Channel );
      //-----------------------------------------------------------------
      /// \brief Extract the requested FrProcData structure from the stream
      ///
      /// \param[in] Frame
      ///     Zero based index of the frame.
      /// \param[in] Channel
      ///     Name of the channel being requested.
      ///
      /// \return
      ///     Upon success, a non-NULL pointer to the FrProcData is
      ///     returned. Upon failure, either an exception is thrown
      //      or a NULL pointer is returned.
      //-----------------------------------------------------------------
      fr_proc_data_type ReadFrProcStruct( frame_offset_type Frame, const std::string& Channel );

      //-----------------------------------------------------------------
      /// \brief Extract the requested FrProcData structure from the stream
      ///
      /// \param[in] Frame
      ///     Zero based index of the frame.
      /// \param[in] Channel
      ///     Zero based index of channel being requested.
      ///
      /// \return
      ///     Upon success, a non-NULL pointer to the FrProcData is
      ///     returned. Upon failure, either an exception is thrown
      //      or a NULL pointer is returned.
      //-----------------------------------------------------------------
      fr_proc_data_type ReadFrProcStruct( frame_offset_type Frame, INT_4U Channel );

      //-----------------------------------------------------------------
      /// \brief Extract the requested FrRawData structure from the stream
      ///
      /// \param[in] Frame
      ///     Zero based index of the frame.
      ///
      /// \return
      ///     Upon success, a non-NULL pointer to the FrRawData is
      ///     returned. Upon failure, either an exception is thrown
      //      or a NULL pointer is returned.
      //-----------------------------------------------------------------
      fr_raw_data_type ReadFrRawData( frame_offset_type Frame );

      //-----------------------------------------------------------------
      /// \brief Extract the requested FrSerData structure from the stream
      ///
      /// \param[in] Frame
      ///     Zero based index of the frame.
      /// \param[in] Channel
      ///     Name of the channel being requested.
      ///
      /// \return
      ///     Upon success, a non-NULL pointer to the FrSerData is
      ///     returned. Upon failure, either an exception is thrown
      //      or a NULL pointer is returned.
      //-----------------------------------------------------------------
      fr_ser_data_type ReadFrSerData( frame_offset_type Frame, const std::string& Channel );

      //-----------------------------------------------------------------
      /// \brief Extract the requested FrSerData structure from the stream
      ///
      /// \param[in] Frame
      ///     Zero based index of the frame.
      /// \param[in] Channel
      ///     Name of the channel being requested.
      ///
      /// \return
      ///     Upon success, a non-NULL pointer to the FrSerData is
      ///     returned. Upon failure, either an exception is thrown
      //      or a NULL pointer is returned.
      //-----------------------------------------------------------------
      fr_ser_data_type ReadFrSerStruct( frame_offset_type Frame, const std::string& Channel );

      //-----------------------------------------------------------------
      /// \brief Extract the requested FrSimEvent structure from the stream
      ///
      /// \param[in] Frame
      ///     Zero based index of the frame.
      /// \param[in] Channel
      ///     Name of the channel being requested.
      ///
      /// \return
      ///     Upon success, a non-NULL pointer to the FrSimEvent is
      ///     returned. Upon failure, either an exception is thrown
      //      or a NULL pointer is returned.
      //-----------------------------------------------------------------
      fr_sim_event_type ReadFrSimEvent( frame_offset_type Frame, const std::string& Channel );

      //-----------------------------------------------------------------
      /// \brief Extract the requested FrSimEvent structure from the stream
      ///
      /// \param[in] Frame
      ///     Zero based index of the frame.
      /// \param[in] Channel
      ///     Name of the channel being requested.
      ///
      /// \return
      ///     Upon success, a non-NULL pointer to the FrSimEvent is
      ///     returned. Upon failure, either an exception is thrown
      //      or a NULL pointer is returned.
      //-----------------------------------------------------------------
      fr_sim_event_type ReadFrSimEventStruct( frame_offset_type Frame, const std::string& Channel );

      //-----------------------------------------------------------------
      /// \brief Extract the requested FrSerData structure from the stream
      ///
      /// \param[in] Frame
      ///     Zero based index of the frame.
      /// \param[in] Channel
      ///     Name of the channel being requested.
      ///
      /// \return
      ///     Upon success, a non-NULL pointer to the FrSimData is
      ///     returned. Upon failure, either an exception is thrown
      //      or a NULL pointer is returned.
      //-----------------------------------------------------------------
      fr_sim_data_type ReadFrSimStruct( frame_offset_type Frame, const std::string& Channel );
      //-----------------------------------------------------------------
      /// \brief Extract the requested FrSerData structure from the stream
      ///
      /// \param[in] Frame
      ///     Zero based index of the frame.
      /// \param[in] Channel
      ///     Name of the channel being requested.
      ///
      /// \return
      ///     Upon success, a non-NULL pointer to the FrSimData is
      ///     returned. Upon failure, either an exception is thrown
      //      or a NULL pointer is returned.
      //-----------------------------------------------------------------
      fr_sim_data_type ReadFrSimData( frame_offset_type Frame, const std::string& Channel );

      //-----------------------------------------------------------------
      /// \brief Extract the next FrameH structure from the stream
      ///
      /// \return
      ///     Upon success, a pointer to a FrameH structure.
      //-----------------------------------------------------------------
      frame_h_type ReadNextFrame( );
    };

    template< DECL >
    IFrameStreamWrapper< DECL_PARAMS >::
    IFrameStreamWrapper( buffer_type* Buffer )
      : IFrameStream( Buffer, SPEC )
    {
    }

    template< DECL >
    IFrameStreamWrapper< DECL_PARAMS >::
    IFrameStreamWrapper( bool AutoDelete, buffer_type* Buffer )
      : IFrameStream( AutoDelete, Buffer, SPEC )
    {
    }

    template< DECL >
    inline std::string IFrameStreamWrapper< DECL_PARAMS >::
    FrameLibraryName( ) const
    {
      return GetFrHeader( ).GetFrameLibraryName( );
    }

    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::frame_h_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadFrameH( frame_offset_type Frame, INT_4U ContainerSet )
    {
      typedef frame_h_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readFrameHSubset( Frame, ContainerSet ) ) );

      return retval;
    }

    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::fr_adc_data_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadFrAdcData( frame_offset_type Frame, const std::string& Channel )
    {
      typedef fr_adc_data_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readFrAdcData( Frame, Channel ) ) );

      return retval;
    }
    
    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::fr_adc_data_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadFrAdcData( frame_offset_type Frame, INT_4U Channel )
    {
      typedef fr_adc_data_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readFrAdcData( Frame, Channel ) ) );

      return retval;
    }
    
    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::fr_adc_data_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadFrAdcStruct( frame_offset_type Frame, const std::string& Channel )
    {
      typedef fr_adc_data_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readFrAdcStruct( Frame, Channel ) ) );

      return retval;
    }
    
    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::fr_adc_data_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadFrAdcStruct( frame_offset_type Frame, INT_4U Channel )
    {
      typedef fr_adc_data_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readFrAdcStruct( Frame, Channel ) ) );

      return retval;
    }
    
    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::fr_event_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadFrEvent( frame_offset_type Frame, const std::string& Channel )
    {
      typedef fr_event_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readFrEvent( Frame, Channel ) ) );

      return retval;
    }
    
    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::fr_event_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadFrEventStruct( frame_offset_type Frame, const std::string& Channel )
    {
      typedef fr_event_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readFrEventStruct( Frame, Channel ) ) );

      return retval;
    }
    
    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::fr_proc_data_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadFrProcData( frame_offset_type Frame, const std::string& Channel )
    {
      typedef fr_proc_data_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readFrProcData( Frame, Channel ) ) );

      return retval;
    }
    
    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::fr_proc_data_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadFrProcData( frame_offset_type Frame, INT_4U Channel )
    {
      typedef fr_proc_data_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readFrProcData( Frame, Channel ) ) );

      return retval;
    }
    
    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::fr_proc_data_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadFrProcStruct( frame_offset_type Frame, const std::string& Channel )
    {
      typedef fr_proc_data_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readFrProcStruct( Frame, Channel ) ) );

      return retval;
    }
    
    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::fr_proc_data_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadFrProcStruct( frame_offset_type Frame, INT_4U Channel )
    {
      typedef fr_proc_data_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readFrProcStruct( Frame, Channel ) ) );

      return retval;
    }
    
    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::fr_raw_data_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadFrRawData( frame_offset_type Frame )
    {
      typedef fr_raw_data_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readFrRawData( Frame ) ) );

      return retval;
    }

    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::fr_ser_data_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadFrSerData( frame_offset_type Frame, const std::string& Channel )
    {
      typedef fr_ser_data_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readFrSerData( Frame, Channel ) ) );

      return retval;
    }
    
    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::fr_ser_data_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadFrSerStruct( frame_offset_type Frame, const std::string& Channel )
    {
      typedef fr_ser_data_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readFrSerStruct( Frame, Channel ) ) );

      return retval;
    }
    
    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::fr_sim_data_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadFrSimData( frame_offset_type Frame, const std::string& Channel )
    {
      typedef fr_sim_data_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readFrSimData( Frame, Channel ) ) );

      return retval;
    }
    
    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::fr_sim_data_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadFrSimStruct( frame_offset_type Frame, const std::string& Channel )
    {
      typedef fr_sim_data_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readFrSimStruct( Frame, Channel ) ) );

      return retval;
    }
    
    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::fr_sim_event_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadFrSimEvent( frame_offset_type Frame, const std::string& Channel )
    {
      typedef fr_sim_event_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readFrSimEvent( Frame, Channel ) ) );

      return retval;
    }
    
    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::fr_sim_event_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadFrSimEventStruct( frame_offset_type Frame, const std::string& Channel )
    {
      typedef fr_sim_event_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readFrSimEventStruct( Frame, Channel ) ) );

      return retval;
    }
    
    template< DECL >
    typename IFrameStreamWrapper<DECL_PARAMS>::frame_h_type
    IFrameStreamWrapper< DECL_PARAMS >::
    ReadNextFrame( )
    {
      typedef frame_h_type ret_type;

      ret_type
	retval( LDASTools::AL::DynamicPointerCast< typename ret_type::element_type >
		( readNextFrame( ) ) );

      return retval;
    }
    
#undef DECL_PARAMS
#undef DECL

  } // namespace - Common
} // namespace - FrameCPP

#endif /* FRAMECPP__COMMON__FrameStreamWrapper_HH */
