#if HAVE_CONFIG_H
#include <framecpp_config.h>
#endif /* HAVE_CONFIG_H */

#include <sstream>

#include "ldastoolsal/unittest.h"
#include "ldastoolsal/fstream.hh"

#include "framecpp/Common/FrameSpec.hh"
#include "framecpp/Common/IOStream.hh"
#include "framecpp/Common/FrameBuffer.hh"
#include "framecpp/Common/FrameStream.hh"

#include "framecpp/FrameCPP.hh"

#define USING_MAKE_FRAME_VERSION	1
#define USING_VERIFY_DOWN_CONVERT	1
#define USING_VERIFY_UP_CONVERT		1

#include "FrSample.hh"

using namespace FrameCPP::Common;

#if FRAME_SPEC_MIN != 6
/// \todo Need to remove this once version 3 and 4 are supported
#undef FRAME_SPEC_MIN
#undef FRAME_SPEC_MAX

#define SMALLEST_FRAME_SPEC_MIN 9
#define SMALLEST_FRAME_SPEC_MAX 8

#define READING_FRAME_SPEC_MIN 8
#define READING_FRAME_SPEC_MAX 8

#define SKIP_CONVERT_TESTING 1
#endif

LDASTools::Testing::UnitTest	Test;

//=======================================================================
// ReadingFrame
//=======================================================================

template< class BT >
void
ReadingFrame( INT_2U Version )
{
  std::ostringstream	header;

  header << "ReadingFrame (Version " << Version << "):";

  stat_data_container_type	stat_data;
  try
  {
    //-------------------------------------------------------------------
    // Frame filename
    //-------------------------------------------------------------------
    std::ostringstream	filename;
    
    filename << "Z-ReadingFrame_ver" << Version
		<< "-" << 600000000
		<< "-" << 1
		<< ".gwf"
      ;

    
    //-------------------------------------------------------------------
    // Creation of the frame object
    //-------------------------------------------------------------------
    make_frame_ret_type
      ofo = makeFrameVersion( Version, stat_data );
    //-------------------------------------------------------------------
    // Creation of the frame file
    //-------------------------------------------------------------------
    FrameBuffer< BT >*	obuf( new FrameBuffer< BT >( std::ios::out ) );
    
    obuf->open( filename.str( ).c_str( ),
		std::ios::out | std::ios::binary );

    OFrameStream	ofs( obuf, Version );

    ofs.WriteFrame( ofo );

    ofs.Close( );
    obuf->close( );

    //-------------------------------------------------------------------
    // Creation of the frame structure by reading of frame file
    //-------------------------------------------------------------------
    FrameBuffer< BT >*	ibuf( new FrameBuffer< BT >( std::ios::in ) );
    
    ibuf->open( filename.str( ).c_str( ),
		std::ios::in | std::ios::binary );

    IFrameStream	ifs( ibuf, Version );

#if 1
    IFrameStream::object_type input_frame( ifs.ReadNextFrameAsObject( ) );
#else
    IFrameStream::object_type input_frame;
#endif

    // :TODO: ifs.Close( );
    ibuf->close( );

    //-------------------------------------------------------------------
    // Validation of the frame written to file with the frame
    //   read from the file.
    //-------------------------------------------------------------------
    if ( input_frame && ofo )
    {
      Test.Check( *ofo == *input_frame )
	<< header.str( )
	<< " frame comparison"
	<< std::endl;
    }
  }
  catch( std::exception& e )
  {
    Test.Check( false )
      << header.str( )
      << " Caught exception: " << e.what( )
      << std::endl;
  }
  catch( ... )
  {
    Test.Check( false )
      << header.str( )
      << " Caught an unknown exception"
      << std::endl;
  }
}

//=======================================================================
// Downconvert of frame data
//=======================================================================

void
DownconvertFrame( INT_2U Version )
{
  typedef FrameBuffer< LDASTools::AL::filebuf > frame_buffer_type;

  std::ostringstream	header;

  header << "DownconvertFrame (Version " << Version << "):";

  try
  {
    //-------------------------------------------------------------------
    // Frame filename
    //-------------------------------------------------------------------
    std::ostringstream	filename;
    
    filename << "Z-ReadingFrame_ver" << ( Version + 1 )
		<< "-" << 600000000
		<< "-" << 1
		<< ".gwf"
      ;
    //-------------------------------------------------------------------
    // Creation of the frame structure by reading of frame file
    //-------------------------------------------------------------------
    frame_buffer_type*	ibuf( new frame_buffer_type( std::ios::in ) );
    
    ibuf->open( filename.str( ).c_str( ),
		std::ios::in | std::ios::binary );
  
    IFrameStream	ifs( ibuf, Version );

    IFrameStream::object_type input_frame( ifs.ReadNextFrameAsObject( ) );

    // :TODO: ifs.Close( );
    ibuf->close( );
    //-------------------------------------------------------------------
    // Verify modified values
    //-------------------------------------------------------------------
    VerifyDownconvert( Version, input_frame.get( ), header.str( ), Test );

  }
  catch( std::exception& e )
  {
    Test.Check( false )
      << header.str( )
      << " Caught exception: " << e.what( )
      << std::endl;
  }
  catch( ... )
  {
    Test.Check( false )
      << header.str( )
      << " Caught an unknown exception"
      << std::endl;
  }
}

//=======================================================================
// Upconvert of frame data
//=======================================================================

void
UpconvertFrame( INT_2U Version )
{
  typedef FrameBuffer< std::filebuf > frame_buffer_type;

  std::ostringstream	header;

  header << "UpconvertFrame (Version " << Version << "):";

  try
  {
    //-------------------------------------------------------------------
    // Frame filename
    //-------------------------------------------------------------------
    std::ostringstream	filename;
    
    filename << "Z-ReadingFrame_ver" << Version
		<< "-" << 600000000
		<< "-" << 1
		<< ".gwf"
      ;
    //-------------------------------------------------------------------
    // Creation of the frame structure by reading of frame file
    //-------------------------------------------------------------------
    frame_buffer_type*	ibuf( new frame_buffer_type( std::ios::in ) );
    
    ibuf->open( filename.str( ).c_str( ),
		std::ios::in | std::ios::binary );
  
    IFrameStream	ifs( ibuf, Version + 1 );

    IFrameStream::object_type input_frame( ifs.ReadNextFrameAsObject( ) );

    //-------------------------------------------------------------------
    // Verify modified values
    //-------------------------------------------------------------------
    VerifyUpconvert( Version, input_frame.get( ), header.str( ), Test );

    // :TODO: ifs.Close( );
    // ibuf->close( );

  }
  catch( std::exception& e )
  {
    Test.Check( false )
      << header.str( )
      << " Caught exception: " << e.what( )
      << std::endl;
  }
  catch( ... )
  {
    Test.Check( false )
      << header.str( )
      << " Caught an unknown exception"
      << std::endl;
  }
}

//=======================================================================
//
//=======================================================================
template< class BT >
void
ReadNonFrameFile( const std::string& Filename )
{
  std::ostringstream	header;

  header << "ReadNonFrameFile ( Filename: " << Filename << "):";

  try
  {
    //-------------------------------------------------------------------
    // Creation of the frame structure by reading of frame file
    //-------------------------------------------------------------------
    FrameBuffer< BT >*	ibuf( new FrameBuffer< BT >( std::ios::in ) );
    
    ibuf->open( Filename.c_str( ),
		std::ios::in | std::ios::binary );

    IFrameStream	ifs( ibuf );

    // :TODO: ifs.Close( );
    ibuf->close( );
    Test.Check( false )
      << header.str( )
      << " Produced no exception"
      << std::endl
      ;
  }
  catch( std::exception& e )
  {
    static const char* const exception_text
      = "Failed to read the core of the FrHeader structure.";
    static const size_t exception_text_size = sizeof( exception_text );

    Test.Check( std::equal( exception_text, &exception_text[ exception_text_size ],
			    e.what( ) )
		)
      << header.str( )
      << " Caught exception: " << e.what( )
      << std::endl;
  }
  catch( ... )
  {
    Test.Check( false )
      << header.str( )
      << " Caught an unknown exception"
      << std::endl;
  }
}

//=======================================================================
//
//=======================================================================
template< class BT >
void
SmallestFrame( INT_2U Version )
{
  std::ostringstream	header;

  header << "SmallestFrame (Version " << Version << "):";

  try
  {
    std::ostringstream	os_filename;
    
    os_filename << "Z-SmallestFrame_ver" << Version
		<< "-" << 600000000
		<< "-" << 1
		<< ".gwf"
      ;

    FrameBuffer< BT >*	obuf( new FrameBuffer< BT >( std::ios::out ) );
    
    obuf->open( os_filename.str( ).c_str( ),
		std::ios::out | std::ios::binary );

    OFrameStream	ofs( obuf, Version );

    ofs.Close( );
    obuf->close( );
    Test.Check( true )
      << header.str( )
      << std::endl;
  }
  catch( std::exception& e )
  {
    Test.Check( false )
      << header.str( )
      << " Caught exception: " << e.what( )
      << std::endl;
  }
  catch( ... )
  {
    Test.Check( false )
      << header.str( )
      << " Caught an unknown exception"
      << std::endl;
  }
}

int
main(int argc, char* argv[])
{
  FrameCPP::Initialize( );

  //---------------------------------------------------------------------
  // Setup
  //---------------------------------------------------------------------
  Test.Init( argc, argv );

  //---------------------------------------------------------------------
  // Test reading of non-frame files
  //---------------------------------------------------------------------
  ReadNonFrameFile< std::filebuf >( "/dev/null" );
  
  //---------------------------------------------------------------------
  // Test creation of minimal frames
  //---------------------------------------------------------------------
  for( int x = SMALLEST_FRAME_SPEC_MIN;
       x <= SMALLEST_FRAME_SPEC_MAX;
       ++x )
  {
    SmallestFrame< std::filebuf >( x );
  }

  //---------------------------------------------------------------------
  // Test reading of frames
  //---------------------------------------------------------------------
  for( int x = READING_FRAME_SPEC_MIN;
       x <= READING_FRAME_SPEC_MAX;
       ++x )
  {
    ReadingFrame< std::filebuf >( x );
  }

#if ! SKIP_CONVERT_TESTING
  //---------------------------------------------------------------------
  // Test upconverting of frame structures.
  //---------------------------------------------------------------------
  for( int x = READING_FRAME_SPEC_MIN;
       x <  READING_FRAME_SPEC_MAX;
       ++x )
  {
    UpconvertFrame( x );
  }
#endif /* 0 */

#if ! SKIP_CONVERT_TESTING
  //---------------------------------------------------------------------
  // Test upconverting of frame structures.
  //---------------------------------------------------------------------
  for( int x = READING_FRAME_SPEC_MIN;
       x <  READING_FRAME_SPEC_MAX;
       ++x )
  {
    DownconvertFrame( x );
  }
#endif /* ! SKIP_CONVERT_TESTING */

  

  //---------------------------------------------------------------------
  // Report findings
  //---------------------------------------------------------------------
  Test.Exit( );
}
