#ifndef FrameCPP__COMMON__Array_HH
#define FrameCPP__COMMON__Array_HH

#if ! defined(SWIGIMPORTED)
#include <vector>
#endif /* ! defined(SWIGIMPORTED) */

#include "framecpp/Common/FrameSpec.hh"

namespace FrameCPP
{
  namespace Common
  {
    //-------------------------------------------------------------------
    /// \brief Storage class for array data streams of a frame
    ///
    /// This template manages arrays intended to be part of a
    /// frame stream.
    ///
    /// \tparam N
    ///     The frame data type used to store the array size within
    ///     the stream.
    /// \tparam T
    ///     The element type to be used for storing the individual
    ///     elements of the array.
    ///   
    //-------------------------------------------------------------------
    template< typename N, typename T >
    class Array
      : public std::vector< T >
    {
    public:
      typedef N n_size_type;
      typedef typename std::vector< T >::const_iterator	const_iterator;

      cmn_streamsize_type Bytes( const StreamBase& Stream ) const;

      cmn_streamsize_type bytes( ) const;

      bool operator==( const Array& RHS ) const;
    private:
    };

    //-------------------------------------------------------------------
    /// \brief Number of bytes needed by the frame stream for this object.
    ///
    /// This member returns the number of bytes needed to store this
    /// object to the specified stream.
    ///
    /// \param[in] Stream
    ///     The frame stream being used.
    ///
    /// \return
    ///     Number of bytes needed to store this object to the specified
    ///     stream.
    //-------------------------------------------------------------------
    template< typename N, typename T>
    cmn_streamsize_type Array< N, T >::
    Bytes( const StreamBase& Stream ) const
    {
      cmn_streamsize_type	retval = sizeof( n_size_type );

      for ( const_iterator
	      cur = this->begin( ),
	      last = this->end( );
	    cur != last;
	    ++cur )
      {
	retval += cur->Bytes( Stream );
      }
      return retval;
    }

    //-------------------------------------------------------------------
    /// \brief Number of bytes needed by the frame stream for this object.
    ///
    /// This member returns the number of bytes needed to store this
    /// object.
    ///
    /// \return
    ///     Number of bytes needed to store this object.
    //-------------------------------------------------------------------
    template< typename N, typename T>
    cmn_streamsize_type Array< N, T >::
    bytes( ) const
    {
      cmn_streamsize_type	retval = sizeof( n_size_type );

      for ( const_iterator
	      cur = this->begin( ),
	      last = this->end( );
	    cur != last;
	    ++cur )
      {
	retval += cur->bytes( );
      }
      return retval;
    }

    //-------------------------------------------------------------------
    /// \brief Compare the equality of two Array objects
    ///
    /// This member returns true if the two Array objects are
    /// equivelant.
    /// 
    /// \param[in] RHS
    ///     Object appearing on the right hand side of the
    ///     equality operator.
    ///
    /// \return
    ///     True if the two objects are equal; false otherwise
    //-------------------------------------------------------------------
    template< typename N, typename T>
    bool Array< N, T >::
    operator==( const Array& RHS ) const
    {
      bool	retval = ( &RHS == this );

      if ( retval == false )
      {
	if ( this->size( ) == RHS.size( ) )
	{
	  retval = true;
	  for ( const_iterator
		  rhs_cur = RHS.begin( ),
		  cur = this->begin( ),
		  last = this->end( );
		cur != last;
		++cur,
		++rhs_cur )
	  {
	    if ( ! ( *cur == *rhs_cur ) )
	    {
	      retval = false;
	      break;
	    }
	  }
	}
      }
      return retval;
    }
  } // namespace - Common
} // namespace - FrameCPP
#endif /* FrameCPP__COMMON__Array_HH */
