#include <sys/stat.h>
#include <unistd.h>

#include "ldastoolsal/unittest.h"
#include "ldastoolsal/DeadLockDetector.hh"

#include "genericAPI/LDASplatform.hh"
#include "genericAPI/Logging.hh"
#include "genericAPI/LogText.hh"

#include "Catalog.hh"
#include "Channel.hh"
#include "createRDS.hh"
#include "Frame.hh"

LDASTools::Testing::UnitTest	Test;

typedef std::list< std::string > base_filename_container_type;
typedef std::list< std::string > filename_container_type;

class Search
{
public:
  Search( );

  void AddPath( const std::string& Path );
  void AddPathVar( const std::string& Var, const std::string& SubDir );

  void Find( const base_filename_container_type& BaseNames,
	     filename_container_type& Files ) const;

private:
  typedef std::list< std::string >	path_container_type;

  path_container_type			paths;
};

class TestSet
{
public:
  typedef INT_4U start_time_type;
  typedef INT_4U dt_type;

  typedef enum {
    SET_S6_LHO,
    SET_S6_LLO,
    SET_S5
  } set_type;

  TestSet( );

  void AddSet( set_type Set );

  void AddSet( const base_filename_container_type& Filename, start_time_type Start, dt_type Dt );

  void Find( const Search& SearchDirs,
	     ::RDSFrame::frame_file_container_type& FrameFiles );

  void Find( const Search& SearchDirs,
	     FrameAPI::Catalog::stream_source_type& FrameFiles );

  void operator()( const std::string& Filename, start_time_type Start, dt_type Dt );

private:
  struct file_set_type {
    base_filename_container_type	files;
    start_time_type			start;
    dt_type				dt;
  };
  typedef std::list< file_set_type > file_set_container_type;

  file_set_container_type		file_set;

  void mkset( start_time_type Start,
	      dt_type Dt,
	      int Count,
	      const std::string& Path,
	      const std::string& SiteType );
  
};

Search	SEARCH_DIRS;

void Initialize( );
void TestCreateRDS( );
void TestCreateRDSResample( );
void TestExceptionMissingChannel( );
void TestOpenData( );

int
main( int ArgC, char** ArgV )
{
#if DEAD_LOCK_DETECTOR_ENABLED
  {
    LDASTools::AL::DeadLockDetector::SetDebugging( 2003 );
  }
#endif /* DEAD_LOCK_DETECTOR_ENABLED */

  //---------------------------------------------------------------------
  // Initialize the testing framework
  //---------------------------------------------------------------------
  Test.Init( ArgC, ArgV );

  try
  {
    Initialize( );

    TestCreateRDS( );
    TestCreateRDSResample( );
    TestOpenData( );

    TestExceptionMissingChannel( );
  }
  catch( const std::exception& E )
  {
    Test.Check( false ) << "Caught an exception: " << E.what( ) << std::endl;
  }
  catch( ... )
  {
    Test.Check( false ) << "Caught an unknown exception" << std::endl;
  }

  //---------------------------------------------------------------------
  // Return the state of testing
  //---------------------------------------------------------------------
  Test.Exit();
}

void
Initialize( )
{
  GenericAPI::LoggingInfo::LogDirectory( "-" );
  GenericAPI::SetLogFormatter( new GenericAPI::Log::Text( "" ) );
  GenericAPI::LDASplatform::AppName( "test_createRDS_cpp" );

  if ( Test.Verbosity( ) > 0 )
  {
    GenericAPI::setLogDebugLevel( Test.Verbosity( ) );
  }
  FrameCPP::Initialize( );
  SEARCH_DIRS.AddPathVar( "LDAS_TEST_FRAMES_DIR", "" );
  SEARCH_DIRS.AddPathVar( "HOME", "tmp" );
}

void
TestCreateRDS( )
{
  static const char* func_name = "TestCreateRDS";

  ::RDSFrame::frame_file_container_type	frame_files;

  TestSet	file_set;

  file_set( "H-SIDv1_H1H2_250mHz-821935753-26.gwf", 821935753, 26 );
  file_set.Find( SEARCH_DIRS, frame_files );

  if ( frame_files.size( ) <= 0 )
  {
    //-------------------------------------------------------------------
    // No files to process
    //-------------------------------------------------------------------
    return;
  }
  Test.Message( 10 ) << func_name << ": Start"
		     << std::endl
    ;
  try
  {
    FrameAPI::Catalog::gps_seconds_type		start( 821935753);
    FrameAPI::Catalog::gps_seconds_type		stop( start + 16 );
    FrameAPI::Catalog::channel_container_type	channels;

    //-------------------------------------------------------------------
    // Attempt to read the file
    //-------------------------------------------------------------------
    FrameAPI::Frame			frame;
    FrameAPI::Frame::channel_type	as_q;

    channels.names.push_back( "H1:AdjacentPSD" );
    Test.Message( 10 ) << func_name << ": About to create frame"
		       << std::endl
      ;
    frame = FrameAPI::createRDSFrame( frame_files, start, stop, channels );
    Test.Message( 10 ) << func_name << ": Created frame"
		       << std::endl
      ;
    as_q = frame.GetChannel( "H1:AdjacentPSD" );
    Test.Message( 10 ) << func_name << ": as_q.name: " << as_q->GetName( ) << std::endl;
    Test.Message( 10 ) << func_name << ": as_q.unitY: " << as_q->GetUnitY( ) << std::endl;
    Test.Message( 10 ) << func_name << ": as_q.data[0].name: "
		       << as_q->RefData( )[ 0 ]->GetName( ) << std::endl;
  }
  catch( const std::exception& E )
  {
    Test.Message( 10 ) << func_name << ": Caught an exception: " << E.what( )
		       << std::endl
      ;
    Test.Check( false ) << func_name << ": Caught an exception: " << E.what( )
			<< std::endl
      ;
  }
  catch( ... )
  {
    Test.Message( 10 ) << func_name << ": Caught an unhandled exception"
		       << std::endl
      ;
    Test.Check( false ) << func_name << ": Caught an unhandled exception"
			<< std::endl
      ;
  }
  Test.Message( 10 ) << func_name << ": Stop"
		     << std::endl
    ;
}

void
TestCreateRDSResample( )
{
  static const char* func_name = "TestCreateRDSResample";

  ::RDSFrame::frame_file_container_type	frame_files;

  TestSet	file_set;

  file_set.AddSet( TestSet::SET_S6_LLO );
  file_set.Find( SEARCH_DIRS, frame_files );

  if ( frame_files.size( ) <= 0 )
  {
    //-------------------------------------------------------------------
    // No files to process
    //-------------------------------------------------------------------
    Test.Message( 30 ) << func_name << "Skipped because no files to process"
		       << std::endl
      ;
    return;
  }
  Test.Message( 10 ) << func_name << ": Start"
		     << std::endl
    ;
  try
  {
    static char					cwd[ 1024 ];

    FrameAPI::RDS::Options::start_type		start( 971603040 );
    FrameAPI::RDS::Options::end_type		stop( start + 96 );
    FrameAPI::Catalog::channel_container_type	channels;
    FrameAPI::RDS::FileOptions		      	options;

    //-------------------------------------------------------------------
    // Setup options
    //-------------------------------------------------------------------

    if ( getcwd( cwd, sizeof( cwd ) ) == NULL )
    {
      throw std::runtime_error( "Unable to get current working directory" );
    }

    options.OutputTimeStart( start );
    options.OutputTimeEnd( stop );
    options.DirectoryOutputFrames( cwd );
    options.DirectoryOutputMD5Sum( cwd );

    //-------------------------------------------------------------------
    // Attempt to create a set of RDS frames from the input set
    //-------------------------------------------------------------------
    channels.names.push_back( "L1:LSC-AS_Q" );
    FrameAPI::createRDSSet( frame_files, channels, options );
  }
  catch( const SwigException& E )
  {
    Test.Check( false ) << func_name << ": Caught a SWIG exception: results:" << E.getResult( )
			<< " info: "<<  E.getInfo( )
			<< std::endl
      ;
  }
  catch( const std::exception& E )
  {
    Test.Check( false ) << func_name << ": Caught an exception: " << E.what( )
			<< std::endl
      ;
  }
  catch( ... )
  {
    Test.Check( false ) << func_name << ": Caught an unknown exception"
			<< std::endl
      ;
  }
  Test.Message( 10 ) << func_name << ": Stop"
		     << std::endl
    ;
}

void
TestExceptionMissingChannel( )
{
  static const char* func_name = "TestCreateRDS";

  ::RDSFrame::frame_file_container_type	frame_files;

  TestSet	file_set;

  file_set( "H-SIDv1_H1H2_250mHz-821935753-26.gwf", 821935753, 26 );
  file_set.Find( SEARCH_DIRS, frame_files );

  if ( frame_files.size( ) <= 0 )
  {
    //-------------------------------------------------------------------
    // No files to process
    //-------------------------------------------------------------------
    return;
  }
  Test.Message( 10 ) << func_name << ": Start"
		     << std::endl
    ;
  try
  {
    FrameAPI::Catalog::gps_seconds_type		start( 821935753);
    FrameAPI::Catalog::gps_seconds_type		stop( start + 16 );
    FrameAPI::Catalog::channel_container_type	channels;

    //-------------------------------------------------------------------
    // Attempt to read the file
    //-------------------------------------------------------------------
    FrameAPI::Frame			frame;
    FrameAPI::Frame::channel_type	as_q;

    channels.names.push_back( "H1:AdjacentPSD" );
    channels.names.push_back( "Missing Channel" );
    Test.Message( 10 ) << func_name << ": About to create frame"
		       << std::endl
      ;
    frame = FrameAPI::createRDSFrame( frame_files, start, stop, channels );
    Test.Message( 10 ) << func_name << ": Created frame"
		       << std::endl
      ;
    as_q = frame.GetChannel( "H1:AdjacentPSD" );
    Test.Message( 10 ) << func_name << ": as_q.name: " << as_q->GetName( ) << std::endl;
    Test.Message( 10 ) << func_name << ": as_q.unitY: " << as_q->GetUnitY( ) << std::endl;
    Test.Message( 10 ) << func_name << ": as_q.data[0].name: "
		       << as_q->RefData( )[ 0 ]->GetName( ) << std::endl;
  }
  catch( const FrameAPI::RDS::MissingChannel& E )
  {
    Test.Check( true ) << func_name << ": Caught expected exception: " << E.what( )
		       << std::endl
      ;
  }
  catch( const std::exception& E )
  {
    Test.Message( 10 ) << func_name << ": Caught an exception: " << E.what( )
		       << std::endl
      ;
    Test.Check( false ) << func_name << ": Caught an exception: " << E.what( )
			<< std::endl
      ;
  }
  catch( ... )
  {
    Test.Message( 10 ) << func_name << ": Caught an unhandled exception"
		       << std::endl
      ;
    Test.Check( false ) << func_name << ": Caught an unhandled exception"
			<< std::endl
      ;
  }
  Test.Message( 10 ) << func_name << ": Stop"
		     << std::endl
    ;
}

void
TestOpenData( )
{
  static const char* func_name = "TestOpenData";
  FrameAPI::Catalog::stream_source_type	frame_files;

  TestSet	file_set;

  file_set( "H-SIDv1_H1H2_250mHz-821935753-26.gwf", 821935753, 26 );
  file_set.Find( SEARCH_DIRS, frame_files );

  if ( frame_files.size( ) <= 0 )
  {
    //-------------------------------------------------------------------
    // No files to process
    //-------------------------------------------------------------------
    return;
  }
  try
  {
    FrameAPI::Catalog::gps_seconds_type		start( 821935753);
    FrameAPI::Catalog::gps_seconds_type		stop( start + 16 );
    FrameAPI::Catalog::channel_container_type	channels;

    //-------------------------------------------------------------------
    // Open, scan the TOC, and close a set of files.
    // This gets the meta data located in the TOC and makes it available
    //   to the caller via the catalog interface
    //-------------------------------------------------------------------
    FrameAPI::Catalog catalog( frame_files );
    //-------------------------------------------------------------------
    // Display the data range
    //-------------------------------------------------------------------
    Test.Message( 10 ) << func_name << ": start time: " << catalog.GPSStartTime( ) << std::endl;
    Test.Message( 10 ) << func_name << ": end time: " << catalog.GPSEndTime( ) << std::endl;
    //-------------------------------------------------------------------
    // Obtain the catalog of channels
    //-------------------------------------------------------------------
    FrameAPI::Catalog::channel_dict_type
      channelDict = catalog.Channels( );
    for ( FrameAPI::Catalog::channel_dict_type::element_type::const_iterator
	    cur = channelDict->begin( ),
	    last = channelDict->end( );
	  cur != last;
	  ++cur )
    {
      Test.Message( 10 ) << func_name << ": Channel: " << cur->first
			 << std::endl;
    }
    //------------------------------------------------------------------
    // Read a collection of channels
    //------------------------------------------------------------------
    // channel_names.push_back( "H1H2:CSD" ); // not available
    channels.names.push_back( "H1:AdjacentPSD" );
    channels.resampling.push_back( 1 );
    FrameAPI::Catalog::data_dict_type
      dataDict( catalog.Fetch( start, stop, channels ) );
    //------------------------------------------------------------------
    // Display some meta data about the channels
    //------------------------------------------------------------------
    FrameAPI::Catalog::DataDictionary::channel_type
      channel( (*dataDict)[ channels.names[ 0 ] ] );
    Test.Message( 10 ) << func_name << ": Channel: unitY: " << channel->GetUnitY( )
		       << std::endl
      ;
  }
  catch( const std::exception& E )
  {
    Test.Check( false ) << func_name << ": Caught an exception: " << E.what( ) << std::endl;
  }
}

TestSet::
TestSet( )
{
}

void TestSet::
AddSet( set_type Set )
{
  std::list< std::string >	files;

  switch( Set )
  {
  case SET_S6_LHO:
    mkset( 971621600, 32, 10,
	   "S6/L0/LHO/H-R-9716/",
	   "H-R-" );
    break;
  case SET_S6_LLO:
    mkset( 971603008, 32, 10,
	   "S6/L0/LLO/L-R-9716/",
	   "L-R-" );
    break;
  case SET_S5:
    break;
  }
}

void TestSet::
AddSet( const base_filename_container_type& Filenames, start_time_type Start, dt_type Dt )
{
  file_set_type	f;

  f.files = Filenames;
  f.start = Start;
  f.dt = Dt;

  file_set.push_back( f );
}

void TestSet::
Find( const Search& SearchDirs,
      ::RDSFrame::frame_file_container_type& FrameFiles )
{
  FrameFiles.clear( );
  for( file_set_container_type::const_iterator
	 cur = file_set.begin( ),
	 last = file_set.end( );
       cur != last;
       ++cur )
  {
    std::list< std::string > files;

    SearchDirs.Find( cur->files, files );
    if ( files.size( ) > 0 )
    {
      FrameFiles.resize( files.size( ) );
      std::copy( files.begin( ), files.end( ), FrameFiles.begin( ) );
      break;
    }
  }
}

void TestSet::
Find( const Search& SearchDirs,
      FrameAPI::Catalog::stream_source_type& FrameFiles )
{
  FrameFiles.clear( );
  for( file_set_container_type::const_iterator
	 cur = file_set.begin( ),
	 last = file_set.end( );
       cur != last;
       ++cur )
  {
    std::list< std::string > files;

    SearchDirs.Find( cur->files, files );
    if ( files.size( ) > 0 )
    {
      FrameFiles.swap( files );
      break;
    }
  }
}

void TestSet::
operator()( const std::string& Filename, start_time_type Start, dt_type Dt )
{
  file_set_type	f; 

  f.files.push_back( Filename );
  f.start = Start;
  f.dt = Dt;

  file_set.push_back( f );
}

inline void TestSet::
mkset( start_time_type Start,
       dt_type Dt,
       int Count,
       const std::string& Path,
       const std::string& SiteType )
{
  base_filename_container_type	files;

  for( int
	 cur = 0,
	 last = Count;
       cur != last;
       ++cur, Start += Dt )
  {
    std::ostringstream	filename;
    filename << Path
	     << SiteType << Start << "-" << Dt << ".gwf";
    files.push_back( filename.str( ) );
  }

  AddSet( files, Start, Dt );
}

Search::
Search( )
{
}

void Search::
AddPath( const std::string& Path )
{
  struct stat	st;

  if ( ( stat( Path.c_str( ), &st ) == 0 )
       && ( S_ISDIR( st.st_mode ) ) )
  {
    paths.push_back( Path );
  }
}

void Search::
AddPathVar( const std::string& Var, const std::string& SubDir )
{
  const char* v = ::getenv( Var.c_str( ) );
  if ( v )
  {
    std::string	path( v );

    if ( SubDir.size( ) > 0)
    {
      path += "/";
      path += SubDir;
    }
    AddPath( path );
  }
}

void Search::
Find( const base_filename_container_type& BaseNames,
      std::list< std::string >& Files ) const
{
  std::list< std::string >	results;

  for ( std::list< std::string >::const_iterator
	  cur_path = paths.begin( ),
	  last_path = paths.end( );
	cur_path != last_path;
	++cur_path )
  {
    for ( std::list< std::string >::const_iterator
	    cur = BaseNames.begin( ),
	    last = BaseNames.end( );
	  cur != last;
	  ++cur )
    {
      std::string	filename( *cur_path );

      filename += "/";
      filename += *cur;

      std::ifstream	ifile( filename.c_str( ) );
      if ( ifile )
      {
	results.push_back( filename );
      }
    }
    if ( results.size( ) == BaseNames.size( ) )
    {
      Files.swap( results );
      return;
    }
    results.clear( );
  }
}
