#' Graph Parameter Estimator
#'
#' \code{graph.param.estimator} estimates the parameter that best approximates
#' the model to the observed graph according to the Graph Information Criterion
#' (GIC).
#'
#' @param Graph the undirected graph (igraph object).
#' If \code{Graph} has the  attribute 'eigenvalues' containing
#' the eigenvalues of \code{Graph}, such values will be used to
#' compute spectral density of the graph.
#'
#' @param model either a string or a function:
#'
#' A string that indicates one of the following models: 'ER' (Erdos-Renyi random
#' graph), 'GRG' (geometric random graph), 'KR' (k regular random graph), 'WS'
#' (Watts-Strogatz model), and 'BA' (Barabási-Albert model).
#'
#' A function that returns a graph (represented by its adjacency matrix)
#' generated by a graph model. It must contain two arguments: the first one
#' corresponds to the graph size and the second to the parameter of the model.
#'
#' @param interval numeric vector containing the values that will be
#' considered for the parameter estimation, or a list containing 'lo' and 'hi'
#' that indicates the model's parameter search interval <\code{lo},\code{hi}>.
#' The \code{graph.param.estimator} will return the element of 'parameter' that
#' minimizes the GIC.
#' If the user does not provide the argument \code{parameters}, and \code{model} is a string,
#' then default values are used for the predefined models ('ER', 'GRG', 'KR', 'WS',
#' and 'BA'). The default \code{parameter} argument corresponds to a sequence from
#'
#' 0 to 1 with step \code{eps} for the 'ER' model (Erdos-Renyi random graph), in
#' which the parameter corresponds to the probability to connect a pair of
#' vertices;
#'
#' 0 to sqrt(2) with step \code{eps} for the 'GRG' model (geometric random graph), in
#' which the parameter corresponds to the radius used to construct the geometric
#' graph in a unit square;
#'
#' 0 to 'n' with step \code{n*eps} for the 'KR' model (k regular random graph), in
#' which the parameter of the model corresponds to the degree \code{k} of a regular
#' graph;
#'
#' 0 to 1 with step \code{eps} for the 'WS' model (Watts-Strogatz model), in which
#' the parameter corresponds to the probability to reconnect a vertex;
#'
#' and 0 to 3 with step \code{eps} for the 'BA' model (Barabási-Albert model), in
#' which the parameter corresponds to the scaling exponent.
#'
#' @param eps precision of the grid and ternary search (default is \code{0.01}).
#'
#' @param search string that indicates the search algorithm to find
#' the parameter with the smallest GIC. If 'grid' (default) parameter is
#' estimated using grid search, and only works when method is not 'fast'.
#' If 'ternary' parameter is estimated using ternary search.
#'
#' @param ... Other relevant parameters for \code{\link{GIC}}.
#'
#' @return A list with class 'statGraph' containing the following components:
#' \item{\code{method:}}{ a string indicating the used method.}
#' \item{\code{info:}}{ a string showing details about the method.}
#' \item{\code{data.name:}}{ a string with the data's name(s).}
#' \item{\code{param:}}{ the parameter estimate. For the 'ER', 'GRG', 'KR', 'WS', and 'BA'
#' models, the parameter corresponds to the probability to connect a pair of
#' vertices, the radius used to construct the geometric graph in a unit square,
#' the degree \code{k} of a regular graph, the probability to reconnect a vertex, and
#' the scaling exponent, respectively.}
#' \item{\code{dist:}}{ the distance between the observed graph and the graph model with the estimated
#' parameter.}
#'
#' @keywords parameter_estimation
#'
#' @references
#' Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012)
#' Discriminating Different Classes of Biological Networks by Analyzing the
#' Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949.
#' doi:10.1371/journal.pone.0049949.
#'
#' Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.
#'
#' Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._,
#' *21*, 65-66.
#'
#' Sheather, S. J. and Jones, M. C. (1991). A reliable data-based bandwidth
#' selection method for kernel density estimation.
#' _Journal of the Royal Statistical Society series B_, 53, 683-690.
#' http://www.jstor.org/stable/2345597.
#'
#' @examples
#' set.seed(1)
#' G <- igraph::sample_gnp(n=50, p=0.5)
#'
#' # Using a string to indicate the graph model
#' result1 <- graph.param.estimator(G, 'ER', eps=0.25)
#' result1
#'
#'
#' # Using a function to describe the graph model
#' # Erdos-Renyi graph
#' set.seed(1)
#' model <- function(n, p) {
#'   return(igraph::sample_gnp(n, p))
#' }
#' result2 <- graph.param.estimator(G, model,  seq(0.2, 0.8, 0.1))
#' result2
#'
#'
#'
#' @export
graph.param.estimator <- function(Graph, model, interval = NULL, eps = 0.01, search = "grid", ...) {
    # interval is a list that contains lo, hi
    if (!valid.input(Graph))
        stop("The input should be an igraph object!")

    data.name <- deparse(substitute(Graph))
    params <- list(...)
    if (search == "ternary") {
        condition <- FALSE
        if ((length(params) == 0))
            condition <- TRUE else if (is.null(params$dist)) {
            condition <- TRUE
        } else if (params$dist == "KL") {
            condition <- TRUE
        }
        if (condition) {
            warning("Ternary search does not guarantee to obtain the parameter with the smallest 'KL' divergence.",
                immediate. = TRUE)
        }
    }
    #
    Graph$density <- graph.spectral.density(Graph, ...)
    n <- igraph::vcount(Graph)
    m <- igraph::ecount(Graph)
    # returns a vector or a list of intervals
    search_interval <- get.model.interval(n = n, m = m, model = model, parameter = interval, eps = eps, search = search)
    # search the parameter that minimizes the norm/divergence
    search_output <- graph.model.param.search(Graph = Graph, model = model, search_interval = search_interval,
        eps = eps, ...)
    ###
    method <- "Graph Parameter Estimator"
    info <- Graph$density$info
    output <- list(method = method, info = info, data.name = data.name, param = search_output$param, dist = search_output$dist)
    class(output) <- "statGraph"
    return(output)
}

# function for parameter searching

graph.model.param.search <- function(Graph, model, search_interval, eps, ...) {
    if (methods::is(search_interval, "atomicVector")) {
        # apply grid search
        return(graph.model.grid.param.search(Graph, model, search_interval, ...))
    } else {
        # apply ternary search
        return(graph.model.ternary.param.search(Graph, model, search_interval, eps, ...))
    }
}

# grid search
graph.model.grid.param.search <- function(Graph, model, search_interval, ...) {
    pmin <- -1
    min_GIC <- Inf
    for (p in search_interval) {
        curr_GIC <- GIC(Graph = Graph, model = model, p = p, ...)$value
        if (curr_GIC < min_GIC) {
            min_GIC <- curr_GIC
            pmin <- p
        }
    }

    out <- list(param = pmin, dist = min_GIC)

    return(out)
}

# Ternary search
graph.model.ternary.param.search <- function(Graph, model, search_interval, eps, ...) {
    pmin_res = NA
    min_GIC = Inf
    for (interval in search_interval) {
        lo = interval$lo
        hi = interval$hi
        curr_GIC = NA
        while (TRUE) {
            if (abs(lo - hi) < eps || lo > hi) {
                pmin <- (lo + hi)/2
                if (model == "KR")
                  pmin = as.integer(round(pmin))
                curr_GIC = GIC(Graph = Graph, model = model, p = pmin, ...)$value
                break
            }

            leftThird <- (2 * lo + hi)/3
            rightThird <- (lo + 2 * hi)/3

            if (model == "KR") {
                leftThird = as.integer(round(leftThird))
                rightThird = as.integer(round(rightThird))
            }
            GIC_1 <- GIC(Graph = Graph, model, p = leftThird, ...)$value
            GIC_2 <- GIC(Graph = Graph, model, p = rightThird, ...)$value
            if (GIC_1 <= GIC_2)
                hi <- rightThird else lo <- leftThird
        }
        if (curr_GIC < min_GIC) {
            pmin_res = pmin
            min_GIC = curr_GIC
        }
    }
    out <- list(param = pmin_res, dist = min_GIC)
}
