% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tidy-params.R
\name{tidy-params}
\alias{tidy-params}
\alias{param_range}
\alias{param_glue}
\title{Tidy parameter selection}
\usage{
param_range(prefix, range, vars = NULL)

param_glue(pattern, ..., vars = NULL)
}
\arguments{
\item{prefix, range}{For \code{param_range()} only, \code{prefix} is a string naming a
parameter and \code{range} is an integer vector providing the indices of a
subset of elements to select. For example, using

\if{html}{\out{<div class="sourceCode">}}\preformatted{  param_range("beta", c(1,2,8))
}\if{html}{\out{</div>}}

would select parameters named \code{beta[1]}, \code{beta[2]}, and \code{beta[8]}.
\code{param_range()} is only designed for the case that the indices are integers
surrounded by brackets. If there are no brackets use
\link[tidyselect:language]{num_range()}.}

\item{vars}{\code{NULL} or a character vector of parameter names to choose from.
This is only needed for the atypical use case of calling the function as a
standalone function outside of \code{vars()}, \code{select()}, etc. Typically this is
left as \code{NULL} and will be set automatically for the user.}

\item{pattern, ...}{For \code{param_glue()} only, \code{pattern} is a string containing
expressions enclosed in braces and \code{...} should be named arguments
providing one character vector per expression in braces in \code{pattern}. It is
easiest to describe how to use these arguments with an example:

\if{html}{\out{<div class="sourceCode">}}\preformatted{param_glue("beta_\{var\}[\{level\}]",
           var = c("age", "income"),
           level = c(3,8))
}\if{html}{\out{</div>}}

would select parameters with names
\code{"beta_age[3]"}, \code{"beta_income[3]"}, \code{"beta_age[8]"}, \code{"beta_income[8]"}.}
}
\description{
Parameter selection in the style of \strong{dplyr} and other tidyverse packages.
}
\details{
As of version \verb{1.7.0}, \strong{bayesplot} allows the \code{pars} argument for \link[=MCMC-overview]{MCMC plots} to use "tidy" variable selection (in the
style of the \strong{dplyr} package). The \code{\link[dplyr:vars]{vars()}} function is
re-exported from \strong{dplyr} for this purpose.

Features of tidy selection includes direct selection (\code{vars(alpha, sigma)}),
everything-but selection (\code{vars(-alpha)}), ranged selection
(\code{vars(`beta[1]`:`beta[3]`)}), support for selection functions
(\code{vars(starts_with("beta"))}), and combinations of these features. See the
\strong{Examples} section, below.

When using \code{pars} for tidy parameter selection, the \code{regex_pars} argument is
ignored because \strong{bayesplot} supports using \link[tidyselect:language]{tidyselect helper functions} (\code{starts_with()}, \code{contains()},
\code{num_range()}, etc.) for the same purpose. \strong{bayesplot} also exports some
additional helper functions to help with parameter selection:
\itemize{
\item \code{param_range()}: like \code{\link[tidyselect:starts_with]{num_range()}} but used
when parameter indexes are in brackets (e.g. \code{beta[2]}).
\item \code{param_glue()}: for more complicated parameter names with multiple
indexes (including variable names) inside the brackets
(e.g., \verb{beta[(Intercept) age_group:3]}).
}

These functions can be used inside of \code{vars()}, \code{dplyr::select()},
and similar functions, just like the
\link[tidyselect:language]{tidyselect helper functions}.
}
\section{Extra Advice}{


Parameter names in \code{vars()} are not quoted. When the names contain special
characters like brackets, they should be wrapped in backticks, as in
\code{vars(`beta[1]`)}.

To exclude a range of variables, wrap the sequence in parentheses and then
negate it. For example, (\code{vars(-(`beta[1]`:`beta[3]`))}) would exclude
\code{beta[1]}, \code{beta[2]}, and \code{beta[3]}.

\code{vars()} is a helper function. It holds onto the names and expressions used
to select columns. When selecting variables inside a \strong{bayesplot}
function, use \code{vars(...)}: \code{mcmc_hist(data, pars = vars(alpha))}. When
using \code{select()} to prepare a dataframe for a \strong{bayesplot} function, do
not use \code{vars()}: \code{data \%>\% select(alpha) \%>\% mcmc_hist()}.

Internally, tidy selection works by converting names and expressions
into position numbers. As a result, integers will select parameters;
\code{vars(1, 3)} selects the first and third ones. We do not endorse this
approach because positions might change as variables are added and
removed from models. To select a parameter that happens to be called \code{1},
use backticks to escape it \code{vars(`1`)}.
}

\examples{
x <- example_mcmc_draws(params = 6)
dimnames(x)
mcmc_hex(x, pars = vars(alpha, `beta[2]`))
mcmc_dens(x, pars = vars(sigma, contains("beta")))
mcmc_hist(x, pars = vars(-contains("beta")))

# using the param_range() helper
mcmc_hist(x, pars = vars(param_range("beta", c(1, 3, 4))))

\donttest{
#############################
## Examples using rstanarm ##
#############################
if (requireNamespace("rstanarm", quietly = TRUE)) {
  # see ?rstanarm::example_model
  fit <- example("example_model", package = "rstanarm", local=TRUE)$value
  print(fit)
  posterior <- as.data.frame(fit)
  str(posterior)

  color_scheme_set("brightblue")
  mcmc_hist(posterior, pars = vars(size, contains("period")))

  # same as previous but using dplyr::select() and piping
  library("dplyr")
  posterior \%>\%
    select(size, contains("period")) \%>\%
    mcmc_hist()

  mcmc_intervals(posterior, pars = vars(contains("herd")))
  mcmc_intervals(posterior, pars = vars(contains("herd"), -contains("Sigma")))

  bayesplot_theme_set(ggplot2::theme_dark())
  color_scheme_set("viridisC")
  mcmc_areas_ridges(posterior, pars = vars(starts_with("b[")))

  bayesplot_theme_set()
  color_scheme_set("purple")
  not_789 <- vars(starts_with("b["), -matches("[7-9]"))
  mcmc_intervals(posterior, pars = not_789)

  # using the param_glue() helper
  just_149 <- vars(param_glue("b[(Intercept) herd:{level}]", level = c(1,4,9)))
  mcmc_intervals(posterior, pars = just_149)

  # same but using param_glue() with dplyr::select()
  # before passing to bayesplot
  posterior \%>\%
    select(param_glue("b[(Intercept) herd:{level}]",
                      level = c(1, 4, 9))) \%>\%
    mcmc_intervals()
}
}
\dontrun{
###################################
## More examples of param_glue() ##
###################################
library(dplyr)
posterior <- tibble(
  b_Intercept = rnorm(1000),
  sd_condition__Intercept = rexp(1000),
  sigma = rexp(1000),
  `r_condition[A,Intercept]` = rnorm(1000),
  `r_condition[B,Intercept]` = rnorm(1000),
  `r_condition[C,Intercept]` = rnorm(1000),
  `r_condition[A,Slope]` = rnorm(1000),
  `r_condition[B,Slope]` = rnorm(1000)
)
posterior

# using one expression in braces
posterior \%>\%
  select(
    param_glue("r_condition[{level},Intercept]", level = c("A", "B"))
  ) \%>\%
  mcmc_hist()

# using multiple expressions in braces
posterior \%>\%
   select(
     param_glue(
       "r_condition[{level},{type}]",
        level = c("A", "B"),
        type = c("Intercept", "Slope"))
   ) \%>\%
   mcmc_hist()
}
}
\seealso{
\code{\link[glue:glue]{glue::glue()}}
}
