/*
    ApplicationGeneralAttributes.m

    Implementation of the ApplicationGeneralAttributes class for the
    ProjectManager application.

    Copyright (C) 2005, 2006  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "ApplicationGeneralAttributes.h"

#import <Foundation/NSString.h>

#import <AppKit/NSTextField.h>
#import <AppKit/NSNibLoading.h>
#import <AppKit/NSTableView.h>

#import "../../CommonFunctions/StringValidation.h"
#import "ApplicationProjectType.h"

@implementation ApplicationGeneralAttributes

- (void) dealloc
{
  TEST_RELEASE(view);

  [super dealloc];
}

- initWithOwner: (ApplicationProjectType *) anOwner
{
  if ([self init])
    {
      owner = anOwner;

      return self;
    }
  else
    {
      return nil;
    }
}

- (void) awakeFromNib
{
  [view retain];
  [view removeFromSuperview];
  DESTROY(bogusWindow);

  [appName setStringValue: [owner applicationName]];
  [copyright setStringValue: [owner copyright]];
  [copyrightDescription setStringValue: [owner copyrightDescription]];
  [description setStringValue: [owner applicationDescription]];
  [iconFilename setStringValue: [owner iconName]];
  [mainNibFile setStringValue: [owner mainNibFile]];
  [principalClass setStringValue: [owner principalClass]];
  [release setStringValue: [owner applicationRelease]];
}


- (void) addAuthor: (id)sender
{
  NSMutableArray * newAuthors = [[[owner authors] mutableCopy] autorelease];

  [newAuthors addObject: @""];
  [owner setAuthors: newAuthors];
  [authors reloadData];
}

- (void) moveAuthorUp: (id)sender
{
  int row = [authors selectedRow];

  if (row > 0)
    {
      NSMutableArray * newAuthors = [[[owner authors] mutableCopy]
        autorelease];
      NSString * originalAuthor;

      originalAuthor = [[newAuthors objectAtIndex: row - 1] retain];
      [newAuthors replaceObjectAtIndex: row - 1
                            withObject: [newAuthors objectAtIndex: row]];
      [newAuthors replaceObjectAtIndex: row
                            withObject: originalAuthor];
      [owner setAuthors: newAuthors];

      [authors reloadData];
      [authors selectRow: row - 1 byExtendingSelection: NO];
    }
}


- (void) moveAuthorDown: (id)sender
{
  int row = [authors selectedRow];
  NSArray * authorsArray = [owner authors];

  if (row >= 0 && row + 1 < (int) [authorsArray count])
    {
      NSMutableArray * newAuthors = [[authorsArray mutableCopy] autorelease];
      NSString * originalAuthor;

      originalAuthor = [[newAuthors objectAtIndex: row + 1] retain];
      [newAuthors replaceObjectAtIndex: row + 1
                            withObject: [newAuthors objectAtIndex: row]];
      [newAuthors replaceObjectAtIndex: row
                            withObject: originalAuthor];
      [owner setAuthors: newAuthors];

      [authors reloadData];
      [authors selectRow: row + 1 byExtendingSelection: NO];
    }
}


- (void) removeAuthor: (id)sender
{
  int row = [authors selectedRow];

  if (row >= 0)
    {
      NSMutableArray * newAuthors = [[[owner authors] mutableCopy]
        autorelease];

      [newAuthors removeObjectAtIndex: row];
      [owner setAuthors: newAuthors];

      [authors reloadData];
    }
}

- (NSView *) view
{
  if (view == nil)
    {
      [NSBundle loadNibNamed: @"ApplicationGeneralAttributes" owner: self];
    }

  return view;
}

- (void) controlTextDidEndEditing: (NSNotification *)aNotification
{
  [owner setApplicationName: NonEmptyString ([appName stringValue])];
  [owner setCopyright: NonEmptyString([copyright stringValue])];
  [owner setCopyrightDescription: NonEmptyString([copyrightDescription
    stringValue])];
  [owner setApplicationDescription: NonEmptyString([description stringValue])];
  [owner setIconName: NonEmptyString([iconFilename stringValue])];
  [owner setMainNibFile: NonEmptyString([mainNibFile stringValue])];
  [owner setPrincipalClass: NonEmptyString([principalClass stringValue])];
  [owner setApplicationRelease: NonEmptyString([release stringValue])];
}

- (int) numberOfRowsInTableView: (NSTableView *)aTableView
{
  return [[owner authors] count];
}

- (id) tableView: (NSTableView *)aTableView 
objectValueForTableColumn: (NSTableColumn *)aTableColumn 
             row: (int)rowIndex
{
  return [[owner authors] objectAtIndex: rowIndex];
}

- (void) tableView: (NSTableView *)aTableView 
    setObjectValue: (id)anObject 
    forTableColumn: (NSTableColumn *)aTableColumn
               row: (int)rowIndex
{
  NSMutableArray * newAuthors = [[[owner authors] mutableCopy] autorelease];

  [newAuthors replaceObjectAtIndex: rowIndex withObject: anObject];
  [owner setAuthors: newAuthors];
}

@end
