/*******************************************************************************
* FILE NAME: ispinbas.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in ispinbas.hpp.                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
// Priority INT_MIN (-2147483647 - 1) + 1024 + 512
#pragma priority( -2147482112 )

extern "C" {
  #define INCL_WININPUT
  #define INCL_WINMESSAGEMGR
  #define INCL_WINSTDSPIN
  #define INCL_WINSYS
  #define INCL_WINWINDOWMGR
  #include <iwindefs.h>
}

#ifdef IC_WIN
  #include <iclspbw.h>
#endif
#include <ispinbas.hpp>
#include <icconst.h>
#include <icolor.hpp>
#include <iexcept.hpp>
#include <ifont.hpp>
#include <inotifev.hpp>
#include <irect.hpp>
#include <ispinszh.hpp>
#include <itrace.hpp>
#include <imphdr.hpp>
#include <icoordsy.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _ISPINBAS_CPP_
  #include <ipagetun.h>
#endif

/*------------------------------------------------------------------------------
| Public spin button styles.                                                   |
------------------------------------------------------------------------------*/
#pragma data_seg(ICLStaticConst)
const IBaseSpinButton::Style
  IBaseSpinButton::master            = SPBS_MASTER,
  IBaseSpinButton::servant           ( 0, ISPBS_SERVANT ),
  IBaseSpinButton::readOnly          = SPBS_READONLY,
  IBaseSpinButton::leftAlign         = SPBS_JUSTLEFT,
  IBaseSpinButton::centerAlign       ( 0, ISPBS_CENTERALIGN ),
  IBaseSpinButton::rightAlign        = SPBS_JUSTRIGHT,
  IBaseSpinButton::noBorder          = SPBS_NOBORDER,
  IBaseSpinButton::pmCompatible      ( 0, ISPBS_PMCOMPATIBILITY),
  IBaseSpinButton::fastSpin          = SPBS_FASTSPIN,
  IBaseSpinButton::border3D          ( 0, IWS_BORDER3D );
#pragma data_seg()

/*------------------------------------------------------------------------------
| The static object that IBaseSpinButton uses to properly change its size.     |
------------------------------------------------------------------------------*/
#pragma data_seg(ICLNonConst)
static ISpinButtonStatics SpinButtonStatics;
#pragma data_seg()

/*------------------------------------------------------------------------------
| The spin button private data class.                                          |
------------------------------------------------------------------------------*/
class IBaseSpinButtonData
{
public:
void
  setLimit ( unsigned long ulLimit)
    { textLimit = ulLimit;}
unsigned long
  limit ( ) const
    { return textLimit; }
unsigned long
  textLimit;

//  Returns the handle of the entry field part of a spinbutton
static IWindowHandle
  handleOfEntryField( const IWindowHandle& spinButton,
                      unsigned long        spinId );

IWinProc
 *fdefaultEntryProc; // Default procedure for Entry field part of spinbutton
IWindowHandle
  fentryHandle;      // Handle of the entry field

}; // IBaseSpinButtonData


class IBaseSpinHandler : public IHandler {
typedef IHandler
  Inherited;
public:

/*------------------------------- Constructors -------------------------------*/
  IBaseSpinHandler ( );

virtual
 ~IBaseSpinHandler ( );

/*----------------------------- Event Processing -----------------------------*/
virtual Boolean
  dispatchHandlerEvent ( IEvent& event );

}; // class IBaseSpinHandler

/*------------------------------------------------------------------------------
| IBaseSpinHandler::IBaseSpinHandler                                           |
|                                                                              |
| Empty constructor here for page tuning.                                      |
------------------------------------------------------------------------------*/
IBaseSpinHandler::IBaseSpinHandler()
{ }

/*------------------------------------------------------------------------------
| IBaseSpinHandler::~IBaseSpinHandler                                          |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IBaseSpinHandler::~IBaseSpinHandler()
{ }

/*------------------------------------------------------------------------------
| IBaseSpinHandler::dispatchHandlerEvent                                       |
|                                                                              |
| Intercepts control color event                                               |
|    This handler returns false for all other events.                          |
------------------------------------------------------------------------------*/
IBase::Boolean IBaseSpinHandler::dispatchHandlerEvent( IEvent& event )
{
  if (( event.eventId() == WM_PRESPARAMCHANGED ) &&
      ( event.parameter1() == PP_BACKGROUNDCOLOR ))
  {
    unsigned long ulClr=0;
    unsigned long ulRetLen = WinQueryPresParam( event.handle(),
                   PP_BACKGROUNDCOLOR, 0, 0, sizeof(ulClr), &ulClr, 0 );

    IBaseSpinButton* spinBut = (IBaseSpinButton*)event.window();
    if ( spinBut->fBaseSpinButtonData->fentryHandle )
      WinSetPresParam( spinBut->fBaseSpinButtonData->fentryHandle,
                       PP_BACKGROUNDCOLOR, sizeof(ulClr), &ulClr );
  }
  return false;
}

/*------------------------------------------------------------------------------
| DefaultHandler                                                               |
|                                                                              |
| This class wrappers an object pointer and destroys the pointer to the object |
| when the wrapper is destructed.                                              |
------------------------------------------------------------------------------*/
static struct DefaultHandler {
  operator IBaseSpinHandler* ();
IBaseSpinHandler
 *operator -> ()
    {
    return *this;
    }
 ~DefaultHandler ( )
    {
    if (ptr)
      {
      delete ptr;
      ptr = 0;
      }
    }
IBaseSpinHandler
 *ptr;
} defaultHandler;

DefaultHandler::operator IBaseSpinHandler* ( )
{
  if ( !ptr )
    ptr = new IBaseSpinHandler;
  return ptr;
}


/*------------------------------------------------------------------------------
| IBaseSpinButton::IBaseSpinButton                                             |
|                                                                              |
| Constructor to create a spin button control.                                 |
------------------------------------------------------------------------------*/
IBaseSpinButton :: IBaseSpinButton ( )
{
   fBaseSpinButtonData = new IBaseSpinButtonData();
   fBaseSpinButtonData->setLimit(255);
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::~IBaseSpinButton                                            |
------------------------------------------------------------------------------*/
IBaseSpinButton::~IBaseSpinButton ( )
{
   // Remove the subclass of the entry field
   if ( (fBaseSpinButtonData->fentryHandle) &&
        IISWINDOW(0, fBaseSpinButtonData->fentryHandle) )
   {
     ISUBCLASSWINDOW( fBaseSpinButtonData->fentryHandle,
                      fBaseSpinButtonData->fdefaultEntryProc);
   }
   removeHandler(defaultHandler);
   delete fBaseSpinButtonData;
   SpinButtonStatics.sizeHandler().stopHandlingEventsFor(this);
   IMousePointerHandler::defaultHandler()->stopHandlingEventsFor(this);
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::initialize                                                  |
------------------------------------------------------------------------------*/
void IBaseSpinButton :: initialize ( unsigned long        windowId,
                                     const IWindowHandle& parent,
                                     const IWindowHandle& owner,
                                     unsigned long        ulStyle,
                                     const IRectangle&    initial )
{
  IASSERTPARM(parent!=0);

  IWindowHandle spb =
      this -> create( windowId,
                      0,
                      ulStyle,
                      WC_SPINBUTTON,
                      parent,
                      owner,
                      initial,
                      0,
                      0 );
  initialize();
  startHandlingEventsFor(spb);

  // Subclass the entry field in the spinbutton to handle font changes
  fBaseSpinButtonData->fentryHandle =
     IBaseSpinButtonData::handleOfEntryField( spb, windowId );
  if (fBaseSpinButtonData->fentryHandle)
  {
    // setup our procedure in the window.
    fBaseSpinButtonData->fdefaultEntryProc = (IWinProc*)
        ISUBCLASSWINDOW( fBaseSpinButtonData->fentryHandle,
                         ispinEntryWinProc );
    if ( fBaseSpinButtonData->fdefaultEntryProc == 0 )
    {
      // throw now rather than trap later.
      ITHROWGUIERROR( "ISUBCLASSWINDOW" );
    }
  }
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::initialize                                                  |
------------------------------------------------------------------------------*/
void IBaseSpinButton :: initialize ( )
{
   SpinButtonStatics.sizeHandler().handleEventsFor(this);
   IMousePointerHandler::defaultHandler()->handleEventsFor(this);
   this->addHandler( defaultHandler );
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::enableDataUpdate                                            |
|                                                                              |
| Set or remove the readOnly style on the spinbutton.                          |
------------------------------------------------------------------------------*/
IBaseSpinButton& IBaseSpinButton::enableDataUpdate( Boolean writeable )
{
  unsigned long ulStyle = style();
  unsigned long ulOldStyle = ulStyle;

  if (writeable)
  {
    ulStyle &= ~readOnly.asUnsignedLong();
  }
  else
  {
    ulStyle |= readOnly.asUnsignedLong();
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);              //Change style
    refresh();                      //force refresh
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::disableDataUpdate                                           |
|                                                                              |
| Remove the readOnly style on the spinbutton.                                 |
------------------------------------------------------------------------------*/
IBaseSpinButton& IBaseSpinButton::disableDataUpdate ( )
{
   return (enableDataUpdate(false));
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::enableFastSpin                                              |
|                                                                              |
| Set or remove the fastSpin style on the the spinbutton.                      |
------------------------------------------------------------------------------*/
IBaseSpinButton& IBaseSpinButton::enableFastSpin (Boolean fast)
{
  unsigned long ulStyle = style();
  unsigned long ulOldStyle = ulStyle;

  if (fast)
  {
    ulStyle |= fastSpin.asUnsignedLong();
  }
  else
  {
    ulStyle &= ~fastSpin.asUnsignedLong();
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::disableFastSpin                                             |
|                                                                              |
| Remove the fastSpin style on the the spinbutton.                             |
------------------------------------------------------------------------------*/
IBaseSpinButton& IBaseSpinButton::disableFastSpin ()
{
   return (enableFastSpin(false));
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::setAlignment                                                |
|                                                                              |
| Change the spin field text alignment.                                        |
------------------------------------------------------------------------------*/
IBaseSpinButton& IBaseSpinButton::setAlignment(Alignment alignment)
{
  unsigned long ulStyle = style();
  unsigned long ulOldStyle = ulStyle;

  switch (alignment)
  {
    case left:
      ulStyle |= leftAlign.asUnsignedLong();
      ulStyle &= ~rightAlign.asUnsignedLong();
    break;

    case right:
      ulStyle |= rightAlign.asUnsignedLong();
      ulStyle &= ~leftAlign.asUnsignedLong();
    break;

    case center:
      ulStyle |= SPBS_JUSTCENTER;
    break;
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);                  //Change style
    refresh();                          //Force refresh
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::addBorder                                                   |
|                                                                              |
| Add or remove the border on the spinbutton.                                  |
------------------------------------------------------------------------------*/
IBaseSpinButton& IBaseSpinButton::addBorder (Boolean add)
{
  unsigned long ulStyle = style();
  unsigned long ulOldStyle = ulStyle;

  if (add)
  {
    ulStyle &= ~noBorder.asUnsignedLong();
  }
  else
  {
    ulStyle |= noBorder.asUnsignedLong();
  }

  if (ulStyle != ulOldStyle)
  {
    setStyle(ulStyle);
    refresh();                      // force refresh
  }
  return *this;
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::removeBorder                                                |
|                                                                              |
| Remove the spin field border.                                                |
------------------------------------------------------------------------------*/
IBaseSpinButton& IBaseSpinButton :: removeBorder ( )
{
   return addBorder(false);
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::isMaster                                                    |
|                                                                              |
| Is the spin button a master?                                                 |
------------------------------------------------------------------------------*/
IBase::Boolean IBaseSpinButton :: isMaster ( ) const
{
   return (style() & master.asUnsignedLong()) ? true : false;
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::isServant                                                   |
|                                                                              |
| Is the spin button a servant?                                                |
------------------------------------------------------------------------------*/
IBase::Boolean IBaseSpinButton :: isServant  ( ) const
{
   return (!isMaster());
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::isWriteable                                                 |
|                                                                              |
| Is the spin field read only?                                                 |
------------------------------------------------------------------------------*/
IBase::Boolean IBaseSpinButton :: isWriteable ( ) const
{
   return (style() & readOnly.asUnsignedLong()) ? false : true;
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::isFastSpinEnabled                                           |
|                                                                              |
| Is fast spin activated?                                                      |
------------------------------------------------------------------------------*/
IBase::Boolean IBaseSpinButton :: isFastSpinEnabled ( ) const
{
   return (style() & fastSpin.asUnsignedLong()) ? true : false;
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::hasBorder                                                   |
|                                                                              |
| Does the spin field have a border?                                           |
------------------------------------------------------------------------------*/
IBase::Boolean IBaseSpinButton :: hasBorder   ( ) const
{
   return (style() & noBorder.asUnsignedLong()) ? false : true;
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::hasFocus                                                    |
| Cannot add override because not virtual at IWindow level - future change     |
------------------------------------------------------------------------------*/
//Boolean IBaseSpinButton::hasFocus() const
//{
//  HWND hwndFocus = IQUERYFOCUS(HWND_DESKTOP);
//  return ((hwndFocus == this->fBaseSpinButtonData->fentryHandle) ||
//          (hwndFocus == handle()));
//}

/*------------------------------------------------------------------------------
| IBaseSpinButton::alignment                                                   |
|                                                                              |
| Returns the spin field alignment.                                            |
------------------------------------------------------------------------------*/
IBaseSpinButton::Alignment IBaseSpinButton :: alignment  ( ) const
{
   unsigned long ulStyle = style();
   if ( (ulStyle & leftAlign.asUnsignedLong()) &&
        !(ulStyle & rightAlign.asUnsignedLong()) )
      return left;
   else if ( (ulStyle & rightAlign.asUnsignedLong()) &&
             !(ulStyle & leftAlign.asUnsignedLong()) )
      return right;
   else
      return center;
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::limit                                                       |
|                                                                              |
| Returns the number of characters allowed.                                    |
------------------------------------------------------------------------------*/
unsigned long IBaseSpinButton::limit ( ) const
{
   return this->fBaseSpinButtonData->limit();
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::setLimit                                                    |
|                                                                              |
| Sets the number of characters permitted in a spin field.                     |
------------------------------------------------------------------------------*/
IBaseSpinButton& IBaseSpinButton :: setLimit ( unsigned long aNumber )
{
   this->fBaseSpinButtonData->setLimit(aNumber);
   unsigned long ulOk = handle().sendEvent(SPBM_SETTEXTLIMIT,
//                                         IEventParameter1((USHORT)(aNumber)),
                                           IEventParameter1(aNumber),
                                           IEventParameter2(0));
   if (!ulOk)
      ITHROWGUIERROR2("SPBM_SETTEXTLIMIT",
                      IErrorInfo::invalidParameter,
                      IException::recoverable);

   setLayoutDistorted(IWindow::minimumSizeChanged, 0);
   return *this;
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::setMaster                                                   |
|                                                                              |
| Assign the master for this servant spin button.                              |
------------------------------------------------------------------------------*/
IBaseSpinButton& IBaseSpinButton :: setMaster ( IBaseSpinButton& master )
{
   IASSERTSTATE(isServant());
   IASSERTPARM(master.isMaster());

   unsigned long ulOk =
           handle().sendEvent(SPBM_SETMASTER,
                              IEventParameter1(master.handle()),
                              IEventParameter2(0));
   if (!ulOk)
      ITHROWGUIERROR2("SPBM_SETMASTER",
                      IErrorInfo::accessError,
                      IException::recoverable);

   return *this;
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::spinDown                                                    |
|                                                                              |
| Spin the button down (backward) the specified amount of times.               |
------------------------------------------------------------------------------*/
IBaseSpinButton& IBaseSpinButton :: spinDown ( unsigned long spinBy )
{
   unsigned long ulOk = handle().sendEvent(SPBM_SPINDOWN,
                                           IEventParameter1(spinBy),
                                           IEventParameter2(0));
   if (!ulOk)
      ITHROWGUIERROR2("SPBM_SPINDOWN",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);

   return *this;
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::spinUp                                                      |
|                                                                              |
| Spin the button up (forward) the specified amount of times.                  |
------------------------------------------------------------------------------*/
IBaseSpinButton& IBaseSpinButton :: spinUp   ( unsigned long spinBy )
{
   unsigned long ulOk = handle().sendEvent(SPBM_SPINUP,
                                           IEventParameter1(spinBy),
                                           IEventParameter2(0));
   if (!ulOk)
      ITHROWGUIERROR2("SPBM_SPINUP",
                      IErrorInfo::invalidRequest,
                      IException::recoverable);

   return *this;
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::enable                                                      |
|                                                                              |
| Enables the window to accept keyboard and mouse input.                       |
------------------------------------------------------------------------------*/
IBaseSpinButton& IBaseSpinButton :: enable ( Boolean enableControl )
{
   IWindow::enable( enableControl);
   refresh();
   return *this;
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::isPMCompatible                                              |
------------------------------------------------------------------------------*/
IBase::Boolean IBaseSpinButton::isPMCompatible ( ) const
{
#ifdef IC_WIN
   return ((extendedStyle() & pmCompatible.asExtendedUnsignedLong()) ?
             true : false );
#else
   return true;
#endif
}


#ifdef IC_PM
/*------------------------------------------------------------------------------
| IBaseSpinButton::foregroundColor                                             |
|                                                                              |
| Returns the foreground color of the spinbutton.                              |
------------------------------------------------------------------------------*/
IColor IBaseSpinButton::foregroundColor () const
{
  IGUIColor guiColor(IGUIColor::windowText);

  if (!isWriteable())
  {
    guiColor = IGUIColor(IGUIColor::outputText);
  }
  return (IWindow::color(PP_FOREGROUNDCOLOR, guiColor));
}
#endif

/*------------------------------------------------------------------------------
| IBaseSpinButton::calcMinimumSize                                             |
|                                                                              |
| Return the minimum size of the spin button based on the font and the text    |
| limit.                                                                       |
------------------------------------------------------------------------------*/
ISize IBaseSpinButton :: calcMinimumSize() const
{
  unsigned long ulClTextLimit = limit();

#ifdef IC_WIN
  // IC_NOTYET - Temporary until font support is completed
  return ISize( 50, 20 );
#else
  IFont font( this );
  unsigned long ulWidth = font.avgCharWidth()   // Average char width
                            * ulClTextLimit;
  unsigned long ulHeight = font.maxCharHeight();

  /*****************************************************************/
  /* Use a step function to reflect the fact that you are more     */
  /* likely to exceed an average character width with an           */
  /* individual character, but a long string will likely be ok     */
  /* with this average value.                                      */
  /*****************************************************************/
  // DEFECT 8842 : earlier algorithm gave uneven length increases;
  // simplify to a smooth curve of lengths over text limit range.
  if (ulClTextLimit <= 2)
     ulWidth = (unsigned long)(ulWidth * 3);
  else if (ulClTextLimit <=16)
     ulWidth = (unsigned long)(ulWidth + 5);
  else if (ulClTextLimit <=24)
     ulWidth = (unsigned long)(ulWidth + 4);
  else if (ulClTextLimit <=32)
     ulWidth = (unsigned long)(ulWidth + 3);
  else if (ulClTextLimit <=40)
     ulWidth = (unsigned long)(ulWidth + 2);
  else if (ulClTextLimit <=48)
     ulWidth = (unsigned long)(ulWidth + 1);

  /*****************************************************************/
  /* Add room for the spin arrows, but still be sure that the size */
  /* does not exceed 35 avg chars                                  */
  /*                                                               */
  /* Note: Spin arrows are 17 pels wide.                           */
  /*****************************************************************/
  ulWidth += 17;
  unsigned long maxWidth = font.avgCharWidth() * 35;
  if ( ulWidth > maxWidth )
  {
     ulWidth = maxWidth;
  }

  if (!(style() & noBorder.asUnsignedLong()))
  {                     // Entry field has border.
     ulWidth += (3 * IQUERYSYSVALUE(SV_CXBORDER));
     ulHeight += font.avgLowercase();
  }

  return ISize(ulWidth, ulHeight);
#endif
}

/*------------------------------------------------------------------------------
| IBaseSpinButton::visibleRectangle                                            |
|                                                                              |
| Calculate the actual displayed (painted) ares of the spin button.            |
------------------------------------------------------------------------------*/
IRectangle IBaseSpinButton::visibleRectangle() const
{
  if ( !hasBorder() )
  {
    IFont         font( this );
    unsigned long ulHeight = font.maxCharHeight();
    IRectangle    trueRect = nativeRect();
    trueRect = IRectangle ( trueRect.bottomLeft(),
                            IPoint(trueRect.right(),
                                   trueRect.bottom()+ulHeight));
    return ICoordinateSystem::convertToApplication( trueRect, parentSize() );
  }
  return Inherited::visibleRectangle();
}


/*------------------------------------------------------------------------------
| IBaseSpinButton::setLayoutDistorted                                          |
|                                                                              |
| Queries whether the clipboard is in text format.                             |
------------------------------------------------------------------------------*/
IWindow& IBaseSpinButton::setLayoutDistorted(
                                            unsigned long layoutAttributeOn,
                                            unsigned long layoutAttributeOff )
{
  unsigned long ulFlagsOn = layoutAttributeOn;
  if (layoutAttributeOn & IWindow::fontChanged)
  {                         // set flag for IControl::characterSize()
     ulFlagsOn |= IWindow::minimumSizeChanged;  // flag for ICanvas
  }
  Inherited::setLayoutDistorted(ulFlagsOn, layoutAttributeOff);

  return *this;
}


/*------------------------------------------------------------------------------
| IBaseSpinButtonData::handleOfEntryField                                      |
------------------------------------------------------------------------------*/
IWindowHandle IBaseSpinButtonData::handleOfEntryField(
                                             const IWindowHandle& spinButton,
                                             unsigned long        spinId )
{
   // Retrieve the edit control child of the spin button.
   IWindowHandle ef = IWindow::handleWithParent( spinId, spinButton );
#ifdef IC_PM
   // Work-around for window id breakage in Warp GA for the child
   // entry field of a spin button control (it should be the window
   // id of the parent spin button, but is 1).
   if (!ef)
   {            // Can't find a child entry field of a spin button.
     ef = IWindow::handleWithParent( 1, spinButton );
   }            // Try again with the one Warp GA uses.
#endif
   return ef;
}


/*------------------------------------------------------------------------------
| ispinEntryWinProc                                                            |
| Window procedure for the entryfield part of a spinbutton.  This procedure    |
| intercepts font changes and sends them to the spinbutton itself.             |
| All messages are sent on the orignal procedure for the entryfield            |
------------------------------------------------------------------------------*/
void* _System ispinEntryWinProc( unsigned long hwnd,
                                 unsigned long msg,
                                 void* mp1, void* mp2 )
{
   IWindowHandle    parent   = IPARENTOF(hwnd);
   IBaseSpinButton* spinBt = (IBaseSpinButton*)
                           IWindow::windowWithHandle( parent, false );
   if (spinBt)
   {
     if ( msg == WM_PRESPARAMCHANGED )
     {
       char fontName[120];   // Use same size as in IFont ctor
       unsigned long fontNameLen =
           WinQueryPresParam( hwnd, PP_FONTNAMESIZE, 0, 0,
                              sizeof( fontName ), &fontName,
                              QPF_NOINHERIT );
       if ( fontNameLen )
       {
         if (( ((long)mp1) == PP_FONTNAMESIZE) ||
                                      (((long)mp1) == PP_FONTHANDLE))
         {
           WinSetPresParam( spinBt->handle(), PP_FONTNAMESIZE,
                            fontNameLen+1, (char*)fontName );
           WinRemovePresParam( hwnd, PP_FONTNAMESIZE );
         }
       }
     }
     return spinBt->fBaseSpinButtonData->fdefaultEntryProc(
                                                  hwnd, msg, mp1, mp2 );
   }
   else
   {
      // We are basically busted if we can't find our parent IWindow.  We do
      // what is probably the most reasonable thing and call the
      // default procedure.
      return IDEFWINDOWPROC( hwnd, msg, mp1, mp2 );
   }
}

