/*******************************************************************************
* FILE NAME: ikeyevt.cpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in ikeyevt.hpp.                                                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

extern "C"
  {
  #define INCL_WININPUT
  #define INCL_NLS
  #include <os2.h>
  }

#ifndef _IKEYEVT_
  #include <ikeyevt.hpp>
#endif

#ifndef _ICCONST_
  #include <icconst.h>
#endif

#ifndef _IEXCEPT_
  #include <iexcept.hpp>
#endif

#ifndef _ISTRING_
  #include <istring.hpp>
#endif

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IKEYEVT_CPP_
  #include <ipagetun.h>
#endif


/*******************************************************************/
/* Initialize static flags used by the testing functions.          */
/*******************************************************************/
const unsigned long IKeyboardEvent :: ulCharacterFlag (KC_CHAR);
const unsigned long IKeyboardEvent :: ulScanCodeFlag (KC_SCANCODE);
const unsigned long IKeyboardEvent :: ulVirtualFlag (KC_VIRTUALKEY);
const unsigned long IKeyboardEvent :: ulUpTransitionFlag (KC_KEYUP);
const unsigned long IKeyboardEvent :: ulRepeatFlag (KC_PREVDOWN);
const unsigned long IKeyboardEvent :: ulUncombinedFlag (KC_LONEKEY);
const unsigned long IKeyboardEvent :: ulShiftFlag (KC_SHIFT);
const unsigned long IKeyboardEvent :: ulCtrlFlag (KC_CTRL);
const unsigned long IKeyboardEvent :: ulAltFlag (KC_ALT);
const unsigned long IKeyboardEvent :: ulForCompositeFlag (KC_DEADKEY);
const unsigned long IKeyboardEvent :: ulCompositeFlag (KC_COMPOSITE);
const unsigned long IKeyboardEvent :: ulInvalidCompositeFlag (KC_INVALIDCOMP);


/*------------------------------------------------------------------------------
| IKeyboardEvent::IKeyboardEvent                                               |
|                                                                              |
| Constructor here for page tuning.                                            |
------------------------------------------------------------------------------*/
IKeyboardEvent :: IKeyboardEvent ( const IEvent & evt )
  : IKeyboardEvent::Inherited ( evt ), pData( 0 )
{ ; }


/*------------------------------------------------------------------------------
| IKeyboardEvent::IKeyboardEvent                                               |
|                                                                              |
| Copy Constructor. (DEFECT 25402)                                             |
------------------------------------------------------------------------------*/
IKeyboardEvent::IKeyboardEvent ( const IKeyboardEvent& event )
  : IKeyboardEvent::Inherited( event ), pData( 0 )
{ ; }


/*------------------------------------------------------------------------------
| IKeyboardEvent::IKeyboardEvent                                               |
|                                                                              |
| Default Constructor. (DEFECT 25402)                                          |
------------------------------------------------------------------------------*/
IKeyboardEvent::IKeyboardEvent ( )
  : IKeyboardEvent::Inherited( 0,0,0,0 ), pData( 0 )
{ ; }


/*------------------------------------------------------------------------------
| IKeyboardEvent::~IKeyboardEvent                                              |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IKeyboardEvent :: ~IKeyboardEvent ( )
{ ; }


/*------------------------------------------------------------------------------
| IKeyboardEvent::isCharacter                                                  |
|                                                                              |
| Check if there is a valid character code available to query.  This is true   |
| if the KC_CHAR flag is set (normal character typing) OR can also be valid    |
| on only the down transition of a ctrl or alt modified keypress of a          |
| non-virtual key                                                              |
------------------------------------------------------------------------------*/
IBase::Boolean  IKeyboardEvent :: isCharacter ( ) const
{
  // Check is the operating system set the character flag
  if ( parameter1().number1() & ulCharacterFlag ) 
    return true;

  // Check if this is a Ctrl or Alt key combination 
  if ((isAltDown() || isCtrlDown()) &&
       !isVirtual() && !isUpTransition() && !parameter2().char2())
    return true;

  return false;
}


/*------------------------------------------------------------------------------
| IKeyboardEvent::character                                                    |
|                                                                              |
| Return the SBCS character code for the event.  Before calling this function, |
| the user should first verify whether the event contains a character code by  |
| calling the isCharacter() function.                                          |
------------------------------------------------------------------------------*/
char  IKeyboardEvent :: character ( ) const
{
  if (isCharacter() == false)
  {                               // no character data
     ITHROWLIBRARYERROR(IC_INVALIDCHARCODE,
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
  }
  else if (parameter2().char2() != 0)
  {                               // check for DBCS input
     ITHROWLIBRARYERROR(IC_DBCSCHARASSERT,
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
  } /* endif */

  return parameter2().char1();
}


/*------------------------------------------------------------------------------
| IKeyboardEvent::mixedCharacter                                               |
|                                                                              |
| Return the SBCS or DBCS character code for the event.  Before calling this   |
| function, the user should first verify whether the event contains a          |
| character code by calling the isCharacter() function.                        |
------------------------------------------------------------------------------*/
IString  IKeyboardEvent :: mixedCharacter ( ) const
{
  if (isCharacter() == false)
  {                               // no character data
     ITHROWLIBRARYERROR(IC_INVALIDCHARCODE,
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
  } /* endif */

  IString strChar(parameter2().char1());

  if (parameter2().char2() != 0)
  {
     strChar += parameter2().char2();
  } /* endif */

  return strChar;
}


/*------------------------------------------------------------------------------
| IKeyboardEvent::scanCode                                                     |
|                                                                              |
| Get the hardware scan code for the key.  Before calling this function, the   |
| user should first verify whether the event contains a scan code by calling   |
| the isScanCode() function.                                                   |
------------------------------------------------------------------------------*/
unsigned long  IKeyboardEvent :: scanCode ( ) const
{
  if (isScanCode() == false)
  {         // no scan code data (maybe event is from an input hook?)
     ITHROWLIBRARYERROR(IC_INVALIDSCANCODE,
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
  } /* endif */
  else if (isCharacter()  &&  parameter2().char2() != 0)
  {                               // not valid for dbcs input
     ITHROWLIBRARYERROR(IC_DBCSSCANCODEASSERT,
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
  }

  return parameter1().char4();
}


/*------------------------------------------------------------------------------
| IKeyboardEvent::repeatCount                                                  |
|                                                                              |
| Return the number of times this key event should be applied.                 |
------------------------------------------------------------------------------*/
unsigned long  IKeyboardEvent :: repeatCount ( ) const
{
  return parameter1().char3();
}


/*------------------------------------------------------------------------------
| IKeyboardEvent::virtualKey                                                   |
|                                                                              |
| Get the virtual key value of the key event.  Before calling this function,   |
| the user should first verify whether the event contains a virtual key code   |
| by calling the isVirtual() function.                                         |
------------------------------------------------------------------------------*/
IKeyboardEvent::VirtualKey  IKeyboardEvent :: virtualKey ( ) const
{
  VirtualKey vkey = other;        // assume no match found

  if (isVirtual() == false)
  {                               // not a virtual key
     ITHROWLIBRARYERROR(IC_INVALIDVIRTUALKEY,
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
  } /* endif */

  switch (parameter2().number2())
  {                               // switch on the virtual key code
     case VK_BUTTON1:
        vkey = button1;
        break;
     case VK_BUTTON2:
        vkey = button2;
        break;
     case VK_BUTTON3:
        vkey = button3;
        break;
     case VK_BREAK:
        vkey = breakKey;
        break;
     case VK_BACKSPACE:
        vkey = backSpace;
        break;
     case VK_TAB:
        vkey = tab;
        break;
     case VK_BACKTAB:
        vkey = backTab;
        break;
     case VK_NEWLINE:
        vkey = newLine;
        break;
     case VK_SHIFT:
        vkey = shift;
        break;
     case VK_CTRL:
        vkey = ctrl;
        break;
     case VK_ALTGRAF:
        vkey = altGraf;
        break;
     case VK_PAUSE:
        vkey = pause;
        break;
     case VK_CAPSLOCK:
        vkey = capsLock;
        break;
     case VK_ESC:
        vkey = esc;
        break;
     case VK_SPACE:
        vkey = space;
        break;
     case VK_PAGEUP:
        vkey = pageUp;
        break;
     case VK_PAGEDOWN:
        vkey = pageDown;
        break;
     case VK_END:
        vkey = end;
        break;
     case VK_HOME:
        vkey = home;
        break;
     case VK_LEFT:
        vkey = left;
        break;
     case VK_UP:
        vkey = up;
        break;
     case VK_RIGHT:
        vkey = right;
        break;
     case VK_DOWN:
        vkey = down;
        break;
     case VK_INSERT:
        vkey = insert;
        break;
     case VK_DELETE:
        vkey = deleteKey;
        break;
     case VK_SCRLLOCK:
        vkey = scrollLock;
        break;
     case VK_NUMLOCK:
        vkey = numLock;
        break;
     case VK_ENTER:
        vkey = enter;
        break;
     case VK_SYSRQ:
        vkey = sysRq;
        break;
     case VK_F2:
        vkey = f2;
        break;
     case VK_F3:
        vkey = f3;
        break;
     case VK_F4:
        vkey = f4;
        break;
     case VK_F5:
        vkey = f5;
        break;
     case VK_F6:
        vkey = f6;
        break;
     case VK_F7:
        vkey = f7;
        break;
     case VK_F8:
        vkey = f8;
        break;
     case VK_F9:
        vkey = f9;
        break;
     case VK_F11:
        vkey = f11;
        break;
     case VK_F12:
        vkey = f12;
        break;
     case VK_F13:
        vkey = f13;
        break;
     case VK_F14:
        vkey = f14;
        break;
     case VK_F15:
        vkey = f15;
        break;
     case VK_F16:
        vkey = f16;
        break;
     case VK_F17:
        vkey = f17;
        break;
     case VK_F18:
        vkey = f18;
        break;
     case VK_F19:
        vkey = f19;
        break;
     case VK_F20:
        vkey = f20;
        break;
     case VK_F21:
        vkey = f21;
        break;
     case VK_F22:
        vkey = f22;
        break;
     case VK_F23:
        vkey = f23;
        break;
     case VK_F24:
        vkey = f24;
        break;
     case VK_ENDDRAG:
        vkey = endDrag;
        break;
     case VK_DBCSFIRST:
        vkey = firstDBCS;
        break;
     case VK_DBCSLAST:
        vkey = lastDBCS;
        break;
     case VK_USERFIRST:
        vkey = firstUser;
        break;
     case VK_USERLAST:
        vkey = lastUser;
        break;

     // The below virtual keys are not supported because a
     // keyboard handler will not get the key press (F1, F10),
     // release (Alt), or both (PrintScreen, though maybe it can
     // get the key release) of these keys.  Supporting
     // enumerators for them implies that all presses and
     // releases of these keys can be processed.
     // The application can still process the non-missing events
     // for the below keys by overriding the key() function.
     case VK_ALT:                 // fall into default case
     case VK_PRINTSCRN:           // fall into default case
     case VK_F1:                  // fall into default case
     case VK_F10:                 // fall into default case
     default:
        break;
  } /* endswitch */

  return vkey;
}
