/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import {
  assertUnreachable,
  HttpStatusCode,
  TalerMerchantApi,
} from "@gnu-taler/taler-util";
import {
  ButtonBetterBulma,
  LocalNotificationBannerBulma,
  useLocalNotificationBetter,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { h, VNode } from "preact";
import { useState } from "preact/hooks";
import {
  FormErrors,
  FormProvider,
} from "../../../../components/form/FormProvider.js";
import { Input } from "../../../../components/form/Input.js";
import { InputSelector } from "../../../../components/form/InputSelector.js";
import { WithId } from "../../../../declaration.js";
import { undefinedIfEmpty } from "../../../../utils/table.js";
import { useSessionContext } from "../../../../context/session.js";

const TALER_SCREEN_ID = 78;

type Entity = TalerMerchantApi.WebhookPatchDetails & WithId;

interface Props {
  onConfirm: () => void;
  onBack?: () => void;
  webhook: Entity;
}
const validType = [
  "pay",
  "refund",
  "order_settled",
  "category_added",
  "category_updated",
  "category_deleted",
  "inventory_added",
  "inventory_updated",
  "inventory_deleted",
] as const;
const validMethod = ["GET", "POST", "PUT", "PATCH", "HEAD"];

export function UpdatePage({ webhook, onConfirm, onBack }: Props): VNode {
  const { i18n } = useTranslationContext();

  const { state: session, lib, logIn } = useSessionContext();
  const [notification, safeFunctionHandler] = useLocalNotificationBetter();

  const [state, setState] = useState<Partial<Entity>>(webhook);

  const errors = undefinedIfEmpty<FormErrors<Entity>>({
    event_type: !state.event_type
      ? i18n.str`Required`
      : // @ts-ignore checking if event type is on the valid list
        validType.indexOf(state.event_type) === -1
        ? i18n.str`Must be one of '${validType.join(", ")}'`
        : undefined,
    http_method: !state.http_method
      ? i18n.str`Required`
      : validMethod.indexOf(state.http_method) === -1
        ? i18n.str`Must be one of '${validMethod.join(", ")}'`
        : undefined,
    url: !state.url
      ? i18n.str`Required`
      : isInvalidUrl(state.url)
        ? i18n.str`URL is invalid`
        : undefined,
  });

  const hasErrors = errors !== undefined;

  const data = state as Entity;
  const update = safeFunctionHandler(
    lib.instance.updateWebhook.bind(lib.instance),
    !session.token || !!errors ? undefined : [session.token, webhook.id, data],
  );
  update.onSuccess = (success) => {
    onConfirm();
    return i18n.str`Webhook updated`;
  };
  update.onFail = (fail) => {
    switch (fail.case) {
      case HttpStatusCode.Unauthorized:
        return i18n.str`Unauthorized.`;
      case HttpStatusCode.NotFound:
        return i18n.str`Not found.`;
      case HttpStatusCode.Conflict:
        return i18n.str`Conflict.`;
    }
  };

  return (
    <div>
      <LocalNotificationBannerBulma notification={notification} />
      <section class="section">
        <section class="hero is-hero-bar">
          <div class="hero-body">
            <div class="level">
              <div class="level-left">
                <div class="level-item">
                  <span class="is-size-4">
                    Webhook: <b>{webhook.id}</b>
                  </span>
                </div>
              </div>
            </div>
          </div>
        </section>
        <hr />

        <section class="section is-main-section">
          <div class="columns">
            <div class="column is-four-fifths">
              <FormProvider
                object={state}
                valueHandler={setState}
                errors={errors}
              >
                <InputSelector<Entity>
                  name="event_type"
                  label={i18n.str`Event`}
                  values={["choose", ...validType]}
                  toStr={(v) => {
                    const idx = validType.indexOf(v);
                    if (idx === -1) return i18n.str`Choose one...`;
                    const d = validType[idx];
                    switch (d) {
                      case "pay":
                        return i18n.str`Payment`;
                      case "refund":
                        return i18n.str`Refund`;
                      case "order_settled":
                        return i18n.str`Order settled`;
                      case "category_added":
                        return i18n.str`Category added`;
                      case "category_updated":
                        return i18n.str`Category updated`;
                      case "category_deleted":
                        return i18n.str`Category deleted`;
                      case "inventory_added":
                        return i18n.str`Inventory added`;
                      case "inventory_updated":
                        return i18n.str`Inventory updated`;
                      case "inventory_deleted":
                        return i18n.str`Inventory deleted`;
                      default: {
                        assertUnreachable(d);
                      }
                    }
                  }}
                  tooltip={i18n.str`The event of the webhook: why the webhook is used`}
                />
                <InputSelector<Entity>
                  name="http_method"
                  label={i18n.str`Method`}
                  toStr={(v) => {
                    const idx = validMethod.indexOf(v);
                    if (idx === -1) return i18n.str`Choose one...`;
                    return [
                      i18n.str`GET`,
                      i18n.str`POST`,
                      i18n.str`PUT`,
                      i18n.str`PATCH`,
                      i18n.str`HEAD`,
                    ][idx];
                  }}
                  values={["choose", ...validMethod]}
                  tooltip={i18n.str`Method used by the webhook`}
                />
                <Input<Entity>
                  name="url"
                  label={i18n.str`URL`}
                  tooltip={i18n.str`URL of the webhook where the customer will be redirected`}
                />
                <Input<Entity>
                  name="header_template"
                  label={i18n.str`Header`}
                  inputType="multiline"
                  tooltip={i18n.str`Header template of the webhook`}
                />
                <Input<Entity>
                  name="body_template"
                  inputType="multiline"
                  label={i18n.str`Body`}
                  tooltip={i18n.str`Body template used by the webhook`}
                />
              </FormProvider>

              <div class="buttons is-right mt-5">
                {onBack && (
                  <button type="button" class="button" onClick={onBack}>
                    <i18n.Translate>Cancel</i18n.Translate>
                  </button>
                )}
                <ButtonBetterBulma
                  data-tooltip={
                    hasErrors
                      ? i18n.str`Please complete the marked fields`
                      : i18n.str`Confirm operation`
                  }
                  onClick={update}
                  type="submit"
                >
                  <i18n.Translate>Confirm</i18n.Translate>
                </ButtonBetterBulma>
              </div>
            </div>
          </div>
        </section>
      </section>
    </div>
  );
}
function isInvalidUrl(url: string) {
  try {
    const asd = new URL("./", url);
    return false;
  } catch (e) {
    return true;
  }
}
