/* zeq.c - is number equal to zero?

   AUTHOR: Gregory Pietsch

   DESCRIPTION:
   
   This function is also the equivalent of the ! operator.  The integer x has
   the precision prec.
   
   This file has been put into the public domain by its author.
   
   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
   OTHER DEALINGS IN THE SOFTWARE.
*/

#include "mpi.h"

#ifndef _OPTIMIZED_FOR_SIZE

/* Nonzero if X is not aligned on an "unsigned long" boundary.  */
#ifdef _ALIGN
#define UNALIGNED(X) \
  (((unsigned long)X&(sizeof(unsigned long)-1)))
#else
#define UNALIGNED(X) (0)
#endif

/* How many bytes are copied each interation of the word copy loop.  */
#define LITTLEBLOCKSIZE (sizeof(unsigned long))

/* How many bytes are copied each interation of the 4X unrolled loop.  */
#define BIGBLOCKSIZE (sizeof(unsigned long)<<2)

/* Threshhold for punting to the byte copier.  */
#define TOO_SMALL(len) ((len)<BIGBLOCKSIZE)

#endif /* _OPTIMIZED_FOR_SIZE */

int
_Zeq (_MPI_T x, int prec)
{
  unsigned char *us1 = x;

#ifndef _OPTIMIZED_FOR_SIZE
  unsigned long *p;

  /* If the size is small, or x is unaligned, punt into the
     byte loop.  This should be rare.  */
  if (!TOO_SMALL (prec) && !UNALIGNED (us1))
    {
      p = (unsigned long *) us1;

      /* Check a big block at a time if possible. */
      while (prec >= BIGBLOCKSIZE)
	{
	  if (*p++ || *p++ || *p++ || *p++)
	    return 0;
	  prec -= BIGBLOCKSIZE;
	}

      /* Check a little block at a time if possible. */
      while (prec >= LITTLEBLOCKSIZE)
	{
	  if (*p++)
	    return 0;
	  prec -= LITTLEBLOCKSIZE;
	}

      /* Pick up any residual with a byte loop.  */
      us1 = (unsigned char *) p;
    }
#endif
  /* The normal byte loop.  */

  while (prec--)
    {
      if (*us1++)
	return 0;
    }
  return 1;
}

#ifndef _OPTIMIZED_FOR_SIZE
#undef UNALIGNED
#undef LITTLEBLOCKSIZE
#undef BIGBLOCKSIZE
#undef TOO_SMALL
#endif /* _OPTIMIZED_FOR_SIZE */

/* END OF FILE */
