/* This file has been put into the public domain by its author.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
   OTHER DEALINGS IN THE SOFTWARE.
*/

/* tan, tanf, tanl - tangent function

   AUTHOR: Gregory Pietsch

   DESCRIPTION

   The functionality described on this reference page is aligned with the 
   ISO C standard. Any conflict between the requirements described here and 
   the ISO C standard is unintentional. This volume of POSIX.1-2008 defers 
   to the ISO C standard.

   These functions shall compute the tangent of their argument x, measured 
   in radians.

   An application wishing to check for error situations should set errno to 
   zero and call feclearexcept(FE_ALL_EXCEPT) before calling these functions. 
   On return, if errno is non-zero or fetestexcept(FE_INVALID | FE_DIVBYZERO 
   | FE_OVERFLOW | FE_UNDERFLOW) is non-zero, an error has occurred.

   RETURN VALUE

   Upon successful completion, these functions shall return the tangent of x.

   If the correct value would cause underflow, and is not representable, 
   a range error may occur, and tan(), tanf(), and tanl() shall return 
   0.0, or (if IEC 60559 Floating-Point is not supported) an implementation-
   defined value no greater in magnitude than DBL_MIN, FLT_MIN, and LDBL_MIN, 
   respectively.

   If x is NaN, a NaN shall be returned.

   If x is +/-0, x shall be returned.

   If x is subnormal, a range error may occur and x should be returned.

   If x is not returned, tan(), tanf(), and tanl() shall return an 
   implementation-defined value no greater in magnitude than DBL_MIN, FLT_MIN, 
   and LDBL_MIN, respectively.

   If x is +/-Inf, a domain error shall occur, and either a NaN (if 
   supported), or an implementation-defined value shall be returned. 

   If the correct value would cause underflow, and is representable, a range 
   error may occur and the correct value shall be returned.

   If the correct value would cause overflow, a range error shall occur and 
   tan(), tanf(), and tanl() shall return +/-HUGE_VAL, +/-HUGE_VALF, and 
   +/-HUGE_VALL, respectively, with the same sign as the correct value of the 
   function.

   ERRORS

   These functions shall fail if:
   Domain Error
   The value of x is +/-Inf. 

   If the integer expression (math_errhandling & MATH_ERRNO) is non-zero, 
   then errno shall be set to EDOM. If the integer expression 
   (math_errhandling & MATH_ERREXCEPT) is non-zero, then the invalid 
   floating-point exception shall be raised.

   Range Error
   The result overflows. 
   
   If the integer expression (math_errhandling & MATH_ERRNO) is non-zero, 
   then errno shall be set to ERANGE. If the integer expression 
   (math_errhandling & MATH_ERREXCEPT) is non-zero, then the overflow 
   floating-point exception shall be raised.

   These functions may fail if:
   Range Error
   The result underflows, or the value of x is subnormal.

   If the integer expression (math_errhandling & MATH_ERRNO) is non-zero, 
   then errno shall be set to ERANGE. If the integer expression 
   (math_errhandling & MATH_ERREXCEPT) is non-zero, then the underflow 
   floating-point exception shall be raised.
*/

#include "xmath.h"

static long double
_Tan (long double xx)
{
  long double dp1, dp2, dp3, lossth, p, q, x, y, z, zz;
  int j, sign;

  /* pi/4 to extended precision */
  if (_Ldbl->_Size > 10)
    {
      dp1 = 7.853981633974483067550664827649598009884357452392578125E-1L;
      dp2 = 2.8605943630549158983813312792950660807511260829685741796657E-18L;
      dp3 = 2.1679525325309452561992610065108379921905808E-35L;
      lossth = 3.6028797018963968E16L;	/* 2^55 */
    }
  else
    {
      dp1 = 7.853981554508209228515625E-1L;
      dp2 = 7.946627356147928367136046290398E-9L;
      dp3 = 3.061616997868382943065164830688E-17L;
      lossth = 5.49755813888e11L;	/* 2^39 */
    }
  /* make argument positive but save the sign */
  sign = _Getsign ((unsigned char *) &xx, _Ldbl);
  _Setsign ((unsigned char *) &xx, _Ldbl, 0);
  /* total loss of precision */
  if (x > lossth)
    return 0.0L;
  /* compute xx mod pi/4 */
  y = floorl (x / _M_PI_4L);
  /* strip high bits of integer part */
  z = ldexpl (y, -4);
  z = floorl (z);
  z = y - ldexpl (z, 4);
  /* integer and frac part modulo one octant */
  j = z;
  /* map zeros and singularities to origin */
  if (j & 1)
    {
      ++j;
      y += 1.0L;
    }
  z = ((x - y * dp1) - y * dp2) - y * dp3;
  zz = z * z;
  switch (_Fpclassifyl (zz))
    {
    case FP_ZERO:
    case FP_SUBNORMAL:
      y = z;
      break;
    default:
      switch (_Ldbl->_Size)
	{
	case 16:
	  p = (((((-9.889929415807650724957118893791829849557E-1L * zz +
		   1.272297782199996882828849455156962260810E3L) * zz +
		  -4.249691853501233575668486667664718192660E5L) * zz +
		 5.160188250214037865511600561074819366815E7L) * zz +
		-2.307030822693734879744223131873392503321E9L) * zz +
	       2.883414728874239697964612246732416606301E10L) * zz;
	  q = (((((zz +
		   -1.317243702830553658702531997959756728291E3L) * zz +
		  4.529422062441341616231663543669583527923E5L) * zz +
		 -5.733709132766856723608447733926138506824E7L) * zz +
		2.758476078803232151774723646710890525496E9L) * zz +
	       -4.152206921457208101480801635640958361612E10L) * zz +
	    8.650244186622719093893836740197250197602E10L;
	  break;
	case 10:
	case 8:
	default:
	  p = ((-1.3093693918138377764608E4L * zz +
		1.1535166483858741613983E6L) * zz +
	       -1.7956525197648487798769E7L) * zz;
	  q = (((zz +
		 1.3681296347069295467845E4L) * zz +
		-1.3208923444021096744731E6L) * zz +
	       2.5008380182335791583922E7L) * zz +
	    -5.3869575592945462988123E7L;
	  break;
	}
      y = z + z * p / q;
    }
  if (j & 2)
    y = -1.0L / y;
  return sign ? -y : y;
}

double (tan) (double x)
{
  double y;

  switch (_Fpclassify (x))
    {
    case FP_SUBNORMAL:
      _Matherr (ERANGE, FE_UNDERFLOW);	/* FALL THROUGH */
    case FP_ZERO:
    case FP_NAN:
      return x;
    case FP_INFINITE:
      _Matherr (EDOM, FE_INVALID);
      return _Dbl->_Nan._D;
    default:
      y = (double) _Tan (x);
      switch (_Fpclassify (y))
	{
	case FP_INFINITE:
	  _Matherr (ERANGE, FE_OVERFLOW);
	  return _Getsign ((unsigned char *) &y,
			   _Dbl) ? -HUGE_VAL : +HUGE_VAL;
	case FP_SUBNORMAL:
	  _Matherr (ERANGE, FE_UNDERFLOW);	/* FALL THROUGH */
	}
      return y;
    }
}

float (tanf) (float x)
{
  float y;

  switch (_Fpclassifyf (x))
    {
    case FP_SUBNORMAL:
      _Matherr (ERANGE, FE_UNDERFLOW);	/* FALL THROUGH */
    case FP_ZERO:
    case FP_NAN:
      return x;
    case FP_INFINITE:
      _Matherr (EDOM, FE_INVALID);
      return _Flt->_Nan._F;
    default:
      y = (float) _Tan (x);
      switch (_Fpclassifyf (y))
	{
	case FP_INFINITE:
	  _Matherr (ERANGE, FE_OVERFLOW);
	  return _Getsign ((unsigned char *) &y,
			   _Flt) ? -HUGE_VALF : +HUGE_VALF;
	case FP_SUBNORMAL:
	  _Matherr (ERANGE, FE_UNDERFLOW);	/* FALL THROUGH */
	}
      return y;
    }
}

long double (tanl) (long double x)
{
  long double y;

  switch (_Fpclassifyl (x))
    {
    case FP_SUBNORMAL:
      _Matherr (ERANGE, FE_UNDERFLOW);	/* FALL THROUGH */
    case FP_ZERO:
    case FP_NAN:
      return x;
    case FP_INFINITE:
      _Matherr (EDOM, FE_INVALID);
      return _Ldbl->_Nan._L;
    default:
      y = _Tan (x);
      switch (_Fpclassifyl (y))
	{
	case FP_INFINITE:
	  _Matherr (ERANGE, FE_OVERFLOW);
	  return _Getsign ((unsigned char *) &y,
			   _Ldbl) ? -HUGE_VALL : +HUGE_VALL;
	case FP_SUBNORMAL:
	  _Matherr (ERANGE, FE_UNDERFLOW);	/* FALL THROUGH */
	}
      return y;
    }
}

/* END OF FILE */
