/* This file has been put into the public domain by its author.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
   OTHER DEALINGS IN THE SOFTWARE.
*/

/* gamma.c - lgamma and tgamma functions

   AUTHOR: Gregory Pietsch
   
   DESCRIPTION:
   
   tgamma, tgammaf, tgammal - compute gamma() function

   SYNOPSIS

   The functionality described on this reference page is aligned with the 
   ISO C standard. Any conflict between the requirements described here and 
   the ISO C standard is unintentional. This volume of POSIX.1-2008 defers 
   to the ISO C standard.

   These functions shall compute Gamma(x) where Gamma(x) is defined as
   
               INF
               --
     _        |  |  
    |         |      -t   x-1
    |(x) = |  |     e   t     dt
            -- 
             0  

   An application wishing to check for error situations should set errno to 
   zero and call feclearexcept(FE_ALL_EXCEPT) before calling these functions. 
   On return, if errno is non-zero or fetestexcept(FE_INVALID | FE_DIVBYZERO 
   | FE_OVERFLOW | FE_UNDERFLOW) is non-zero, an error has occurred.

   RETURN VALUE

   Upon successful completion, these functions shall return the gamma of x.

   If x is a negative integer, a domain error may occur and either a NaN 
   (if supported) or an implementation-defined value shall be returned. 
   On systems that support the IEC 60559 Floating-Point option, a domain 
   error shall occur and a NaN shall be returned. 

   If x is +/-0, tgamma(), tgammaf(), and tgammal() shall return +/-HUGE_VAL, 
   +/-HUGE_VALF, and +/-HUGE_VALL, respectively.  On systems that support the 
   IEC 60559 Floating-Point option, a pole error shall occur; otherwise, a 
   pole error may occur.

   If the correct value would cause overflow, a range error shall occur and 
   tgamma(), tgammaf(), and tgammal() shall return +/-HUGE_VAL, +/-HUGE_VALF, 
   or +/-HUGE_VALL, respectively, with the same sign as the correct value of 
   the function.

   If the correct value would cause underflow, and is not representable, 
   a range error may occur, and tgamma(), tgammaf(), and tgammal() shall 
   return 0.0, or (if IEC 60559 Floating-Point is not supported) an 
   implementation-defined value no greater in magnitude than DBL_MIN, FLT_MIN, 
   and LDBL_MIN, respectively.

   If the correct value would cause underflow, and is representable, a range 
   error may occur and the correct value shall be returned.

   If x is subnormal and 1/x is representable, 1/x should be returned.

   If x is NaN, a NaN shall be returned.

   If x is +Inf, x shall be returned.

   If x is -Inf, a domain error shall occur, and a NaN shall be returned.

   ERRORS

   These functions shall fail if:

   Domain Error 
   The value of x is a negative integer, or x is -Inf. 
   If the integer expression (math_errhandling & MATH_ERRNO) is non-zero, 
   then errno shall be set to EDOM. If the integer expression 
   (math_errhandling & MATH_ERREXCEPT) is non-zero, then the invalid 
   floating-point exception shall be raised.

   Pole Error
   The value of x is zero. 
   If the integer expression (math_errhandling & MATH_ERRNO) is non-zero, 
   then errno shall be set to ERANGE. If the integer expression 
   (math_errhandling & MATH_ERREXCEPT) is non-zero, then the divide-by-zero 
   floating-point exception shall be raised.

   Range Error
   The value overflows. 
   If the integer expression (math_errhandling & MATH_ERRNO) is non-zero, 
   then errno shall be set to ERANGE. If the integer expression 
   (math_errhandling & MATH_ERREXCEPT) is non-zero, then the overflow 
   floating-point exception shall be raised.

   These functions may fail if:
   
   Domain Error
   The value of x is a negative integer. 
   If the integer expression (math_errhandling & MATH_ERRNO) is non-zero, 
   then errno shall be set to EDOM. If the integer expression 
   (math_errhandling & MATH_ERREXCEPT) is non-zero, then the invalid 
   floating-point exception shall be raised.

   Pole Error
   The value of x is zero. 
   If the integer expression (math_errhandling & MATH_ERRNO) is non-zero, 
   then errno shall be set to ERANGE. If the integer expression 
   (math_errhandling & MATH_ERREXCEPT) is non-zero, then the divide-by-zero 
   floating-point exception shall be raised.

   Range Error
   The result underflows. 
   If the integer expression (math_errhandling & MATH_ERRNO) is non-zero, 
   then errno shall be set to ERANGE. If the integer expression 
   (math_errhandling & MATH_ERREXCEPT) is non-zero, then the underflow 
   floating-point exception shall be raised.

   lgamma, lgammaf, lgammal, signgam - log gamma function

   DESCRIPTION

   The functionality described on this reference page is aligned with the 
   ISO C standard. Any conflict between the requirements described here and 
   the ISO C standard is unintentional. This volume of POSIX.1-2008 defers 
   to the ISO C standard.

   These functions shall compute the natural logarithm of |Gamma(x)| where 
   Gamma(x) is defined as above.
  
   The argument x need not be a non-positive integer (Gamma(x) is defined over 
   the reals, except the non-positive integers).

   If x is NaN, -Inf, or a negative integer, the value of signgam is 
   unspecified. 

   These functions need not be thread-safe.

   An application wishing to check for error situations should set errno to 
   zero and call feclearexcept(FE_ALL_EXCEPT) before calling these functions.
   On return, if errno is non-zero or fetestexcept(FE_INVALID | FE_DIVBYZERO | 
   FE_OVERFLOW | FE_UNDERFLOW) is non-zero, an error has occurred.

   RETURN VALUE

   Upon successful completion, these functions shall return the logarithmic 
   gamma of x.

   If x is a non-positive integer, a pole error shall occur and lgamma(), 
   lgammaf(), and lgammal() shall return +HUGE_VAL, +HUGE_VALF, and +HUGE_VALL, 
   respectively.

   If the correct value would cause overflow, a range error shall occur and 
   lgamma(), lgammaf(), and lgammal() shall return +/-HUGE_VAL, +/-HUGE_VALF, 
   and +/-HUGE_VALL (having the same sign as the correct value), respectively.

   If x is NaN, a NaN shall be returned.

   If x is 1 or 2, +0 shall be returned.

   If x is +/-Inf, +Inf shall be returned.

   ERRORS

   These functions shall fail if:
   Pole Error
   The x argument is a negative integer or zero. 
   If the integer expression (math_errhandling & MATH_ERRNO) is non-zero, 
   then errno shall be set to ERANGE. If the integer expression 
   (math_errhandling & MATH_ERREXCEPT) is non-zero, then the divide-by-zero 
   floating-point exception shall be raised.

   Range Error
   The result overflows. 
   If the integer expression (math_errhandling & MATH_ERRNO) is non-zero, 
   then errno shall be set to ERANGE. If the integer expression 
   (math_errhandling & MATH_ERREXCEPT) is non-zero, then the overflow 
   floating-point exception shall be raised.
*/

#include "xmath.h"

int signgam;

#define MAXSTIR 143.01608
static double SQTPI = 2.50662827463100050242E0;
#define MAXGAM 171.624376956302725
static double LOGPI = 1.14472988584940017414;
/* log( sqrt( 2*pi ) ) */
static double LS2PI = 0.91893853320467274178;
#define MAXLGM 2.556348e305

/* Stirling's formula for the gamma function.  */
static double
stirf (double x)
{
  double v, w, y;

  w = 1.0 / x;
  w = 1.0 + w * (((((7.87311395793093628397E-4) * w +
		    -2.29549961613378126380E-4) * w +
		   -2.68132617805781232825E-3) * w +
		  3.47222221605458667310E-3) * w + 8.33333333333482257126E-2);
  y = exp (x);
  if (x > MAXSTIR)
    {
      /* avoid overflow in pow() */
      v = pow (x, 0.5 * x - 0.25);
      y = v * (v / y);
    }
  else
    y = pow (x, x - 0.5) / y;
  y = SQTPI * y * w;
  return y;
}

double (tgamma) (double x)
{
  double p, q, z;
  int i, sign;

  signgam = 1;
  switch (_Fpclassify (x))
    {
    case FP_NAN:
      return x;
    case FP_INFINITE:
      sign = _Getsign ((unsigned char *) &x, _Dbl);
      if (sign)
	{
	  _Matherr (EDOM, FE_INVALID);
	  return _Dbl->_Nan._D;
	}
      else
	return x;
    case FP_ZERO:
      _Matherr (ERANGE, FE_DIVBYZERO);
      sign = _Getsign ((unsigned char *) &x, _Dbl);
      return sign ? -HUGE_VAL : HUGE_VAL;
    case FP_SUBNORMAL:
      z = 1.0 / x;
      return z;
    default:
      q = fabs (x);
      sign = _Getsign ((unsigned char *) &x, _Dbl);
      if (q > 33.0)
	{
	  if (sign)
	    {
	      p = floor (q);
	      if (p == q)
		{
		  _Matherr (EDOM, FE_INVALID);
		  return _Dbl->_Nan._D;
		}
	      i = p;
	      if ((i & 1) == 0)
		signgam = -1;
	      z = q - p;
	      if (z > 0.5)
		{
		  p += 1.0;
		  z = q - p;
		}
	      z = q * sin (M_PI * z);
	      if (z == 0.0)
		{
		  _Matherr (ERANGE, FE_OVERFLOW);
		  return signgam * HUGE_VAL;
		}
	      z = fabs (z);
	      z = M_PI / (z * stirf (q));
	    }
	  else
	    z = stirf (x);
	  return signgam * z;
	}
      z = 1.0;
      while (x >= 3.0)
	{
	  x -= 1.0;
	  z *= x;
	}
      while (x < 0.0)
	{
	  if (x >= -1.E-9)
	    goto small;
	  z /= x;
	  x += 1.0;
	}
      while (x < 2.0)
	{
	  if (x < 1.E-9)
	    goto small;
	  z /= x;
	  x += 1.0;
	}
      if (x == 2.0)
	return z;
      x -= 2.0;
      p =
	(((((1.60119522476751861407E-4 * x +
	     1.19135147006586384913E-3) * x +
	    1.04213797561761569935E-2) * x +
	   4.76367800457137231464E-2) * x +
	  2.07448227648435975150E-1) * x +
	 4.94214826801497100753E-1) * x + 9.99999999999999996796E-1;
      q =
	((((((-2.31581873324120129819E-5 * x +
	      5.39605580493303397842E-4) * x +
	     -4.45641913851797240494E-3) * x +
	    1.18139785222060435552E-2) * x +
	   3.58236398605498653373E-2) * x +
	  -2.34591795718243348568E-1) * x +
	 7.14304917030273074085E-2) * x + 1.00000000000000000320E0;
      return z * p / q;
    small:
      if (x == 0.0)
	{
	  _Matherr (EDOM, FE_INVALID);
	  return _Dbl->_Nan._D;
	}
      else
	return z / ((1.0 + 0.5772156649015329 * x) * x);
    }
}

double (lgamma) (double x)
{
  double p, q, u, w, z;
  int i;

  signgam = 1;
  switch (_Fpclassify (x))
    {
    case FP_NAN:
      return x;
    case FP_INFINITE:
      _Setsign ((unsigned char *) &x, _Dbl, 0);
      return x;
    case FP_SUBNORMAL:
    case FP_ZERO:
      _Matherr (ERANGE, FE_DIVBYZERO);
      return HUGE_VAL;
    default:
      if (x == 1.0 || x == 2.0)
	return 0.0;
      if (x < -34.0)
	{
	  q = -x;
	  w = lgamma (q);	/* note that this modifies signgam */
	  p = floor (q);
	  if (p == q)
	    {
	      _Matherr (ERANGE, FE_OVERFLOW);
	      return signgam * HUGE_VAL;
	    }
	  i = p;
	  signgam = (i & 1) ? +1 : -1;
	  z = q - p;
	  if (z > 0.5)
	    {
	      p += 1.0;
	      z = p - q;
	    }
	  z = q * sin (M_PI * z);
	  if (z == 0.0)
	    {
	      _Matherr (ERANGE, FE_OVERFLOW);
	      return signgam * HUGE_VAL;
	    }
	  z = LOGPI - log (z) - w;
	  return z;
	}
      if (x < 13.0)
	{
	  z = 1.0;
	  p = 0.0;
	  u = x;
	  while (u >= 3.0)
	    {
	      p -= 1.0;
	      u = x + p;
	      z *= u;
	    }
	  while (u < 2.0)
	    {
	      if (u == 0.0)
		{
		  _Matherr (ERANGE, FE_OVERFLOW);
		  return signgam * HUGE_VAL;
		}
	      z /= u;
	      p += 1.0;
	      u = x + p;
	    }
	  if (z < 0.0)
	    {
	      signgam = -1;
	      z = -z;
	    }
	  else
	    signgam = 1;
	  if (u == 2.0)
	    return log (z);
	  p -= 2.0;
	  x += p;
	  p = (x * (((((-1.37825152569120859100E3 * x +
			-3.88016315134637840924E4) * x +
		       -3.31612992738871184744E5) * x +
		      -1.16237097492762307383E6) * x +
		     -1.72173700820839662146E6) * x +
		    -8.53555664245765465627E5)
	       / ((((((x + -3.51815701436523470549E2)
		      * x + -1.70642106651881159223E4)
		     * x + -2.20528590553854454839E5)
		    * x + -1.13933444367982507207E6)
		   * x + -2.53252307177582951285E6) 
		   * 		  x + -2.01889141433532773231E6));
	  return log (z) + p;
	}
      if (x > MAXLGM)
	{
	  _Matherr (ERANGE, FE_OVERFLOW);
	  return signgam * HUGE_VAL;
	}
      q = (x - 0.5) * log (x) - x + LS2PI;
      if (x > 1.0e8)
	return (q);
      p = 1.0 / (x * x);
      if (x >= 1000.0)
	q += ((7.9365079365079365079365e-4 * p
	       - 2.7777777777777777777778e-3) * p
	      + 0.0833333333333333333333) / x;
      else
	q += ((((8.11614167470508450300E-4 * p +
		 -5.95061904284301438324E-4) * p +
		7.93650340457716943945E-4) * p +
	       -2.77777777730099687205E-3) * p +
	      8.33333333333331927722E-2) / x;
      return q;
    }
}

