/* This file has been put into the public domain by its author.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
   OTHER DEALINGS IN THE SOFTWARE.
*/

/* clog, clogf, clogl - complex natural logarithm functions

   AUTHOR: Gregory Pietsch

   DESCRIPTION:

   The functionality described on this reference page is aligned with the 
   ISO C standard. Any conflict between the requirements described here and 
   the ISO C standard is unintentional. This volume of POSIX.1-2008 defers to 
   the ISO C standard.

   These functions shall compute the complex natural (base e) logarithm of z, 
   with a branch cut along the negative real axis.

   RETURN VALUE:

   These functions shall return the complex natural logarithm value, in the 
   range of a strip mathematically unbounded along the real axis and in the 
   interval [-(i * pi), +(i * pi)] along the imaginary axis.

   ERRORS

   No errors are defined.
*/

#include <complex.h>
#include "xmath.h"

double _Complex (clog) (double _Complex z)
{
  double r, f;

  r = cabs (z);
  f = carg (z);
  return CMPLX (log (r), f);
}

float _Complex (clogf) (float _Complex z)
{
  float r, f;

  r = cabsf (z);
  f = cargf (z);
  return CMPLXF (logf (r), f);
}

long double _Complex (clogl) (long double _Complex z)
{
  long double r, f;

  r = cabsl (z);
  f = cargl (z);
  return CMPLXL (logl (r), f);
}

/* END OF FILE */
