/* This file has been put into the public domain by its author.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
   OTHER DEALINGS IN THE SOFTWARE.
*/

/* cacosh, cacoshf, cacoshl - complex arc hyperbolic cosine functions

   AUTHOR: Gregory Pietsch

   DESCRIPTION:

   The functionality described on this reference page is aligned with the 
   ISO C standard. Any conflict between the requirements described here and 
   the ISO C standard is unintentional. This volume of POSIX.1-2008 defers 
   to the ISO C standard.

   These functions shall compute the complex arc hyperbolic cosine of z, with 
   a branch cut at values less than 1 along the real axis.

   RETURN VALUE:

   These functions shall return the complex arc hyperbolic cosine value, in 
   the range of a half-strip of non-negative values along the real axis and 
   in the interval [-(i*pi), +(i*pi)] along the imaginary axis.

   ERRORS:

   No errors are defined.
*/

#include <complex.h>
#include "xmath.h"

/* cacosh(z) = i * cacos(z) */

double _Complex (cacosh) (double _Complex z)
{
  z = cacos (z);
  return CMPLX (-cimag (z), creal (z));
}

float _Complex (cacoshf) (float _Complex z)
{
  z = cacosf (z);
  return CMPLXF (-cimagf (z), crealf (z));
}

long double _Complex (cacoshl) (long double _Complex z)
{
  z = cacosl (z);
  return CMPLXL (-cimagl (z), creall (z));
}

/* END OF FILE */
