/* This file has been put into the public domain by its author.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
   IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
   OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
   ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
   OTHER DEALINGS IN THE SOFTWARE.
*/

/* cacos, cacosf, cacosl - complex arc cosine functions

   AUTHOR: Gregory Pietsch

   DESCRIPTION:

   The functionality described on this reference page is aligned with the 
   ISO C standard. Any conflict between the requirements described here and 
   the ISO C standard is unintentional. This volume of POSIX.1-2008 defers 
   to the ISO C standard.

   These functions shall compute the complex arc cosine of z, with branch 
   cuts outside the interval [-1, +1] along the real axis.

   RETURN VALUE:

   These functions shall return the complex arc cosine value, in the range of 
   a strip mathematically unbounded along the imaginary axis and in the 
   interval [0, pi] along the real axis.

   ERRORS:

   No errors are defined.
*/

#include <complex.h>
#include "xmath.h"

/* acos(z) = pi/2 - asin(z) */

double _Complex (cacos) (double _Complex z)
{
  z = casin (z);
  return CMPLX (M_PI_2 - creal (z), -cimag (z));
}

float _Complex (cacosf) (float _Complex z)
{
  z = casinf (z);
  return CMPLXF (_M_PI_2F - crealf (z), -cimagf (z));
}

long double _Complex (cacosl) (long double _Complex z)
{
  z = casinl (z);
  return CMPLXL (_M_PI_2L - creall (z), -cimagl (z));
}

/* END OF FILE */
